package org.phosphoresce.lib.commons.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * Zipファイル作成時のリソース管理保持クラス<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2007/01/16	Kitagawa		新規作成
 * 2007/01/16	Kitagawa		JDK1.5+、ストリームエントリー対応
 */
public class ZipEntryContainer {

	/** エントリマップ */
	private Map<String, InputStream> entries;

	/** エントリストリームクローズフラグ */
	private Map<String, Boolean> closedFlags;

	/**
	 * コンストラクタ<br>
	 */
	public ZipEntryContainer() {
		super();
		this.entries = new LinkedHashMap<String, InputStream>();
		this.closedFlags = new LinkedHashMap<String, Boolean>();
	}

	/**
	 * 指定されたエントリファイル名でストリームを追加します。<br>
	 * @param name エントリファイル名
	 * @param stream ストリームオブジェクト
	 */
	public void add(String name, InputStream stream) {
		entries.put(name, stream);
		closedFlags.put(name, false);
	}

	/**
	 * 指定されたエントリファイル名でファイルを追加します。<br>
	 * @param name エントリファイル名
	 * @param file ファイルオブジェクト
	 * @throws FileNotFoundException 指定されたファイルが存在しない場合にスローされます
	 */
	public void add(String name, File file) throws FileNotFoundException {
		add(name, new FileInputStream(file));
	}

	/**
	 * 指定されたエントリファイル名でファイルを追加します。<br>
	 * @param name エントリファイル名
	 * @param filename ファイル名
	 * @throws FileNotFoundException 指定されたファイルが存在しない場合にスローされます
	 */
	public void add(String name, String filename) throws FileNotFoundException {
		add(name, new File(filename));
	}

	/**
	 * 保持されているエントリのキーセットを取得します。<br>
	 * @return キーセット
	 */
	public Set<String> keySet() {
		return entries.keySet();
	}

	/**
	 * 指定されたエントリ名のストリームオブジェクトを取得します。<br>
	 * @param name エントリ名
	 * @return ストリームオブジェクト
	 * @throws IOException 既に利用済みでクローズされているエントリを取得しようとした場合にスローされます
	 */
	public InputStream get(String name) throws IOException {
		if (closedFlags.get(name)) {
			throw new IOException(name + " is already closed entry");
		}
		return entries.get(name);
	}

	/**
	 * 指定されたエントリをクローズします。<br>
	 * @param name エントリ名
	 */
	void closeStream(String name) {
		closedFlags.put(name, true);
	}
}
