package org.phosphoresce.lib.commons.util;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;

/**
 * 複数ペアバリュークラス<br>
 * <br>
 * 当クラスはSerializableを実装したマップキー等で利用するクラスとなります。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/10/23	Kitagawa		新規作成
 * 2012/12/04	Kitagawa		Comparableを実装
 *-->
 */
public class PairedValue implements Serializable, Comparable<PairedValue> {

	/** バリューリスト */
	private List<String> values;

	/**
	 * コンストラクタ<br>
	 * @param values バリューリスト
	 */
	public PairedValue(String... values) {
		super();
		this.values = new LinkedList<String>();
		if (values != null) {
			for (String key : values) {
				this.values.add(key);
			}
		}
	}

	/**
	 * 指定されたインデックスのバリューを取得します。<br>
	 * @param index インデックス
	 * @return 指定されたインデックスのバリュー
	 */
	public String get(int index) {
		return values.get(index);
	}

	/**
	 * ハッシュコードを取得します。<br>
	 * 設定されている全てのキー文字列のハッシュが利用されます。
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		StringBuilder builder = new StringBuilder();
		for (String key : values) {
			result = prime * result + ((key == null) ? 0 : key.hashCode());
			builder.append(key);
		}
		result = prime * result + builder.toString().hashCode();
		return result;
	}

	/**
	 * オブジェクトの等価比較を行います。<br>
	 * @return オブジェクトが等価である場合にtrueを返却
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object object) {
		if (this == object) {
			return true;
		}
		if (object == null) {
			return false;
		}
		if (getClass() != object.getClass()) {
			return false;
		}
		PairedValue other = (PairedValue) object;
		if (values == null) {
			if (other.values != null) {
				return false;
			}
		} else if (values.size() != other.values.size()) {
			return false;
		} else {
			for (int i = 0; i <= values.size() - 1; i++) {
				String value = values.get(i);
				String ovalue = other.values.get(i);
				if (value == null && ovalue != null) {
					return false;
				}
				if (value != null && ovalue == null) {
					return false;
				}
				if (value != null && !value.equals(ovalue)) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return values.toString();
	}

	/**
	 * 指定されたペアバリューオブジェクトと比較を行います。<br>
	 * 先頭値から順に比較を行い、等価でない場合にその結果が返却されます。<br>
	 * 指定された値がnullの場合は自身が大きいと判断して結果が返却されます。<br>
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(PairedValue compare) {
		if (compare == null) {
			return 1;
		}

		int result = 0;
		for (int i = 0; i <= values.size() - 1; i++) {
			if (compare.values.size() - 1 < i) {
				return 1;
			}
			String v1 = values.get(i);
			String v2 = compare.values.get(i);
			if (v1 == null && v2 == null) {
				result = 0;
			} else if (v1 != null && v2 == null) {
				result = 1;
			} else if (v1 == null && v2 != null) {
				result = -1;
			} else {
				result = v1.compareTo(v2);
			}
			if (result != 0) {
				return result;
			}
		}

		if (compare.values.size() > values.size()) {
			return -1;
		}

		return 0;
	}
}
