package org.phosphoresce.lib.commons.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * アノテーション操作ユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/19	Kitagawa		新規作成
 *-->
 */
public class AnnotationUtil {

	/**
	 * コンストラクタ<br>
	 */
	private AnnotationUtil() {
		super();
	}

	/**
	 * 指定対象クラスからアノテーションクラスを取得します。<br>
	 * @param annotationClass 取得アノテーションクラス
	 * @param targetClass 対象クラス
	 * @return アノテーションクラス
	 */
	@SuppressWarnings("unchecked")
	public static <A extends Annotation> A getClassAnnotation(Class<A> annotationClass, Class<?> targetClass) {
		if (annotationClass == null || targetClass == null) {
			return null;
		}
		A result = null;
		Annotation[] declareds = targetClass.getDeclaredAnnotations();
		if (declareds != null) {
			for (Annotation declared : declareds) {
				if (annotationClass.equals(declared.annotationType())) {
					result = (A) declared;
				}
			}
		}
		if (result == null) {
			result = getClassAnnotation(annotationClass, targetClass.getSuperclass());
		}
		return result;
	}

	/**
	 * 指定対象メソッドからアノテーションクラスを取得します。<br>
	 * @param annotationClass 取得アノテーションクラス
	 * @param targetClass 対象クラス
	 * @param methodName メソッド名
	 * @param parameterTypes メソッドパラメータクラス
	 * @return アノテーションクラス
	 */
	public static <A extends Annotation> A getMethodAnnotation(Class<A> annotationClass, Class<?> targetClass, String methodName, Class<?>... parameterTypes) {
		Method method = ReflectionUtil.getMethod(targetClass, methodName, parameterTypes);
		return method.getAnnotation(annotationClass);
	}

	/**
	 * 指定対象フィールドからアノテーションクラスを取得します。<br>
	 * @param annotationClass 取得アノテーションクラス
	 * @param targetClass 対象クラス
	 * @param fieldName フィールド名
	 * @return アノテーションクラス
	 */
	public static <A extends Annotation> A getFieldAnnotation(Class<A> annotationClass, Class<?> targetClass, String fieldName) {
		Field field = ReflectionUtil.getField(targetClass, fieldName);
		return field.getAnnotation(annotationClass);
	}
}
