package org.phosphoresce.commons.socket.util;

import java.io.FileInputStream;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

/**
 * ソケットオブジェクトファクトリクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/05	Kitagawa		新規作成
 *-->
 */
public final class SocketFactory {

	/**
	 * コンストラクタ<br>
	 */
	private SocketFactory() {
		super();
	}

	/**
	 * 指定されたキーストアパスとパスワードでサーバーソケット用のSSLコンテキストを生成します。<br>
	 * @param keystorePath キーストアパス
	 * @param keystorePasswd キーストアパスワード
	 * @return SSLContextオブジェクト
	 * @throws GeneralSecurityException キーストア操作時のセキュリティ例外が発生した場合にスローされます
	 * @throws IOException キーストアファイルの読み込みに失敗した場合にスローされます
	 */
	public static SSLContext createServerSocketSSLContext(String keystorePath, String keystorePasswd) throws GeneralSecurityException, IOException {
		final char[] keystorePasswdChars = keystorePasswd.toCharArray();
		KeyStore keystore = KeyStore.getInstance("JKS");
		keystore.load(new FileInputStream(keystorePath), keystorePasswdChars);

		KeyManagerFactory factory = KeyManagerFactory.getInstance("SunX509");
		factory.init(keystore, keystorePasswdChars);

		SSLContext context = SSLContext.getInstance("SSL");
		context.init(factory.getKeyManagers(), null, null);
		return context;
	}

	/**
	 * ソケット用SSLコンテキストを生成します。<br>
	 * @return SSLContextオブジェクト
	 * @throws GeneralSecurityException SSLContextのインスタンス取得に失敗した場合にスローされます
	 */
	public static SSLContext createScoketSSLContext() throws GeneralSecurityException {
		KeyManager[] keymanager = null;
		TrustManager[] trustmanager = { new X509TrustManager() {
			public void checkClientTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
			}

			public void checkServerTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
			}

			public X509Certificate[] getAcceptedIssuers() {
				return null;
			}
		} };

		SSLContext context = SSLContext.getInstance("SSL");
		context.init(keymanager, trustmanager, null);
		return context;
	}

	/**
	 * 指定されたアドレスとポートでSSLソケットを生成します。<br>
	 * SSLコンテキストはSSLソケット生成のタイミングで新規にインスタンスが生成されます。<br>
	 * @param address ソケットアドレス
	 * @param port ソケットポート
	 * @return SSLソケットオブジェクト
	 * @throws GeneralSecurityException SSLContextのインスタンス取得に失敗した場合にスローされます
	 * @throws IOException SSLソケットの確立に失敗した場合にスローされます
	 */
	public static SSLSocket createSSLSocket(String address, int port) throws GeneralSecurityException, IOException {
		SSLSocketFactory factory = createScoketSSLContext().getSocketFactory();
		return (SSLSocket) factory.createSocket(address, port);
	}

	/**
	 * 指定されたポート及び、キーストア情報でSSLサーバーソケットを生成します。<br>
	 * SSLコンテキストはSSLソケット生成のタイミングで新規にインスタンスが生成されます。<br>
	 * @param port ソケットポート
	 * @param keystorePath キーストアパス
	 * @param keystorePasswd キーストアパスワード
	 * @return SSLサーバーソケットオブジェクト
	 * @throws GeneralSecurityException キーストア操作時のセキュリティ例外が発生した場合にスローされます
	 * @throws IOException キーストアファイルの読み込みに失敗した場合にスローされます
	 */
	public static SSLServerSocket createSSLServerSocket(int port, String keystorePath, String keystorePasswd) throws GeneralSecurityException, IOException {
		SSLServerSocketFactory factory = createServerSocketSSLContext(keystorePath, keystorePasswd).getServerSocketFactory();
		return (SSLServerSocket) factory.createServerSocket(port);
	}

	/**
	 * 指定されたアドレスとポートでSSLソケットを生成します。<br>
	 * @param context SSLコンテキストオブジェクト
	 * @param address ソケットアドレス
	 * @param port ソケットポート
	 * @return SSLソケットオブジェクト
	 * @throws IOException SSLソケットの確立に失敗した場合にスローされます
	 */
	public static SSLSocket createSSLSocket(SSLContext context, String address, int port) throws IOException {
		SSLSocketFactory factory = context.getSocketFactory();
		return (SSLSocket) factory.createSocket(address, port);
	}

	/**
	 * 指定されたポート及び、キーストア情報でSSLサーバーソケットを生成します。<br>
	 * @param context SSLコンテキストオブジェクト
	 * @param port ソケットポート
	 * @return SSLサーバーソケットオブジェクト
	 * @throws IOException キーストアファイルの読み込みに失敗した場合にスローされます
	 */
	public static SSLServerSocket createSSLServerSocket(SSLContext context, int port) throws IOException {
		SSLServerSocketFactory factory = context.getServerSocketFactory();
		return (SSLServerSocket) factory.createServerSocket(port);
	}

	/**
	 * 指定されたアドレスとポートでソケットを生成します。<br>
	 * 当メソッドは単純にSocketインスタンスを生成するのみですが、他のメソッドインタフェースとの
	 * 一貫性の為に設けられたメソッドです。<br>
	 * @param address ソケットアドレス
	 * @param port ソケットポート
	 * @return ソケットオブジェクト
	 * @throws IOException ソケットの確立に失敗した場合にスローされます
	 */
	public static Socket createSocket(String address, int port) throws IOException {
		return new Socket(address, port);
	}

	/**
	 * 指定されたポートでサーバーソケットを生成します。<br>
	 * 当メソッドは単純にServerSocketインスタンスを生成するのみですが、他のメソッドインタフェースとの
	 * 一貫性の為に設けられたメソッドです。<br>
	 * @param port ソケットポート
	 * @return サーバーソケットオブジェクト
	 * @throws IOException ソケットの確立に失敗した場合にスローされます
	 */
	public static ServerSocket createServerSocket(int port) throws IOException {
		return new ServerSocket(port);
	}
}
