package org.phosphoresce.commons.socket.http.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.io.HttpSocketContentInputStream;
import org.phosphoresce.commons.socket.http.io.HttpSocketContentOutputStream;
import org.phosphoresce.commons.util.StreamUtil;

/**
 * HTTPプロトコルソケット通信時ユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/10	Kitagawa		新規作成
 *-->
 */
public final class HttpSocketClientUtil {

	/**
	 * コンストラクタ<br>
	 */
	private HttpSocketClientUtil() {
		super();
	}

	/**
	 * リクエスト情報を入力ストリームから読み込みます。<br>
	 * 当メソッドではコンテンツボディデータの読み込みは行わず、コンテンツ入力ストリーム設定のみ行われます。<br>
	 * @param stream 入力ストリームオブジェクト(通常はリクエストクライアントのソケットのストリームが想定されます)
	 * @param bufferSize ストリームデータ入力バッファサイズ
	 * @return リクエスト情報
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public static HttpSocketRequest readRequest(InputStream stream, int bufferSize) throws IOException {
		// リクエスト情報生成
		HttpSocketRequest request = new HttpSocketRequest();

		// リクエスト識別情報取得
		request.setIdentifier(StreamUtil.readLine(stream));

		// ヘッダ情報取得
		while (true) {
			String header = StreamUtil.readLine(stream);
			if (header.length() == 0) {
				break;
			}
			request.addHeader(header);
		}

		// コンテンツボディストリーム設定
		request.setContentStream(new HttpSocketContentInputStream(stream, request.isChunkedContent(), request.getContentLength(), bufferSize));

		return request;
	}

	/**
	 * レスポンス情報を入力ストリームから読み込みます。<br>
	 * 当メソッドではコンテンツボディデータの読み込みは行わず、コンテンツ入力ストリーム設定のみ行われます。<br>
	 * @param stream 入力ストリームオブジェクト(通常はレスポンスサーバーのソケットのストリームが想定されます)
	 * @param bufferSize ストリームデータ入力バッファサイズ
	 * @return レスポンス情報
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public static HttpSocketResponse readResponse(InputStream stream, int bufferSize) throws IOException {
		// レスポンス情報生成
		HttpSocketResponse response = new HttpSocketResponse();

		// レスポンス識別情報取得
		response.setIdentifier(StreamUtil.readLine(stream));

		// ヘッダ情報取得
		while (true) {
			String header = StreamUtil.readLine(stream);
			if (header.length() == 0) {
				break;
			}
			response.addHeader(header);
		}

		// コンテンツボディストリーム設定
		response.setContentStream(new HttpSocketContentInputStream(stream, response.isChunkedContent(), response.getContentLength(), bufferSize));

		return response;
	}

	/**
	 * リクエスト情報をソケットストリームに書き込みます。<br>
	 * ボディコンテンツデータはリクエストから提供されるコンテンツ入力ストリームのデータを基に出力されます。<br>
	 * @param stream ソケットストリームオブジェクト
	 * @param request リクエスト情報オブジェクト
	 * @param bufferSize ストリームバッファサイズ
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public static void writeRequest(OutputStream stream, HttpSocketRequest request, int bufferSize) throws IOException {
		// リクエスト識別情報送信
		StreamUtil.writeLine(stream, request.getIdentifier());

		// ヘッダ情報送信
		String[] headerStrings = request.getHeaderStrings();
		String[] cookieStrings = request.getCookieStrings();
		for (int i = 0; i <= headerStrings.length - 1; i++) {
			StreamUtil.writeLine(stream, headerStrings[i]);
		}
		for (int i = 0; i <= cookieStrings.length - 1; i++) {
			StreamUtil.writeLine(stream, cookieStrings[i]);
		}
		StreamUtil.writeLine(stream);

		// コンテンツボディ情報送信
		if (request.getContentStream() != null) {
			HttpSocketContentOutputStream contentOutputStream = new HttpSocketContentOutputStream(stream, request.isChunkedContent(), bufferSize);
			InputStream contentInputStream = request.getContentStream();
			while (contentInputStream.available() > 0) {
				byte[] bytes = new byte[contentInputStream.available()];
				contentInputStream.read(bytes);
				contentOutputStream.write(bytes);
			}
			// Chunked形式の場合は終了識別行を出力
			if (request.isChunkedContent()) {
				contentOutputStream.writeEOC();
			}
			// コンテンツ出力ストリームのフラッシュ
			contentOutputStream.flush();
		}
	}

	/**
	 * レスポンス基本をソケットストリームに書き込みます。<br>
	 * ボディコンテンツデータはリクエストから提供されるコンテンツ入力ストリームのデータを基に出力されます。<br>
	 * @param stream ソケットストリームオブジェクト
	 * @param response レスポンス情報オブジェクト
	 * @param bufferSize ストリームバッファサイズ
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public static void writeResponse(OutputStream stream, HttpSocketResponse response, int bufferSize) throws IOException {
		// レスポンス識別情報送信
		StreamUtil.writeLine(stream, response.getIdentifier());

		// ヘッダ情報送信
		String[] headerStrings = response.getHeaderStrings();
		String[] cookieStrings = response.getCookieStrings();
		for (int i = 0; i <= headerStrings.length - 1; i++) {
			StreamUtil.writeLine(stream, headerStrings[i]);
		}
		for (int i = 0; i <= cookieStrings.length - 1; i++) {
			StreamUtil.writeLine(stream, cookieStrings[i]);
		}
		StreamUtil.writeLine(stream);

		// コンテンツボディ情報送信
		if (response.getContentStream() != null) {
			HttpSocketContentOutputStream contentOutputStream = new HttpSocketContentOutputStream(stream, response.isChunkedContent(), bufferSize);
			InputStream contentInputStream = response.getContentStream();
			while (contentInputStream.available() > 0) {
				int readSize = contentInputStream.available() > bufferSize ? bufferSize : contentInputStream.available();
				byte[] bytes = new byte[readSize];
				contentInputStream.read(bytes);
				contentOutputStream.write(bytes);
			}
			// Chunked形式の場合は終了識別行を出力
			if (response.isChunkedContent()) {
				contentOutputStream.writeEOC();
			}
			// コンテンツ出力ストリームのフラッシュ
			contentOutputStream.flush();
		}
	}
}
