package org.phosphoresce.commons.socket.http.io;

import java.io.IOException;
import java.net.Socket;

import org.phosphoresce.commons.socket.core.io.SocketStream;
import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;

/**
 * HTTPプロトコルソケットストリームクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/10	Kitagawa		新規作成
 *-->
 */
public class HttpSocketStream extends SocketStream {

	/**
	 * コンストラクタ<br>
	 * @param socket ソケットオブジェクト
	 * @throws IOException クラス初期化時のストリームオブジェクト取得に失敗した場合にスローされます
	 */
	public HttpSocketStream(Socket socket) throws IOException {
		super(socket);
	}

	/**
	 * コンストラクタ<br>
	 * @param socket ソケットオブジェクト
	 * @param match 置換対象文字列
	 * @param replace 置換文字列
	 * @throws IOException クラス初期化時のストリームオブジェクト取得に失敗した場合にスローされます
	 */
	public HttpSocketStream(Socket socket, String match, String replace) throws IOException {
		super(socket, match, replace);
	}

	/**
	 * リクエスト基本情報を指定されたリクエストオブジェクトに読み込みます。<br>
	 * @param request リクエスト基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void readRequest(HttpSocketRequest request) throws IOException {
		// リクエスト識別情報取得
		request.setIdentifier(readLine());

		// ヘッダ情報取得
		while (true) {
			String header = readLine();
			if (header.length() == 0) {
				break;
			}
			request.addHeader(header);
		}
	}

	/**
	 * レスポンス基本情報を指定されたレスポンス情報に読み込みます。<br>
	 * @return レスポンス基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void readResponse(HttpSocketResponse response) throws IOException {
		// レスポンス識別情報取得
		response.setIdentifier(readLine());

		// ヘッダ情報取得
		while (true) {
			String header = readLine();
			if (header.length() == 0) {
				break;
			}
			response.addHeader(header);
		}
	}

	/**
	 * リクエスト基本情報を書き込みます。<br>
	 * @param request リクエスト基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void writeRequest(HttpSocketRequest request) throws IOException {
		// リクエスト識別情報送信
		writeLine(request.getIdentifier());

		// ヘッダ情報送信
		String[] headerStrings = request.getHeaderStrings();
		for (int i = 0; i <= headerStrings.length - 1; i++) {
			writeLine(headerStrings[i]);
		}
		writeSeparate();
	}

	/**
	 * レスポンス基本情報を書き込みます。<br>
	 * @param response レスポンス基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void writeResponse(HttpSocketResponse response) throws IOException {
		// レスポンス識別情報送信
		writeLine(response.getIdentifier());

		// ヘッダ情報送信
		String[] headerStrings = response.getHeaderStrings();
		for (int i = 0; i <= headerStrings.length - 1; i++) {
			writeLine(headerStrings[i]);
		}
		writeSeparate();
	}
}
