package org.phosphoresce.common.graphics.paint;

import java.awt.AlphaComposite;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;

/**
 * グラデーションペイントコンテキスト上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/08	Kitagawa		新規作成
 *-->
 */
public abstract class AbstractGradientPaintContext extends AbstractPaintContext {

	/** グラデーションカラー定義 */
	private GradientColorDefine colorDefine = null;

	/** グラデーションアルファ定義 */
	private GradientAlphaDefine alphaDefine = null;

	/** 滑らかさ */
	private float smoothness = 1.0f;

	/** 倍率 */
	private float ratio = 1.0f;

	/**
	 * コンストラクタ<br>
	 * @param model Paintデータを受け取るColorModelオブジェクト
	 * @param deviceBounds 描画されるグラフィックスプリミティブのデバイス空間でのバウンディングボックス
	 * @param userBounds 描画されるグラフィックスプリミティブのユーザ空間でのバウンディングボックス
	 * @param transform ユーザ空間からデバイス空間へのAffineTransform
	 * @param hints コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 * @param angle 角度
	 * @param colorDefine グラデーションカラー定義
	 * @param alphaDefine グラデーションアルファ定義
	 * @param smoothness 滑らかさ(0.0f～1.0f)
	 * @param ratio 比率(0.0f～100.0f)
	 */
	public AbstractGradientPaintContext(ColorModel model, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform transform, RenderingHints hints, double angle, //
			GradientColorDefine colorDefine, GradientAlphaDefine alphaDefine, float smoothness, float ratio) {
		super(model, deviceBounds, userBounds, transform, hints, angle);
		this.colorDefine = colorDefine;
		this.alphaDefine = alphaDefine;
		this.smoothness = smoothness;
		this.ratio = ratio;
	}

	/**
	 * 指定されたイメージオブジェクトに対して回転角0度時のグラデーションイメージを描画します。<br>
	 * <br>
	 * ここで指定されるイメージサイズは回転後のバウンディングボックスサイズが考慮されたサイズとなっており、
	 * 実際の描画バウンディングボックスサイズとは異なることに注意してください。<br>
	 * @param image 描画バッファリングイメージ
	 * @see org.phosphoresce.common.graphics.paint.AbstractPaintContext#fillPaintImage(java.awt.image.BufferedImage)
	 */
	protected void fillPaintImage(BufferedImage image) {
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		if (alphaDefine != null) {
			fillAlphaGradientImage(graphics, new Dimension(image.getWidth(), image.getHeight()), alphaDefine);
		}
		if (colorDefine != null) {
			graphics.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_IN));
			fillColorGradientImage(graphics, new Dimension(image.getWidth(), image.getHeight()), colorDefine);
		}
	}

	/**
	 * 指定されたグラフィックスオブジェクトに対して回転角0度時のグラデーションイメージを描画します。<br>
	 * <br>
	 * ここで指定されるイメージサイズは回転後のバウンディングボックスサイズが考慮されたサイズとなっており、
	 * 実際の描画バウンディングボックスサイズとは異なることに注意してください。<br>
	 * これは、グラデーションパターンとして回転角0度におけるイメージを生成することのみを目的としているためです。<br>
	 * <br>
	 * また、当メソッドは実際のグラデーションカラーイメージ及び、アルファイメージを生成するために使用されます。<br>
	 * 意図的に別々の動作を行わせるためにはcreateColorGradientImage、createAlphaGradientImageをオーバーライドして下さい。<br>
	 * @param graphics グラフィックスオブジェクト
	 * @param dimension 描画サイズ
	 * @param gradient グラデーション定義
	 */
	protected abstract void fillGradientImage(Graphics2D graphics, Dimension dimension, GradientDefine gradient);

	/**
	 * 指定されたグラフィックスオブジェクトに対して回転角0度時のグラデーションカラーイメージを描画します。<br>
	 * <br>
	 * ここで指定されるイメージサイズは回転後のバウンディングボックスサイズが考慮されたサイズとなっており、
	 * 実際の描画バウンディングボックスサイズとは異なることに注意してください。<br>
	 * これは、グラデーションパターンとして回転角0度におけるイメージを生成することのみを目的としているためです。<br>
	 * <br>
	 * また、カラーイメージはアルファ値は保持しないイメージを作成します。<br>
	 * @param graphics グラフィックスオブジェクト
	 * @param dimension 描画サイズ
	 * @param colorDefine カラーグラデーション定義
	 */
	protected void fillColorGradientImage(Graphics2D graphics, Dimension dimension, GradientColorDefine colorDefine) {
		fillGradientImage(graphics, dimension, colorDefine);
	}

	/**
	 * 指定されたイメージオブジェクトに対して回転角0度時のグラデーションアルファイメージを生成します。<br>
	 * <br>
	 * ここで指定されるイメージサイズは回転後のバウンディングボックスサイズが考慮されたサイズとなっており、
	 * 実際の描画バウンディングボックスサイズとは異なることに注意してください。<br>
	 * これは、グラデーションパターンとして回転角0度におけるイメージを生成することのみを目的としているためです。<br>
	 * <br>
	 * また、アルファイメージは通常のグラデーションイメージとXOR演算して使用されます。<br>
	 * @param image グラデーションイメージ描画バッファリングイメージ
	 * @param alphaDefine アルファグラデーション定義
	 */
	protected void fillAlphaGradientImage(Graphics2D graphics, Dimension dimension, GradientAlphaDefine alphaDefine) {
		fillGradientImage(graphics, dimension, alphaDefine);
	}

	/**
	 * グラデーションカラー定義を取得します。<br>
	 * @return グラデーションカラー定義
	 */
	public final GradientColorDefine getColorDefine() {
		return colorDefine;
	}

	/**
	 * グラデーションアルファ定義を取得します。<br>
	 * @return グラデーションアルファ定義
	 */
	public final GradientAlphaDefine getAlphaDefine() {
		return alphaDefine;
	}

	/**
	 * 滑らかさを取得します。<br>
	 * @return 滑らかさ
	 */
	public float getSmoothness() {
		return smoothness;
	}

	/**
	 * 倍率を取得します。<br>
	 * @return 倍率
	 */
	public float getRatio() {
		return ratio;
	}
}
