package org.phosphoresce.common.graphics.paint;

import java.awt.PaintContext;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.ColorModel;

/**
 * グラデーションペイント上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/08	Kitagawa		新規作成
 *-->
 */
public abstract class AbstractGradientPaint extends AbstractPaint {

	/** ディフォルトの滑らかさ */
	public static final float DEFAULT_SMOOTHNESS = 1.0f;

	/** ディフォルトの比率 */
	public static final float DEFAULT_RAITO = 1.0f;

	/** グラデーションカラー定義 */
	private GradientColorDefine colorDefine = null;

	/** グラデーションアルファ定義 */
	private GradientAlphaDefine alphaDefine = null;

	/** 滑らかさ */
	private float smoothness = DEFAULT_SMOOTHNESS;

	/** 倍率 */
	private float ratio = DEFAULT_RAITO;

	/**
	 * コンストラクタ<br>
	 * @param colorDefine グラデーションカラー定義
	 * @param alphaDefine グラデーションアルファ定義
	 * @param angle 角度
	 * @param smoothness 滑らかさ(0.0f～1.0f)
	 * @param ratio 比率(0.0f～100.0f)
	 */
	public AbstractGradientPaint(GradientColorDefine colorDefine, GradientAlphaDefine alphaDefine, double angle, float smoothness, float ratio) {
		super(angle);
		this.setColorDefine(colorDefine);
		this.setAlphaDefine(alphaDefine);
		this.setSmoothness(smoothness);
		this.setRatio(ratio);
	}

	/**
	 * コンストラクタ<br>
	 */
	private AbstractGradientPaint() {
		this(null, null, DEFAULT_ANGLE, DEFAULT_SMOOTHNESS, DEFAULT_RAITO);
	}

	/**
	 * ペイントオブジェクトのTransparencyを取得します。<br>
	 * 当グラデーションペイントクラスはアルファ値をColorとは別途管理するため、
	 * ここで返却されるTransparencyは設定されているアルファオブジェクトに依存します。<br>
	 * @return ペイントオブジェクトのTransparency
	 * @see java.awt.Transparency#getTransparency()
	 */
	public int getTransparency() {
		if (alphaDefine != null) {
			GradientAlphaDefine[] defines = (GradientAlphaDefine[]) alphaDefine.getAllDefines();
			for (int i = 0; i <= defines.length - 1; i++) {
				if (defines[i].getAlpha() < 1.0f) {
					return TRANSLUCENT;
				}
			}
		}
		return OPAQUE;
	}

	/**
	 * カラーパターンを生成するために使用するPaintContextを生成します。<br>
	 * @param model Paintデータを受け取るColorModelオブジェクト
	 * @param deviceBounds 描画されるグラフィックスプリミティブのデバイス空間でのバウンディングボックス
	 * @param userBounds 描画されるグラフィックスプリミティブのユーザ空間でのバウンディングボックス
	 * @param transform ユーザ空間からデバイス空間へのAffineTransform
	 * @param hints コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 * @return PaintContextオブジェクト
	 * @see java.awt.Paint#createContext(java.awt.image.ColorModel, java.awt.Rectangle, java.awt.geom.Rectangle2D, java.awt.geom.AffineTransform, java.awt.RenderingHints)
	 */
	public abstract PaintContext createContext(ColorModel model, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform transform, RenderingHints hints);

	/**
	 * グラデーションカラー定義を取得します。<br>
	 * @return グラデーションカラー定義
	 */
	public GradientColorDefine getColorDefine() {
		return colorDefine;
	}

	/**
	 * グラデーションカラー定義を設定します。<br>
	 * @param colorDefine グラデーションカラー定義
	 */
	public void setColorDefine(GradientColorDefine colorDefine) {
		this.colorDefine = colorDefine;
	}

	/**
	 * グラデーションアルファ定義を取得します。<br>
	 * @return グラデーションアルファ定義
	 */
	public GradientAlphaDefine getAlphaDefine() {
		return alphaDefine;
	}

	/**
	 * グラデーションアルファ定義を設定します。<br>
	 * @param alphaDefine グラデーションアルファ定義
	 */
	public void setAlphaDefine(GradientAlphaDefine alphaDefine) {
		this.alphaDefine = alphaDefine;
	}

	/**
	 * 滑らかさを取得します。<br>
	 * @return 滑らかさ
	 */
	public float getSmoothness() {
		return smoothness;
	}

	/**
	 * 滑らかさを設定します。<br>
	 * @param smoothness 滑らかさ
	 */
	public void setSmoothness(float smoothness) {
		this.smoothness = smoothness;
		if (this.smoothness < 0.0f) {
			this.smoothness = 0.0f;
		} else if (this.smoothness > 1.0f) {
			this.smoothness = 1.0f;
		}
	}

	/**
	 * 倍率を取得します。<br>
	 * @return 倍率
	 */
	public float getRatio() {
		return ratio;
	}

	/**
	 * 倍率を設定します。<br>
	 * @param ratio 倍率(0.0f～100.0f)
	 */
	public void setRatio(float ratio) {
		this.ratio = ratio;
		if (this.ratio < 0.0f) {
			this.ratio = 0.0f;
		} else if (this.ratio > 100.0f) {
			this.ratio = 100.0f;
		}
	}
}
