package org.phosphoresce.commons.csv;

import java.io.Serializable;

import org.phosphoresce.commons.util.StringUtil;

/**
 * CSVトークンエレメント保持クラス。<br>
 * 当クラスが保持するトークン値はCSV値としてエンコードされた文字列ではなく
 * 通常の文字列(「foo,"bar"」を「"foo,""bar"""」としては扱わない)で保持します。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2007/02/16	Kitagawa		新規作成
 * 2008/11/25	Kitagawa		toStringオーバーライド
 *-->
 */
public final class CSVToken implements Serializable {

	/** CSVトークン区切り文字 */
	public static final String SEPARATOR = ",";

	/** CSVトークンクォート文字 */
	public static final String QUOTE = "\"";

	/** CSVトークン改行文字 */
	public static final String NEWLINE = "\n";

	/** トークンエレメント値 */
	private String value = null;

	/**
	 * コンストラクタ<br>
	 */
	public CSVToken() {
		super();
		this.value = "";
	}

	/**
	 * コンストラクタ<br>
	 * @param value トークンエレメント値
	 */
	public CSVToken(String value) {
		super();
		this.value = value;
	}

	/**
	 * トークンエレメント値を取得します。<br>
	 * @return トークンエレメント値
	 */
	public String getValue() {
		return value;
	}

	/**
	 * トークンエレメント値を設定します。<br>
	 * @param value トークンエレメント値
	 */
	public void setValue(String value) {
		this.value = value == null ? "" : value;
	}

	/**
	 * CSVトークン値としてエンコードされた値を取得します。<br>
	 * @return CSVトークン値としてエンコードされた値
	 */
	public String getEncodedValue() {
		return encode(value);
	}

	/**
	 * 指定された文字列をCSVトークンとしてのエレメント文字列に変換して提供します。<br>
	 * @param value 変換対象文字列
	 * @return CSVトークンとしてのエレメント文字列
	 */
	public static String encode(String value) {
		if (StringUtil.isEmpty(value)) {
			return "";
		} else {
			if (value.indexOf(QUOTE) >= 0 //
					|| value.indexOf(SEPARATOR) >= 0 //
					|| value.indexOf(NEWLINE) >= 0 //
			) {
				StringBuffer buffer = new StringBuffer();

				buffer.append(QUOTE);
				buffer.append(StringUtil.replace(value, QUOTE, QUOTE + QUOTE));
				buffer.append(QUOTE);

				return buffer.toString();
			} else {
				return value;
			}
		}
	}

	/**
	 * 指定されたCSVトークン文字列を通常の文字列に変換して提供します。<br>
	 * @param value 変換対象文字列
	 * @return 変換された通常文字列
	 */
	public static String decode(String value) {
		String render = value;
		if (render.startsWith(QUOTE) && render.endsWith(QUOTE)) {
			render = render.substring(1, render.length() - 1);
		}
		render = StringUtil.replace(render, QUOTE + QUOTE, QUOTE);
		return render;
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		//		StringBuffer buffer = new StringBuffer();
		//		buffer.append("{");
		//		buffer.append("}");
		//		return buffer.toString();
		return value;
	}
}

