package org.phosphoresce.socket.proxy.http.servicetimeout.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandlerAdapter;
import org.phosphoresce.socket.proxy.http.servicetimeout.ServiceTimeoutGlobal;
import org.phosphoresce.socket.proxy.http.servicetimeout.config.ServiceTimeoutConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシサービス時間外チェックコンテンツリクエスト処理ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・コンテキストパスがサービスタイムアウトコンテンツエイリアス以外の場合は処理をスキップ<br>
 * ・サービスタイムアウトコンテンツの場合にコンテンツを送信<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/29	Kitagawa		新規作成
 *-->
 */
public class ServiceTimeoutContentRequestHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ServiceTimeoutContentRequestHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.http.server.handler.ReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		ServiceTimeoutConfig serviceTimeoutConfig = ServiceTimeoutConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		String clientAddress = clientSocket.getInetAddress().getHostAddress();
		HttpSocketRequest request = getRequest();

		// コンテキストパスがサービス時間チェックコンテンツエイリアス以外の場合は処理をスキップ
		if (!serviceTimeoutConfig.getContentAlias().equals(request.getContextPath())) {
			return true;
		}

		// サービスタイムアウトコンテンツの場合にコンテンツを送信
		HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
		factory.putParameter(ServiceTimeoutGlobal.VELOCITY_BIND_KEY_SIGNON_CONFIG, serviceTimeoutConfig);
		factory.setResource(serviceTimeoutConfig.getContentResourcePath(request.getResourcePath()));
		HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), factory.create(), config.getSocketBufferSize());

		return false;
	}
}
