package org.phosphoresce.socket.proxy.http.servicetimeout.handler;

import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandlerAdapter;
import org.phosphoresce.socket.proxy.http.servicetimeout.ServiceTimeoutGlobal;
import org.phosphoresce.socket.proxy.http.servicetimeout.config.ServiceTimeoutConfig;
import org.phosphoresce.socket.proxy.http.servicetimeout.config.ServiceTimeoutMappingConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシサービス時間外チェック処理ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・リクエストされたパスがサービス時間チェック定義に設定されたパス以外の場合は処理をスキップ<br>
 * ・サービス時間帯チェック定義取得<br>
 * ・リクエストされたパスがサービス時間帯チェック定義に設定されたパス以外の場合は処理をスキップ
 * ・環境定義よりサービス時間帯であるかどうかを判定<br>
 * ・サービス時間帯である場合にその旨のコンテンツレスポンスを送信<br>
 * <br>
 * <b>備考</b>
 * 尚、当クラスはサービス時間チェックハンドル処理が行われた場合、クライアントへのレスポンスはこのクラス処理で完結し、以降のハンドル処理に継続することはありません。<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/29	Kitagawa		新規作成
 *-->
 */
public class ServiceTimeoutCheckHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ServiceTimeoutCheckHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.http.server.handler.ReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		ServiceTimeoutConfig serviceTimeoutConfig = ServiceTimeoutConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		String clientAddress = clientSocket.getInetAddress().getHostAddress();
		HttpSocketRequest request = getRequest();

		// サービス時間帯チェック定義取得
		ServiceTimeoutMappingConfig timeoutConfig = serviceTimeoutConfig.getMappingConfig(getRequest().getPath());

		// リクエストされたパスがサービス時間帯チェック定義に設定されたパス以外の場合は処理をスキップ
		if (timeoutConfig == null) {
			log.trace("SERVICE-TIMEOUT-CHECK Skipped " + getRequest().getPath());
			return true;
		}

		// 環境定義よりサービス時間帯であるかどうかを判定
		boolean isServiceTimeout = false;
		SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		Date nowTime = format.parse("1900/01/01 " + new SimpleDateFormat("HH:mm:ss").format(new Date()));
		Date startTime = format.parse("1900/01/01 " + timeoutConfig.getServiceStartTime());
		Date endTime = format.parse("1900/01/01 " + timeoutConfig.getServiceEndTime());
		if (!(nowTime.after(startTime) && nowTime.before(endTime))) {
			isServiceTimeout = true;
		}

		// サービス時間帯である場合にその旨のコンテンツレスポンスを送信
		if (isServiceTimeout) {
			log.debug("SERVICE-TIMEOUT-CHECK Timeout " + getRequest().getPath() + " (" + timeoutConfig.getServiceStartTime() + " - " + timeoutConfig.getServiceEndTime() + ")");
			HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
			factory.putParameter(ServiceTimeoutGlobal.VELOCITY_BIND_KEY_SIGNON_CONFIG, serviceTimeoutConfig);
			factory.setResource(serviceTimeoutConfig.getContentResourcePath(serviceTimeoutConfig.getServiceTimeoutContent()));
			HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), factory.create(), config.getSocketBufferSize());
			return false;
		}

		return true;
	}
}
