package org.phosphoresce.socket.proxy.http.servicetimeout.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

import org.phosphoresce.commons.util.FileUtil;
import org.phosphoresce.commons.util.StringUtil;
import org.phosphoresce.socket.proxy.http.HttpReverseProxySetting;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;

/**
 * HTTPリバースプロキシサービス時間外チェックプラグイン環境定義クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/29	Kitagawa		新規作成
 *-->
 */
public class ServiceTimeoutConfig implements Serializable {

	/** 環境設定リソース名 */
	public static final String CONFIG_NAME = "service-timeout-config.properties";

	/** セルフインスタンス */
	private static ServiceTimeoutConfig instance;

	/** サービスタイムアウトコンテンツエイリアス */
	private String contentAlias;

	/** サービスタイムアウトコンテンツリソース */
	private String serviceTimeoutContent;

	/** サービスタイムアウト環境設定リスト */
	private List mappingConfigs = new LinkedList();

	/**
	 * コンストラクタ<br>
	 */
	private ServiceTimeoutConfig() {
		super();
	}

	// Singleton Method

	/**
	 * セルフインスタンスを取得します。<br>
	 * @return セルフインスタンス
	 * @throws IOException Propertiesからのロード時に入出力例外が発生した場合にスローされます
	 */
	public static ServiceTimeoutConfig instance() throws Throwable {
		if (instance == null) {
			instance = new ServiceTimeoutConfig();
			instance.load();
		}
		return instance;
	}

	/**
	 * プロパティリソースから環境定義を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	public void load() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		if (!new File(setting.getConfigPath() + "/" + CONFIG_NAME).exists()) {
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(setting.getConfigPath() + "/" + CONFIG_NAME);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// フィールド値設定
		this.contentAlias = properties.getProperty("contentAlias");
		this.serviceTimeoutContent = properties.getProperty("serviceTimeoutContent");

		// マッピング定義読み込み
		loadMappingConfig();
	}

	/**
	 * マッピング定義環境設定情報を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	private void loadMappingConfig() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		if (!new File(setting.getConfigPath() + "/" + CONFIG_NAME).exists()) {
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(setting.getConfigPath() + "/" + CONFIG_NAME);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// 定義読み込みループ処理
		int signonDefinitionCount = Integer.parseInt(properties.getProperty("mappingDefinitionCount"));
		for (int i = 0; i <= signonDefinitionCount; i++) {
			String matchUrl = properties.getProperty("mapping." + i + ".matchUrl");
			String serviceStartTime = properties.getProperty("mapping." + i + ".serviceStartTime");
			String serviceEndTime = properties.getProperty("mapping." + i + ".serviceEndTime");
			ServiceTimeoutMappingConfig mappingConfig = new ServiceTimeoutMappingConfig();
			mappingConfig.setMatchUrl(matchUrl);
			mappingConfig.setServiceStartTime(serviceStartTime);
			mappingConfig.setServiceEndTime(serviceEndTime);

			this.mappingConfigs.add(mappingConfig);
		}
	}

	// Accessor Method

	/**
	 * サービスタイムアウトコンテンツエイリアスを取得します。<br>
	 * @return サービスタイムアウトコンテンツエイリアス
	 */
	public String getContentAlias() {
		return StringUtil.isEmpty(contentAlias) ? "/" : contentAlias.startsWith("/") ? contentAlias : "/" + contentAlias;
	}

	/**
	 * サービスタイムアウトコンテンツエイリアスを設定します。<br>
	 * @param contentAlias サービスタイムアウトコンテンツエイリアス
	 */
	public void setContentAlias(String contentAlias) {
		this.contentAlias = contentAlias;
	}

	/**
	 * サービスタイムアウトコンテンツリソースを取得します。<br>
	 * @return サービスタイムアウトコンテンツリソース
	 */
	public String getServiceTimeoutContent() {
		return serviceTimeoutContent;
	}

	/**
	 * サービスタイムアウトコンテンツリソースを設定します。<br>
	 * @param serviceTimeoutContent サービスタイムアウトコンテンツリソース
	 */
	public void setServiceTimeoutContent(String serviceTimeoutContent) {
		this.serviceTimeoutContent = serviceTimeoutContent;
	}

	/**
	 * 指定されたリダイレクト定義を追加します。<br>
	 * @param rootConfig リダイレクト定義
	 */
	public void addSignonConfig(ServiceTimeoutMappingConfig config) {
		mappingConfigs.add(config);
	}

	/**
	 * 指定されたURLに合致するリダイレクト定義を取得します。<br>
	 * 合致する定義が存在しない場合はnullを返却します。<br>
	 * @param url 転送対象判別URL
	 * @return リダイレクト定義
	 */
	public ServiceTimeoutMappingConfig getMappingConfig(String url) {
		for (Iterator iterator = mappingConfigs.iterator(); iterator.hasNext();) {
			ServiceTimeoutMappingConfig config = (ServiceTimeoutMappingConfig) iterator.next();
			if (url.matches(config.getMatchUrl())) {
				return config;
			}
		}
		return null;
	}

	/**
	 * 指定されたリダイレクト定義を削除します。<br>
	 * @param rootConfig リダイレクト定義
	 */
	public void removeMappingConfig(ServiceTimeoutMappingConfig config) {
		mappingConfigs.remove(config);
	}

	/**
	 * 指定されたURLに合致するリダイレクト定義を削除します。<br>
	 * @param url 転送対象判別URL
	 */
	public void removeMappingConfig(String url) {
		ServiceTimeoutMappingConfig config = getMappingConfig(url);
		if (config != null) {
			removeMappingConfig(config);
		}
	}

	// Supplement Method

	/**
	 * 指定されたサービスタイムアウトコンテンツファイルリソースパスを取得します。<br>
	 * @param filename リソースファイル名
	 * @return サービスタイムアウトコンテンツファイルリソースパス
	 * @throws Throwable 指定されたパスからFileオブジェクトを正しく生成できなかった場合に発生
	 */
	public String getContentResourcePath(String filename) throws Throwable {
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		return FileUtil.connectPath(new String[] { config.getServerContentRootPath(), getContentAlias(), filename });
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("contentAlias=");
		buffer.append(contentAlias);
		buffer.append(",");
		buffer.append("serviceTimeoutContent=");
		buffer.append(serviceTimeoutContent);
		buffer.append(",");
		buffer.append("mappingConfigs=");
		buffer.append(mappingConfigs);
		buffer.append("}");
		return buffer.toString();
	}
}
