package org.phosphoresce.socket.proxy.http.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Date;

import org.phosphoresce.commons.socket.http.HttpMimeMapping;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.io.HttpSocketContentInputStream;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシサーバー静的リソースレスポンスファクトリクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/25	Kitagawa		新規作成
 *-->
 */
public class HttpReverseProxyStaticResourceResponseFactory extends HttpReverseProxyResponseFactory {

	/** リソースパス */
	private String resource;

	/** MIMEタイプマッピング */
	private HttpMimeMapping mime;

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッション
	 * @param controllerSession コントローラーセッション
	 */
	public HttpReverseProxyStaticResourceResponseFactory(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * コンストラクタ<br>
	 */
	public HttpReverseProxyStaticResourceResponseFactory() {
		super(null, null);
	}

	/**
	 * レスポンスオブジェクトを生成します。<br>
	 * @return レスポンスオブジェクト
	 * @throws Throwable 正常にレスポンスオブジェクトを生成できなかった場合に発生
	 * @see org.phosphoresce.socket.proxy.http.util.HttpReverseProxyResponseFactory#create()
	 */
	public HttpSocketResponse create() throws Throwable {
		HttpReverseProxyConfig rConfig = HttpReverseProxyConfig.instance();

		HttpSocketResponse response = createRenderResponse();

		// MIMEタイプ設定
		String extension = resource.indexOf(".") >= 0 ? resource.substring(resource.lastIndexOf(".") + 1) : null;
		HttpMimeMapping mime = this.mime == null ? HttpMimeMapping.getMapping(extension) : this.mime;
		if (mime != null) {
			if (mime.getMimeType().startsWith("text")) {
				response.addHeader("Content-Type", mime.getMimeType() + "; charset=" + rConfig.getServerContentCharset());
			} else {
				response.addHeader("Content-Type", mime.getMimeType());
			}
		}

		// リソース情報取得
		File file = new File(resource);
		long streamLength = file.length();
		InputStream stream = new FileInputStream(file);

		// リソース日付設定
		response.addHeader("Date", HttpReverseProxyUtil.createDateHeaderString(new Date(file.lastModified())));

		// コンテンツストリーム設定
		response.setContentStream(new HttpSocketContentInputStream(stream, false, streamLength));

		return response;
	}

	/**
	 * リソースパスを取得します。<br>
	 * @return リソースパス
	 */
	public String getResource() {
		return resource;
	}

	/**
	 * リソースパスを設定します。<br>
	 * @param resource リソースパス
	 */
	public void setResource(String resource) {
		this.resource = resource;
	}

	/**
	 * MIMEタイプマッピングを取得します。<br>
	 * @return MIMEタイプマッピング
	 */
	public HttpMimeMapping getMime() {
		return mime;
	}

	/**
	 * MIMEタイプマッピングを設定します。<br>
	 * @param mime MIMEタイプマッピング
	 */
	public void setMime(HttpMimeMapping mime) {
		this.mime = mime;
	}
}
