package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.commons.socket.http.util.HttpSocketLoggingUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシクライアントリクエスト受信ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・クライアントソケットに対する各種ソケットパラメータの設定<br>
 * ・クライアントソケットからのリクエスト情報を受信し、コントローラセッションにリクエスト情報を設定<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/17	Kitagawa		新規作成
 *-->
 */
public class ReciveClientRequestHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ReciveClientRequestHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();

		// クライアントソケットに対する各種ソケットパラメータの設定
		clientSocket.setKeepAlive(true);
		clientSocket.setSoLinger(true, 10);
		clientSocket.setTcpNoDelay(true);
		clientSocket.setReceiveBufferSize(config.getSocketBufferSize());
		clientSocket.setSendBufferSize(config.getSocketBufferSize());
		clientSocket.setSoTimeout(config.getSocketTimeout());

		// クライアントソケットからのリクエスト情報を受信し、コントローラセッションにリクエスト情報を設定
		setRequest(HttpSocketClientUtil.readRequest(clientSocket.getInputStream(), config.getSocketBufferSize()));
		if (HttpSocketLoggingUtil.isTraceTarget(getRequest())) {
			log.trace("\n" + HttpSocketLoggingUtil.createSocketRequestTrace(getRequest()));
		}

		return true;
	}
}
