package portablesimulator.skillset;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.TreeSet;
import portablesimulator.csv.Repository;

public final class SkillSet implements Comparable {

    private static final boolean debug = false;

    public SkillSet() {
        this(5);
    }

    public SkillSet(int capacity) {
        listKind = new SkillKind[capacity];
        listPositive = new boolean[capacity];
        listPoint = new int[capacity];
        isRefListKind = false;
        size = 0;
    }

    public void initWithColumn(SkillSet filter) {
        if (filter.listKind == null) {
            this.isRefListKind = false;

            this.listKind = null;
            this.listPositive = null;
            this.listPoint = null;
            this.size = 0;
        } else {
            this.isRefListKind = true;
            filter.isRefListKind = true;

            this.listKind = filter.listKind;
            this.listPositive = new boolean[filter.listPositive.length];
            this.listPoint = new int[filter.listPoint.length];
            this.size = filter.size;
        }
    }

    private void copyWhenReferences() {
        if (isRefListKind) {
            SkillKind[] newKind = new SkillKind[listKind.length];
            for (int i = 0; i < size; ++i) {
                newKind[i] = listKind[i];
            }
            listKind = newKind;
            isRefListKind = false;
        }
    }

    public void ensureCapacity(int newSize) {
        if (listKind == null || newSize > listKind.length) {
            SkillKind[] newKind = new SkillKind[newSize];
            int[] newPoint = new int[newSize];
            boolean[] newRange = new boolean[newSize];
            if (listKind != null) {
                for (int i = 0; i < listPoint.length; ++i) {
                    newKind[i] = listKind[i];
                    newPoint[i] = listPoint[i];
                    newRange[i] = listPositive[i];
                }
            }
            listKind = newKind;
            listPoint = newPoint;
            listPositive = newRange;
            isRefListKind = false;
        }
    }
    private SkillKind[] listKind;
    private boolean isRefListKind;
    private boolean[] listPositive;
    private int[] listPoint;
    private int size;

    public void clear() {
        size = 0;
    }

    public int indexOfKind(SkillKind kind) {
        for (int i = 0; i < size; ++i) {
            if (kind(i) == kind) {
                return i;
            }
        }
        return -1;
    }

    public int pointOfKind(SkillKind kind) {
        for (int i = 0; i < size; ++i) {
            if (listKind[i] == kind) {
                return listPoint[i];
            }
        }
        return 0;
    }

    public void set(SkillKind kind, int point, boolean range) {
        if (kind == null) {
            throw new NullPointerException();
        }
        int index = indexOfKind(kind);
        if (index < 0) {
            copyWhenReferences();
            index = size;
            size++;
            if (listKind == null) {
                ensureCapacity(10);
            } else if (index >= listKind.length) {
                ensureCapacity(listKind.length * 2 + 10);
            }
        }
        listKind[index] = kind;
        listPositive[index] = range;
        listPoint[index] = point;
    }

    public void removeAtIndex(int x) {
        copyWhenReferences();
        while(x < size - 1) {
            listKind[x] = listKind[x + 1];
            listPositive[x] = listPositive[x + 1];
            listPoint[x] = listPoint[x + 1];
            ++ x;
        }
        size --;
    }

    public boolean isFixedBy(SkillSet filter) {
        if (this.listKind == filter.listKind) {
            return true;
        }
        if (size != filter.size) {
            return false;
        }
        for (int i = 0; i < filter.size; ++i) {
            if (listKind[i] != filter.listKind[i]) {
                return false;
            }
        }
        return true;
    }

    public SkillSet fixColumnBy(SkillSet filter) {
        SkillSet obj = new SkillSet();
        obj.initWithColumn(filter);
        for (int i = 0; i < filter.size; ++i) {
            SkillKind kind = filter.listKind[i];
            int index = indexOfKind(kind);
            int point = index < 0 ? 0 : listPoint[index];
            boolean range = index < 0 ? true : listPositive[index];

            obj.listPoint[i] = point;
            obj.listPositive[i] = range;
        }
        return obj;
    }

    public boolean isEmpty() {
        return size == 0;
    }

    public int size() {
        return size;
    }

    public void minus_fixed(SkillSet slotSkill) {
        if (debug) {
            if (size != slotSkill.size) {
                throw new IllegalArgumentException("minus_fixed request fixed column");
            }
            for (int i = 0; i < size; ++i) {
                if (listKind[i] != slotSkill.listKind[i]) {
                    throw new IllegalArgumentException("minus_fixed request fixed column");
                }

                listPoint[i] -= slotSkill.listPoint[i];
            }
        } else {
            for (int i = 0; i < size; ++i) {
                listPoint[i] -= slotSkill.listPoint[i];
            }
        }
    }

    public void minus_only(SkillSet slotSkill) {
        for (int i = 0; i < size; ++i) {
            int x = slotSkill.indexOfKind(listKind[i]);
            if (x >= 0) {
                listPoint[i] -= slotSkill.point(x);
            }
        }
    }

    public void sum_all(SkillSet sum) {
        if (size == 0) {
            set_all(sum);
            return;
        }
        if (sum.size == 0) {
            return;
        }
        if (isFixedBy(sum)) {
            sum_fixed(sum);
            return;
        }
        for (int i = 0; i < sum.size; ++i) {
            final SkillKind kind = sum.listKind[i];
            final int point = sum.listPoint[i];
            boolean range = sum.listPositive[i];

            int index = indexOfKind(kind);
            int n;
            if (index < 0) {
                index = size;
                n = 0;
            } else {
                range = listPositive[index];
                n = listPoint[index];
            }

            set(kind, n + sum.listPoint[i], range);
        }
    }

    public void sum_fixed(SkillSet slotSkill) {
        if (false) {
            if (size != slotSkill.size) {
                throw new IllegalArgumentException("sum_fixed request fixed column " + this + " + " + slotSkill);
            }
            for (int i = 0; i < size; ++i) {
                if (listKind[i] != slotSkill.listKind[i]) {
                    throw new IllegalArgumentException("sum_fixed request fixed column");
                }

                listPoint[i] += slotSkill.listPoint[i];
            }
        } else {
            for (int i = 0; i < size; ++i) {
                listPoint[i] += slotSkill.listPoint[i];
            }
        }
    }

    public void set_all(SkillSet slotSkill) {
        this.initWithColumn(slotSkill);
        for (int i = 0; i < size; ++i) {
            listPositive[i] = slotSkill.listPositive[i];
            listPoint[i] = slotSkill.listPoint[i];
        }
    }

    public boolean isOver(SkillSet slotSkill) {
        for(int i = 0; i < slotSkill.size; ++ i) {
            SkillKind kind = slotSkill.listKind[i];

            int targetKind = indexOfKind(kind);
            if (targetKind == -1) {
                continue;
            }

            boolean range = slotSkill.listPositive[i];

            int point1 = this.listPoint[i];
            int point2 = slotSkill.listPoint[targetKind];

            if (range) {
                if (point1 < point2) {
                    return false;
                }
            } else {
                if (point2 > point1) {
                    return false;
                }
            }
        }
        for(int i = 0; i < size; ++ i) {
            SkillKind kind = listKind[i];
            int targetKind = slotSkill.indexOfKind(kind);
            if (targetKind == -1) {
                return false;
            }
        }
        return true;
    }

    public String skillToString(Comparator<SkillPoint> comp, boolean ignoreZero) {
        Collection<SkillPoint> value = null;

        if (comp == null) {
            value = new ArrayList<SkillPoint>();
        } else {
            value = new TreeSet<SkillPoint>(comp);
        }
        for (int i = 0; i < size; ++i) {
            if (ignoreZero) {
                if (listPoint[i] != 0) {
                    value.add(new SkillPoint(listKind[i], listPoint[i], listPositive[i]));
                }
            } else {
                value.add(new SkillPoint(listKind[i], listPoint[i], listPositive[i]));
            }
        }

        StringBuilder str = new StringBuilder();
        str.append("{");
        for (SkillPoint p : value) {
            if (str.length() != 1) {
                str.append(",");
            }
            str.append(p.skillKind.name);
            if (p.skillPoint > 0) {
                str.append("+");
            }
            str.append(p.skillPoint);
            if (p.positiveRange == false) {
                str.append("--");
            }
        }
        str.append("}");
        return str.toString();
    }

    public String toString() {
        return skillToString(null, false);
    }

    public List<String> getNamedSkillNameForDisplay() {
        List<String> result = new ArrayList<String>();
        SkillCategories categories = Repository.getSkillCategories();

        List<SkillPoint> list = new ArrayList<SkillPoint>();
        for (int i = 0; i < size; ++i) {
            SkillPoint point = new SkillPoint(listKind[i], listPoint[i], listPositive[i]);
            SkillPoint name = categories.getNearSkillPoint(point);
            if (name != null) {
                result.add(name.pointName);
            }
        }
        return result;
    }

    public List<String> getNamedSkillNameForSearch() {
        List<String> result = new ArrayList<String>();
        SkillCategories categories = Repository.getSkillCategories();

        for (int i = 0; i < size; ++i) {
            SkillPoint point = new SkillPoint(listKind[i], listPoint[i], listPositive[i]);
            SkillPoint name = categories.getMatchSkillPoint(point);
            if (name != null) {
                result.add(name.pointName);
            } else if ((listPoint[i] % 5) != 0 && listPoint[i] < 0) {
                int x = listPoint[i];
                while (x % 5 != 0) {
                    x --;
                }
                point = new SkillPoint(listKind[i], x, false);
                name = categories.getMatchSkillPoint(point);
                if (name != null) {
                    result.add(name.pointName + "O");
                }
            }
        }
        return result;
    }

    public SkillSet getNamedSkillSetForDisplay() {
        List<SkillPoint> list = new ArrayList<SkillPoint>();
        SkillCategories categories = Repository.getSkillCategories();
        SkillSet result = new SkillSet();

        for (int i = 0; i < size; ++i) {
            SkillPoint point = new SkillPoint(listKind[i], listPoint[i], listPositive[i]);
            SkillPoint name = categories.getNearSkillPoint(point);
            if (name != null) {
                result.set(name.skillKind, name.skillPoint, name.positiveRange);
            }
        }
        return result;
    }

    public int hashCode() {
        int n = 0;
        for (int i = 0; i < listKind.length; ++i) {
            if (listKind[i] == null) {
                continue;
            }
            try {
                String name = listKind[i].name;
                if (name != null) {
                    n += name.hashCode() * listPoint[i];
                }
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
        return n & 0x7fffffff;
    }

    public boolean equals(Object o) {
        return compareTo(o) == 0;
    }

    public int compareTo(Object o) {
        SkillSet set1 = this;
        SkillSet set2 = (SkillSet) o;
        if (this == o) {
            return 0;
        }

        if (set1.isFixedBy(set2)) {
            for (SkillKind kind : set1.listKind) {
                int point1 = set1.pointOfKind(kind);
                int point2 = set2.pointOfKind(kind);

                if (point1 < point2) {
                    return -1;
                }
                if (point1 > point2) {
                    return 1;
                }
            }
            return 0;
        }

        for (SkillKind kind : Repository.getSkillCategories().listupKind()) {
            int point1 = set1.pointOfKind(kind);
            int point2 = set2.pointOfKind(kind);

            if (point1 < point2) {
                return -1;
            }
            if (point1 > point2) {
                return 1;
            }
        }
        return 0;
    }

    public final SkillKind kind(int x) {
        return listKind[x];
    }

    public final boolean positive(int x) {
        return listPositive[x];
    }

    public final int point(int x) {
        return listPoint[x];
    }

    public int totalDiffPoint() {
        int total = 0;
        int point;
        for (int i = 0; i < size; ++i) {
            if (listPositive[i]) {
                point = listPoint[i];
            } else {
                point = -listPoint[i];
            }
            if (point > 0) {
                total += point;
            }
        }
        return total;
    }

    public int maxDiffPoint() {
        int max = 0;
        int point;
        for (int i = 0; i < size; ++i) {
            if (listPositive[i]) {
                point = listPoint[i];
            } else {
                point = -listPoint[i];
            }
            if (point > 0) {
                if (max < point) {
                    max = point;
                }
            }
        }
        return max;
    }
    public static final Comparator<SkillPoint> SKILLPOINT_POINT_COMPARATOR = new Comparator<SkillPoint>() {

        public int compare(SkillPoint A, SkillPoint B) {
            int n = A.skillPoint - B.skillPoint;
            if (n < 0) {
                return 1;
            }
            if (n > 0) {
                return -1;
            }

            n = A.skillKind.name.compareTo(B.skillKind.name);
            return n;
        }
    };
    public static final Comparator<SkillPoint> SKILLPOINT_NAME_COMPARATOR = new Comparator<SkillPoint>() {

        public int compare(SkillPoint A, SkillPoint B) {
            int n = A.skillKind.name.compareTo(B.skillKind.name);
            return n;
        }
    };

    public boolean isZero() {
        for (int i = 0; i < size; ++i) {
            if (listPoint[i] != 0) {
                return false;
            }
        }
        return true;
    }
}
