/*
 Copyright (c) 2009, hkrn All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer. Redistributions in binary
 form must reproduce the above copyright notice, this list of conditions and
 the following disclaimer in the documentation and/or other materials
 provided with the distribution. Neither the name of the hkrn nor
 the names of its contributors may be used to endorse or promote products
 derived from this software without specific prior written permission. 
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 DAMAGE.
 */

//
//  TestMMLEngine.m
//  OCMML
//
//  Created by hkrn on 09/03/01.
//  Copyright 2009 hkrn. All rights reserved.
//
//  $Id: TestMMLEngine.m 97 2009-05-06 11:50:00Z hikarin $
//

#import "TestMMLEngine.h"

@implementation MMLEngine (Test)

- (NSString *)stringToParse
{
    return m_string;
}

- (NSString *)trimmedString
{
    NSString *newString;
    NSScanner *aScan = [NSScanner scannerWithString:m_string];
    [aScan setCharactersToBeSkipped:[NSCharacterSet whitespaceAndNewlineCharacterSet]];
    [aScan scanCharactersFromSet:[NSCharacterSet alphanumericCharacterSet] intoString:&newString];
    return newString;
}

- (void)resetString:(NSString *)aString
{
    [self setStringToParse:aString];
    m_stringLocation = 0;
}

- (BOOL)isRelativeDir
{
    return m_isRelativeDir;
}

- (NSUInteger)stringLocation
{
    return m_stringLocation;
}

- (void)nextToken
{
    m_stringLocation++;
}

@end

@implementation TestMMLEngine

- (NSString *)stringWithLines:(const char *)line, ...
{
    NSMutableString *ret = [NSMutableString stringWithCString:line];
    va_list ap;
    va_start(ap, line);
    const char *bytes = va_arg(ap, const char *);
    while (bytes != nil) {
         [ret appendString:[NSString stringWithCString:bytes]];
        bytes = va_arg(ap, const char *);
        if (bytes != nil) {
            [ret appendString:[NSString stringWithFormat:@"%c", 015]];
        }
    }
    va_end(ap);
    return ret;
}

- (void)setUp
{
    engine = [[MMLEngine alloc] init];
}

- (void)tearDown
{
    [engine release];
}

- (void)testCharacterToken
{
    [engine resetString:@"0a"];
    STAssertEquals([engine characterToken], (unichar)'0', @"");
    STAssertEquals([engine nextCharacterToken], (unichar)'0', @"");
    STAssertEquals([engine characterToken], (unichar)'a', @"");
    [engine nextToken];
    STAssertEquals([engine characterToken], (unichar)0, @"");
}

- (void)testIntToken
{
    [engine resetString:@"1"];
    STAssertEquals([engine intTokenOrDefault:0], 1, @"");
    [engine resetString:@"+2"];
    STAssertEquals([engine intTokenOrDefault:0], 2, @"");
    [engine resetString:@"-3"];
    STAssertEquals([engine intTokenOrDefault:0], -3, @"");
    [engine resetString:@"a"];
    STAssertEquals([engine intTokenOrDefault:0], 0, @"");
    [engine resetString:@""];
    STAssertEquals([engine intTokenOrDefault:0], 0, @"");
}

- (void)testKeySignToken
{
    [engine resetString:@"-"];
    STAssertEquals([engine keySignToken], -1, @"");
    [engine resetString:@"+#"];
    STAssertEquals([engine keySignToken], 2, @"");
    [engine resetString:@"0-+#"];
    STAssertEquals([engine keySignToken], 0, @"");
}

- (void)testParsingRepeat
{
    [engine resetString:@"/:abc:/"];
    [engine parseRepeat];
    STAssertEqualObjects([engine stringToParse], @"abcabc", @"");
    [engine resetString:@"/:2abc:/"];
    [engine parseRepeat];
    STAssertEqualObjects([engine stringToParse], @"abcabc", @"");
    [engine resetString:@"/:abc/:def:/:/"];
    [engine parseRepeat];
    STAssertEqualObjects([engine stringToParse], @"abcdefdefabcdefdef", @"");
}

- (void)testParsingComment
{
    [engine resetString:@"foo/* bar */baz 1/* 3 */5"];
    [engine parseComment];
    STAssertEqualObjects([engine stringToParse], @"foobaz 15", @"");
    STAssertEquals([engine.warnings count], (NSUInteger)0, @"");
    [engine resetString:@"*/foo/*bar"];
    [engine parseComment];
    STAssertEqualObjects([engine stringToParse], @"*/foo/*bar", @"");
    STAssertEquals([engine.warnings count], (NSUInteger)2, @"");
    NSDictionary *dic;
    NSNumber *type;
    dic = [engine.warnings objectAtIndex:0];
    type = [dic objectForKey:@"type"];
    STAssertEquals((enum MMLEngineWarningType)[type intValue], kMMLEngineWarningUnopenedComment, @"");
    dic = [engine.warnings objectAtIndex:1];
    type = [dic objectForKey:@"type"];
    STAssertEquals((enum MMLEngineWarningType)[type intValue], kMMLEngineWarningUnclosedComment, @"");
}

- (void)testParseMacroForOctaveReverse
{
    STAssertTrue([engine isRelativeDir], @"");
    [engine resetString:@"#OCTAVE REVERSE"];
    [engine parseMacro];
    STAssertFalse([engine isRelativeDir], @"");
}

- (void)testParseWave10
{
    [engine resetString:@"#WAV10 1,0"];
    [engine parseMacro];
    STAssertEqualObjects([engine stringToParse], @"", @"");
}

- (void)testParseMacroForReplacement
{
    /*
    [engine resetString:[self stringWithLines:"$foo = bar;", "$a#+() = baz;", "$foo $a#+()", nil]];
    [engine parseMacro];
    STAssertEqualObjects([engine trimmedString], @"bar", @"");
    [engine resetString:[self stringWithLines:"$a  =  a;", "$b=$a;", "$b", nil]];
    [engine parseMacro];
    STAssertEqualObjects([engine trimmedString], @"a", @"");
    [engine resetString:[self stringWithLines:"$bad=$test;", "$bad", nil]];
    [engine parseMacro];
    STAssertEqualObjects([engine trimmedString], @"$bad=$test;$bad", @"");
    [engine resetString:[self stringWithLines:"$init=0;", "$initialize=1;", "$initialize", nil]];
    [engine parseMacro];
    STAssertEqualObjects([engine trimmedString], @"1", @"");
     */
}

@end
