package jp.sourceforge.ocmml.android.modulators;

import java.util.Random;

public final class Noise extends Modulator {
    private static final int NOISE_PHASE_SHIFT = 30;
    private static final int NOISE_TABLE_MASK = TABLE_LENGTH - 1;
    private static final int NOISE_PHASE_MASK = ((1 << NOISE_PHASE_SHIFT) - 1);

    private static void initialize() {
        if (!sInitialized) {
            Random random = new Random();
            for (int i = 0; i < TABLE_LENGTH; i++) {
                sTable[i] = random.nextDouble() * 2.0 - 1.0;
            }
            sInitialized = true;
        }
    }

    public Noise() {
        super();
        initialize();
        setNoiseFrequency(1.0);
        setResettingPhase(true);
        mCounter = 0;
        mResettingPhase = false;
    }

    public double getNextSampleOfs(int ofs) {
        double value = sTable[(mPhase + (ofs << PHASE_SHIFT))
                & NOISE_TABLE_MASK];
        addPhase(1);
        return value;
    }

    public void resetPhase() {
        if (mResettingPhase != null && mResettingPhase)
            mPhase = 0;
    }

    public void addPhase(int time) {
        mCounter = (long) (mCounter + mFrequencyShift * time);
        mPhase = (int) ((mPhase + (mCounter >> NOISE_PHASE_SHIFT)) & NOISE_TABLE_MASK);
        mCounter &= NOISE_PHASE_MASK;
    }

    public void restoreFrequency() {
        mFrequencyShift = (int) mNoiseFrequency;
    }

    @Override
    public double getNextSample() {
        double value = sTable[mPhase];
        addPhase(1);
        return value;
    }

    @Override
    public double getFrequency() {
        return mFrequency;
    }

    @Override
    public void setFrequency(double value) {
        mFrequency = value;
    }

    public void setNoiseFrequency(double value) {
        mNoiseFrequency = value * (1 << NOISE_PHASE_SHIFT);
        mFrequencyShift = (int) mNoiseFrequency;
    }

    public Boolean getResettingPhase() {
        return mResettingPhase;
    }

    public void setResettingPhase(Boolean resettingPhase) {
        mResettingPhase = resettingPhase;
    }

    private static Boolean sInitialized = false;
    private static double[] sTable = new double[TABLE_LENGTH];
    private double mNoiseFrequency;
    private long mCounter;
    private Boolean mResettingPhase;
}
