//
//  FJNPersistentWindowController.m
//  FJNCoreData
//
//  Created by FUJIDANA on 06/08/17.
//  Copyright 2006-2008 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "FJNPersistentWindowController.h"
#import "FJNCoreDataAppDelegate.h"


@interface FJNPersistentWindowController ()

+ (NSURL *)URLWithPersistentStoreDirectory:(NSString *)directory filename:(NSString *)filename;
- (void)askToForceCloseWithError:(NSError *)error callbackSelector:(SEL)callback;

@end


@implementation FJNPersistentWindowController

#pragma mark Class methods

+ (void)reviewChangesAndQuitEnumeration:(BOOL)cont
{
	if (cont)
	{
		NSArray *windows = [NSApp windows];
		unsigned count = [windows count];
		while (count--)
		{
			NSWindow *window = [windows objectAtIndex:count];
			FJNPersistentWindowController *windowController = [window windowController];
			if (windowController && [windowController isKindOfClass:[FJNPersistentWindowController class]])
			{
				NSManagedObjectContext *context = [windowController managedObjectContext];
				NSError *error;
				if ([context hasChanges] && [context save:&error] == NO)
				{
					[windowController askToForceCloseWithError:error
											  callbackSelector:@selector(reviewChangesAndQuitEnumeration:)];
					return;
				}
			}
		}
	}
	[NSApp replyToApplicationShouldTerminate:cont];
}

+ (NSURL *)URLWithPersistentStoreDirectory:(NSString *)directory filename:(NSString *)filename
{
	NSArray *paths = NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES);
	NSString *basePath = ([paths count] > 0) ? [paths objectAtIndex:0] : NSTemporaryDirectory();
	NSString *directoryPath = [basePath stringByAppendingPathComponent:directory];
	
	NSFileManager *fileManager = [NSFileManager defaultManager];
	
	if (![fileManager fileExistsAtPath:directoryPath isDirectory:NULL])
	{
		[fileManager createDirectoryAtPath:directoryPath attributes:nil];
	}
	return [NSURL fileURLWithPath:[directoryPath stringByAppendingPathComponent:filename]];
}

#pragma mark Initializing methods

- (id)initWithWindow:(NSWindow *)window storeDirectory:(NSString *)directory filename:(NSString *)filename type:(NSString *)type
{
	self = [super initWithWindow:window];
	if (self != nil)
	{
		_pStoreURL  = [[[self class] URLWithPersistentStoreDirectory:directory filename:filename] retain];
		_pStoreType = [type copy];
	}
	return self;
}

- (id)initWithWindowNibName:(NSString *)windowNibName storeDirectory:(NSString *)directory filename:(NSString *)filename type:(NSString *)type
{
	self = [super initWithWindowNibName:windowNibName];
	if (self != nil)
	{
		_pStoreURL  = [[[self class] URLWithPersistentStoreDirectory:directory filename:filename] retain];
		_pStoreType = [type copy];
	}
	return self;
}

- (id)initWithWindowNibName:(NSString *)windowNibName owner:(id)owner storeDirectory:(NSString *)directory filename:(NSString *)filename type:(NSString *)type
{
	self = [super initWithWindowNibName:windowNibName owner:owner];
	if (self != nil)
	{
		_pStoreURL  = [[[self class] URLWithPersistentStoreDirectory:directory filename:filename] retain];
		_pStoreType = [type copy];
	}
	return self;
}

- (id)initWithWindowNibPath:(NSString *)windowNibPath owner:(id)owner storeDirectory:(NSString *)directory filename:(NSString *)filename type:(NSString *)type
{
	self = [super initWithWindowNibPath:windowNibPath owner:owner];
	if (self != nil)
	{
		_pStoreURL  = [[[self class] URLWithPersistentStoreDirectory:directory filename:filename] retain];
		_pStoreType = [type copy];
	}
	return self;
}

- (void)dealloc
{
    [_managedObjectContext release], _managedObjectContext = nil;
    [_persistentStoreCoordinator release], _persistentStoreCoordinator = nil;
    [_managedObjectModel release], _managedObjectModel = nil;
	[_pStoreURL release], _pStoreURL = nil;
	[_pStoreType release], _pStoreType = nil;
    [super dealloc];
}

#pragma mark Accessor methods

- (NSManagedObjectModel *)managedObjectModel
{
	if (_managedObjectModel != nil) return _managedObjectModel;
	
//	NSMutableSet *allBundles = [[NSMutableSet alloc] init];
//	[allBundles addObject: [NSBundle mainBundle]];
//	[allBundles addObjectsFromArray:[NSBundle allFrameworks]];
//	
//	_managedObjectModel = [[NSManagedObjectModel mergedModelFromBundles:[allBundles allObjects]] retain];
//	[allBundles release];
	_managedObjectModel = [[NSManagedObjectModel mergedModelFromBundles:nil] retain];
	
	return _managedObjectModel;
}

- (NSPersistentStoreCoordinator *)persistentStoreCoordinator
{
	if (_persistentStoreCoordinator != nil) return _persistentStoreCoordinator;
	
	NSError *error;
	_persistentStoreCoordinator = [[NSPersistentStoreCoordinator alloc] initWithManagedObjectModel:[self managedObjectModel]];
	
	if (![_persistentStoreCoordinator addPersistentStoreWithType:_pStoreType
												   configuration:nil
															 URL:_pStoreURL
														 options:nil
														   error:&error])
	{
		[[NSApplication sharedApplication] presentError:error];
	}
	
	return _persistentStoreCoordinator;
}

- (NSManagedObjectContext *)managedObjectContext
{
	if (_managedObjectContext != nil) return _managedObjectContext;
	
	NSPersistentStoreCoordinator *coordinator = [self persistentStoreCoordinator];
	if (coordinator != nil)
	{
		_managedObjectContext = [[NSManagedObjectContext alloc] init];
		[_managedObjectContext setPersistentStoreCoordinator:coordinator];
	}
	
	return _managedObjectContext;
}

- (NSURL *)URLForPersistentStore
{
	return _pStoreURL;
}

#pragma mark Methods overriding NSWindowController

- (void)windowDidLoad
{
	[super windowDidLoad];
	[[self window] setDelegate:self];
}

#pragma mark Mehtods delegated by NSWindow

- (NSUndoManager *)windowWillReturnUndoManager:(NSWindow *)window
{
	return [[self managedObjectContext] undoManager];
}

- (BOOL)windowShouldClose:(id)window
{
	if ([_managedObjectContext commitEditing] == NO) return NO;
	
	NSError *error;
	if ([_managedObjectContext hasChanges] == NO || [_managedObjectContext save:&error]) return YES;
	
	[self askToForceCloseWithError:error callbackSelector:NULL];
	return NO;
}

- (void)askToForceCloseWithError:(NSError *)error callbackSelector:(SEL)callback
{
	[[self window] makeKeyAndOrderFront:nil];
	
	NSAlert *alert =[[[NSAlert alloc] init] autorelease];
	[alert setAlertStyle:NSWarningAlertStyle];
	[alert setMessageText:NSLocalizedStringFromTable(@"The database could not saved because invalid data were found. Do you want to close the window anyway?", @"CoreDataLocalizable", @"windowCloseAlert.message")];
	[alert setInformativeText:NSLocalizedStringFromTable(@"Your changes will be lost if you close the window.", @"CoreDataLocalizable", @"windowCloseAlert.information")];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Close anyway", @"CoreDataLocalizable", @"windowCloseAlert.closeButton")];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Cancel", @"CoreDataLocalizable", @"alert.cancelButton")];
	if (error)
	{
		[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Review Details...", @"CoreDataLocalizable", @"windowCloseAlert.detailsButton")];
	}
	NSMutableDictionary *contextInfo = [[NSMutableDictionary alloc] initWithCapacity:2];
	if (callback) [contextInfo setObject:[NSValue valueWithPointer:callback] forKey:@"callback"];
	if (error) [contextInfo setObject:error forKey:@"error"];
	
	[alert beginSheetModalForWindow:[self window]
					  modalDelegate:self
					 didEndSelector:@selector(forceCloseAlertDidEnd:returnCode:contextInfo:)
						contextInfo:(void *)contextInfo];
}

- (void)forceCloseAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	SEL callback = NULL;
	NSError *error = nil;
	if (contextInfo && [(id)contextInfo isKindOfClass:[NSDictionary class]])
	{
		error = [(NSDictionary *)contextInfo objectForKey:@"error"];
		callback = [[(NSDictionary *)contextInfo objectForKey:@"callback"] pointerValue];
		[(NSDictionary *)contextInfo autorelease]; 
	}
	
    if (returnCode == NSAlertFirstButtonReturn) // close anyway
	{
		// close window without asking the user for confirmation
		[[self managedObjectContext] processPendingChanges];
		[[self managedObjectContext] rollback];
		[self close];
		if (callback) (void (*)(id, SEL, BOOL))objc_msgSend([self class], callback, YES);
	}
	else if (returnCode == NSAlertSecondButtonReturn) // cancel
	{
		// Send callback indicating save cancel
        if (callback)
			(void (*)(id, SEL, BOOL))objc_msgSend([self class], callback, NO);
    }
	else if (returnCode == NSAlertThirdButtonReturn) // review details...
	{
		// Send callback indicating save cancel
        if (callback)
			(void (*)(id, SEL, BOOL))objc_msgSend([self class], callback, NO);
		if (error)
			[[alert window] close];
			[self presentError:error
				modalForWindow:[self window]
					  delegate:nil
			didPresentSelector:NULL
				   contextInfo:NULL];
	}
}

#pragma mark Action methods

- (IBAction)saveAction:(id)sender
{
	NSError *error = nil;
	if ([[self managedObjectContext] save:&error] == NO)
	{
		[self presentError:error
			modalForWindow:[self window]
				  delegate:nil
		didPresentSelector:NULL
			   contextInfo:NULL];
	}
}

- (IBAction)revertAction:(id)sender
{
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Revert", @"CoreDataLocalizable", @"revertAlert.revertButton")];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Cancel", @"CoreDataLocalizable", @"alert.cancelButton")];
	[alert setMessageText:NSLocalizedStringFromTable(@"Do you want to revert to the most recently saved version of the database?", @"CoreDataLocalizable", @"revertAlert.message")];
	[alert setInformativeText:NSLocalizedStringFromTable(@"Your current changes will be lost.", @"CoreDataLocalizable", @"revertAlert.information")];	
	[alert setAlertStyle:NSInformationalAlertStyle];
	
	[alert beginSheetModalForWindow:[self window]
					  modalDelegate:self
					 didEndSelector:@selector(revertActionAlertDidEnd:returnCode:contextInfo:)
						contextInfo:NULL];
}

- (void)revertActionAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSAlertFirstButtonReturn)
	{
		[[self managedObjectContext] processPendingChanges];
		[[self managedObjectContext] rollback];
	}
}

#pragma mark Methods overriding NSResponder

// Based on "NSPersistentDocument Core Data Tutorial
// > Localizing and Customizing Model Property Names and Error Messages"
// > Customizing the Document Alert Panel", 
// But I modified several part so that unmodified errors should be passed to superclass.
// (See "Handling Received Errors" section in "Error Handling Programming Guide For Cocoa".)

- (NSError *)willPresentError:(NSError *)inError
{
	if ([[inError domain] isEqualToString:NSCocoaErrorDomain] && [inError code] == NSValidationMultipleErrorsError)
	{
		NSArray *detailedErrors = [[inError userInfo] objectForKey:NSDetailedErrorsKey];
		
		// present error messages for up to 3 validation errors at a time.
		unsigned numErrors = [detailedErrors count];
		NSMutableString *errorString = [NSMutableString stringWithFormat:NSLocalizedStringFromTable(@"%u validation errors have occurred", @"CoreDataLocalizable", nil), numErrors];
		if (numErrors > 3)
		{
			[errorString appendFormat:NSLocalizedStringFromTable(@".\nThe first 3 are:\n", @"CoreDataLocalizable", nil)];
		}
		else
		{
			[errorString appendFormat:@":\n"];
		}
		unsigned i, displayErrors = numErrors > 3 ? 3 : numErrors;
		for (i = 0; i < displayErrors; i++)
		{
			[errorString appendFormat:@"%@\n",
			 [[detailedErrors objectAtIndex:i] localizedDescription]];
		}
		
		// Create a new error with the new userInfo
		NSMutableDictionary *newUserInfo = [NSMutableDictionary dictionaryWithDictionary:[inError userInfo]];
		[newUserInfo setObject:errorString forKey:NSLocalizedDescriptionKey];
		
		return [NSError errorWithDomain:[inError domain] code:[inError code] userInfo:newUserInfo];  
	}
	return [super willPresentError:inError];
}

#pragma mark Methods implementing NSToolbarItemValidation protocol

- (BOOL)validateUserInterfaceItem:(id <NSValidatedUserInterfaceItem>)theItem
{
	if ([theItem action] == @selector(saveAction:) || [theItem action] == @selector(revertAction:))
	{
		return [[self managedObjectContext] hasChanges];
	}
	return YES;
}

@end
