package jp.kirikiri.tjs2;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.DoubleBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import java.util.ArrayList;


/**
 * TJS2 バイトコードを読み込んで、ScriptBlock を返す
 *
 */
public class ByteCodeLoader {

	private static final boolean LOAD_SRC_POS = false;

	public static final int FILE_TAG_LE = ('T') | ('J'<<8) | ('S'<<16) | ('2'<<24);
	public static final int VER_TAG_LE = ('1') | ('0'<<8) | ('0'<<16) | (0<<24);
	private static final int OBJ_TAG_LE = ('O') | ('B'<<8) | ('J'<<16) | ('S'<<24);
	private static final int DATA_TAG_LE = ('D') | ('A'<<8) | ('T'<<16) | ('A'<<24);

	private static final int TYPE_VOID = 0;
	private static final int TYPE_OBJECT = 1;
	private static final int TYPE_INTER_OBJECT = 2;
	private static final int TYPE_STRING = 3;
	private static final int TYPE_OCTET = 4;
	private static final int TYPE_REAL = 5;
	private static final int TYPE_BYTE = 6;
	private static final int TYPE_SHORT = 7;
	private static final int TYPE_INTEGER = 8;
	private static final int TYPE_LONG = 9;
	private static final int TYPE_INTER_GENERATOR = 10; // temporary
	private static final int TYPE_UNKNOWN = -1;

	static private byte[] mByteArray;
	static private short[] mShortArray;
	static private int[] mIntArray;
	static private long[] mLongArray;
	static private double[] mDoubleArray;
	static private long[] mDoubleTmpArray;
	static private String[] mStringArray;
	static private ByteBuffer[] mByteBufferArray;
	static private short[] mVariantTypeData;

	static private final int MIN_BYTE_COUNT = 64;
	static private final int MIN_SHORT_COUNT = 64;
	static private final int MIN_INT_COUNT = 64;
	static private final int MIN_DOUBLE_COUNT = 8;
	static private final int MIN_LONG_COUNT = 8;
	static private final int MIN_STRING_COUNT = 1024;

	static private final int MIN_VARIANT_DATA_COUNT = 400*2;

	static private boolean mDeleteBuffer;
	static private byte[] mReadBuffer;
	static private final int MIN_READ_BUFFER_SIZE = 160 * 1024;

	static class ObjectsCache {
		public InterCodeObject[] mObjs;
		public ArrayList<VariantRepalace> mWork;
		public int[] mParent;
		public int[] mPropSetter;
		public int[] mPropGetter;
		public int[] mSuperClassGetter;
		public int[][] mProperties;

		private static final int MIN_COUNT = 500;
		public void create( int count ) {
			if( count < MIN_COUNT )
				count = MIN_COUNT;

			if( mWork == null ) mWork = new ArrayList<VariantRepalace>();
			mWork.clear();

			if( mObjs == null || mObjs.length < count ) {
				mObjs = new InterCodeObject[count];
				mParent = new int[count];
				mPropSetter = new int[count];
				mPropGetter = new int[count];
				mSuperClassGetter = new int[count];
				mProperties = new int[count][];
			}
		}
		public void release() {
			mWork = null;
			mObjs = null;
			mParent = null;
			mPropSetter = null;
			mPropGetter = null;
			mSuperClassGetter = null;
			mProperties = null;
		}
	}
	static private ObjectsCache mObjectsCache;
	static public void initialize() {
		mDeleteBuffer = false;
		mReadBuffer = null;
		mByteArray = null;
		mShortArray = null;
		mIntArray = null;
		mLongArray = null;
		mDoubleArray = null;
		mDoubleTmpArray = null;
		mStringArray = null;
		mByteBufferArray = null;
		mObjectsCache = new ObjectsCache();
		mVariantTypeData = null;
	}
	public static void finalizeApplication() {
		mDeleteBuffer = true;
		mReadBuffer = null;
		mByteArray = null;
		mShortArray = null;
		mIntArray = null;
		mLongArray = null;
		mDoubleArray = null;
		mDoubleTmpArray = null;
		mStringArray = null;
		mByteBufferArray = null;
		mObjectsCache = null;
		mVariantTypeData = null;
	}
	static public void allwaysFreeReadBuffer() {
		mDeleteBuffer = true;
		mReadBuffer = null;
		mByteArray = null;
		mShortArray = null;
		mIntArray = null;
		mLongArray = null;
		mDoubleArray = null;
		mDoubleTmpArray = null;
		mStringArray = null;
		mByteBufferArray = null;
		mObjectsCache.release();
		mVariantTypeData = null;
	}

	public ByteCodeLoader() {
	}

	public ScriptBlock readByteCode( TJS owner, String name, BinaryStream input ) throws TJSException {
		try {
			int size = (int) input.getSize();
			if( mReadBuffer == null || mReadBuffer.length < size) {
				int buflen = size < MIN_READ_BUFFER_SIZE ? MIN_READ_BUFFER_SIZE : size;
				mReadBuffer = new byte[buflen];
			}
			byte[] databuff = mReadBuffer;
			//byte[] databuff = new byte[size];
			input.read(databuff);
			input.close();
			input  = null;
			ByteBuffer bbuff = ByteBuffer.wrap(databuff);
			bbuff.order(ByteOrder.LITTLE_ENDIAN);
			IntBuffer ibuff = bbuff.asIntBuffer();
			int tag = ibuff.get(); // TJS2
			if( tag != FILE_TAG_LE ) return null;
			int ver = ibuff.get(); // 100'\0'
			if( ver != VER_TAG_LE ) return null;

			int filesize = ibuff.get();
			if( filesize != size ) return null;

			tag = ibuff.get(); // DATA
			if( tag != DATA_TAG_LE ) return null;
			size = ibuff.get();
			readDataArea( databuff, ibuff.position()<<2, size );
			//readDataArea2( databuff, ibuff.position()*4, size );
			ibuff.position( ibuff.position() + (size>>>2) - 1 - 1 );

			tag = ibuff.get(); // OBJS
			if( tag != OBJ_TAG_LE ) return null;
			int objsize = ibuff.get();
			ScriptBlock block = new ScriptBlock(owner, name, 0, null, null );
			ShortBuffer sbuff = bbuff.asShortBuffer();
			readObjects( block, ibuff, sbuff );
			//readObjects2( block, ibuff );
			return block;
		} finally {
			if( mDeleteBuffer ) {
				mReadBuffer = null;
				mByteArray = null;
				mShortArray = null;
				mIntArray = null;
				mLongArray = null;
				mDoubleArray = null;
				mDoubleTmpArray = null;
				mStringArray = null;
				mByteBufferArray = null;
				mObjectsCache.release();
				mVariantTypeData = null;
			}
		}
	}

	/**
	 * InterCodeObject へ置換するために一時的に覚えておくクラス
	 */
	static class VariantRepalace {
		public Variant Work;
		public int Index;
		public VariantRepalace( Variant w, int i ) {
			Work = w;
			Index = i;
		}
	}

	private void readObjects( ScriptBlock block, IntBuffer ibuff, ShortBuffer sbuff ) throws TJSException {
		int toplevel = ibuff.get();
		int objcount = ibuff.get();
		//Log.v("test","count:"+objcount);
		mObjectsCache.create(objcount);
		//InterCodeObject[] objs = new InterCodeObject[objcount];
		//ArrayList<VariantRepalace> work = new ArrayList<VariantRepalace>();
		InterCodeObject[] objs = mObjectsCache.mObjs;
		ArrayList<VariantRepalace> work = mObjectsCache.mWork;

		//int[] parent = new int[objcount];
		//int[] propSetter = new int[objcount];
		//int[] propGetter = new int[objcount];
		//int[] superClassGetter = new int[objcount];
		//int[][] properties = new int[objcount][];
		int[] parent = mObjectsCache.mParent;
		int[] propSetter = mObjectsCache.mPropSetter;
		int[] propGetter = mObjectsCache.mPropGetter;
		int[] superClassGetter = mObjectsCache.mSuperClassGetter;
		int[][] properties = mObjectsCache.mProperties;
		for( int o = 0; o < objcount; o++ ) {
			int tag = ibuff.get();
			if( tag != FILE_TAG_LE ) {
				throw new TJSException(Error.ByteCodeBroken);
			}
			int objsize = ibuff.get();
			parent[o] = ibuff.get();
			int name = ibuff.get();
			int contextType = ibuff.get();
			int maxVariableCount = ibuff.get();
			int variableReserveCount = ibuff.get();
			int maxFrameCount = ibuff.get();
			int funcDeclArgCount = ibuff.get();
			int funcDeclUnnamedArgArrayBase = ibuff.get();
			int funcDeclCollapseBase = ibuff.get();
			propSetter[o] = ibuff.get();
			propGetter[o] = ibuff.get();
			superClassGetter[o] = ibuff.get();

			int count = ibuff.get();

			LongBuffer srcpos;
			if( LOAD_SRC_POS ) {
				int[] codePos = new int[count];
				int[] srcPos = new int[count];
				ibuff.get( codePos );
				ibuff.get( srcPos );
				// codePos/srcPos は今のところ使ってない、ソート済みなので、longにする必要はないが……
				ByteBuffer code2srcpos = ByteBuffer.allocate(count<<3);
				code2srcpos.order(ByteOrder.LITTLE_ENDIAN);
				srcpos = code2srcpos.asLongBuffer();
				for( int i = 0; i < count; i++ ) {
					srcpos.put( ((long)(codePos[i]) << 32) | (long)(srcPos[i]) );
				}
				srcpos.flip();
			} else {
				ibuff.position(ibuff.position()+(count<<1));
				srcpos = null;
			}

			count = ibuff.get();
			short[] code = new short[count];
			sbuff.position( ibuff.position()<<1 );
			sbuff.get( code );
			if( (count & 1) == 1 ) {
				sbuff.get();
			}
			ibuff.position( sbuff.position() >>> 1);
			count = ibuff.get();
			int vcount = count<<1;
			if( mVariantTypeData == null || mVariantTypeData.length < vcount ) {
				mVariantTypeData = new short[vcount];
			}
			short[] data = mVariantTypeData;
			//short[] data = new short[count<<1];
			sbuff.position( ibuff.position()<<1 );
			sbuff.get( data, 0, vcount );
			ibuff.position( sbuff.position() >>> 1);

			Variant[] vdata = new Variant[count];
			int datacount = count;
			Variant tmp;
			for( int i = 0; i < datacount; i++ ) {
				int pos = i << 1;
				int type = data[pos];
				int index = data[pos+1];
				switch( type ) {
				case TYPE_VOID:
					vdata[i] = new Variant(); // null
					break;
				case TYPE_OBJECT:
					vdata[i] = new Variant(null,null); // null Array Dictionary はまだサポートしていない TODO
					break;
				case TYPE_INTER_OBJECT:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_INTER_GENERATOR:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_STRING:
					vdata[i] = new Variant( mStringArray[index] );
					break;
				case TYPE_OCTET:
					vdata[i] = new Variant( mByteBufferArray[index] );
					break;
				case TYPE_REAL:
					vdata[i] = new Variant( mDoubleArray[index] );
					break;
				case TYPE_BYTE:
					vdata[i] = new Variant( mByteArray[index] );
					break;
				case TYPE_SHORT:
					vdata[i] = new Variant( mShortArray[index] );
					break;
				case TYPE_INTEGER:
					vdata[i] = new Variant( mIntArray[index] );
					break;
				case TYPE_LONG:
					vdata[i] = new Variant( mLongArray[index] );
					break;
				case TYPE_UNKNOWN:
				default:
					vdata[i] = new Variant(); // null;
					break;
				}
			}
			count = ibuff.get();
			int[] scgetterps = new int[count];
			ibuff.get( scgetterps );
			// properties
			count = ibuff.get();
			if( count > 0 ) {
				properties[o] = new int[count<<1];
				ibuff.get( properties[o] );
			}

			//IntVector superpointer = IntVector.wrap( scgetterps );
			InterCodeObject obj = new InterCodeObject( block, mStringArray[name], contextType, code, vdata, maxVariableCount, variableReserveCount,
					maxFrameCount, funcDeclArgCount, funcDeclUnnamedArgArrayBase, funcDeclCollapseBase, true, srcpos, scgetterps );
			//objs.add(obj);
			objs[o] = obj;
		}
		Variant val = new Variant();
		for( int o = 0; o < objcount; o++ ) {
			InterCodeObject parentObj = null;
			InterCodeObject propSetterObj = null;
			InterCodeObject propGetterObj = null;
			InterCodeObject superClassGetterObj = null;

			if( parent[o] >= 0 ) {
				parentObj = objs[parent[o]];
			}
			if( propSetter[o] >= 0 ) {
				propSetterObj = objs[propSetter[o]];
			}
			if( propGetter[o] >= 0 ) {
				propGetterObj = objs[propGetter[o]];
			}
			if( superClassGetter[o] >= 0 ) {
				superClassGetterObj = objs[superClassGetter[o]];
			}
			objs[o].setCodeObject(parentObj, propSetterObj, propGetterObj, superClassGetterObj );
			if( properties[o] != null ) {
				InterCodeObject obj = parentObj; // objs.get(o).mParent;
				int[] prop = properties[o];
				int length = prop.length >>> 1;
				for( int i = 0; i < length; i++ ) {
					int pos = i << 1;
					int pname = prop[pos];
					int pobj = prop[pos+1];
					val.set( objs[pobj] );
					obj.propSet( Interface.MEMBERENSURE|Interface.IGNOREPROP, mStringArray[pname], val, obj );
				}
				properties[o] = null;
			}
		}
		int count = work.size();
		for( int i = 0; i < count; i++ ) {
			VariantRepalace w = work.get(i);
			w.Work.set( objs[w.Index] );
		}
		work.clear();
		InterCodeObject top = null;
		if( toplevel >= 0 ) {
			top = objs[toplevel];
		}
		block.setObjects( top, objs, objcount );
	}

	private void readDataArea( byte[] buff, int offset, int size ) {
		ByteBuffer bbuff = ByteBuffer.wrap(buff);
		bbuff.order(ByteOrder.LITTLE_ENDIAN);
		IntBuffer ibuff = bbuff.asIntBuffer();
		ibuff.position(offset>>>2);
		int count = ibuff.get();
		if( count > 0 ) {
			if( mByteArray == null || mByteArray.length < count ) {
				int c = count < MIN_BYTE_COUNT ? MIN_BYTE_COUNT : count;
				mByteArray = new byte[c];
			}
			bbuff.position(offset+4);
			bbuff.get(mByteArray,0,count);
			int stride = ( count + 3 ) >>> 2;
			ibuff.position(ibuff.position()+stride);
		}
		count = ibuff.get();
		if( count > 0 ) {	// load short
			if( mShortArray == null || mShortArray.length < count ) {
				int c = count < MIN_SHORT_COUNT ? MIN_SHORT_COUNT : count;
				mShortArray = new short[c];
			}
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*2 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			ShortBuffer sb = sbuff.asShortBuffer();
			sb.get(mShortArray,0,count);
			int stride = ( count + 1 ) >>> 1;
			ibuff.position(ibuff.position()+stride);
		}
		count = ibuff.get();
		if( count > 0 ) {
			if( mIntArray == null || mIntArray.length < count ) {
				int c = count < MIN_INT_COUNT ? MIN_INT_COUNT : count;
				mIntArray = new int[c];
			}
			ibuff.get(mIntArray,0,count);
		}
		count = ibuff.get();
		if( count > 0 ) {	// load long
			if( mLongArray == null || mLongArray.length < count ) {
				int c = count < MIN_LONG_COUNT ? MIN_LONG_COUNT : count;
				mLongArray = new long[c];
			}
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()<<2, count<<3 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			lb.get(mLongArray,0,count);
			ibuff.position(ibuff.position()+(count<<1));
		}
		count = ibuff.get();
		if( count > 0 ) {	// load double
			if( mDoubleArray == null || mDoubleArray.length < count ) {
				int c = count < MIN_DOUBLE_COUNT ? MIN_DOUBLE_COUNT : count;
				mDoubleArray = new double[c];
			}

			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()<<2, count<<3 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			if( mDoubleTmpArray == null || mDoubleTmpArray.length < count ) {
				int c = count < MIN_DOUBLE_COUNT ? MIN_DOUBLE_COUNT : count;
				mDoubleTmpArray = new long[c];
			}
			long[] tmp = mDoubleTmpArray;
			lb.get(tmp,0,count);
			for( int i = 0; i < count; i++ ) {
				mDoubleArray[i] = Double.longBitsToDouble(tmp[i]);
			}
			ibuff.position(ibuff.position()+(count<<1));
		}
		count = ibuff.get();
		if( count > 0 ) {
			if( mStringArray == null || mStringArray.length < count ) {
				int c = count < MIN_STRING_COUNT ? MIN_STRING_COUNT : count;
				mStringArray = new String[c];
			}

			bbuff.position(0);
			CharBuffer cbuff = bbuff.asCharBuffer();
			int stroff = (ibuff.position()<<1)+2;
			for( int i = 0; i < count; i++ ) {
				int len = ibuff.get();
				cbuff.position(stroff);
				mStringArray[i] = TJS.mapGlobalStringMap(cbuff.subSequence(0, len).toString());
				len = ((len+1)>>>1)<<1;
				stroff += len + 2;
				ibuff.position( (stroff>>1) - 1 );
			}
		}

		count = ibuff.get();
		if( count > 0 ) {
			if( mByteBufferArray == null || mByteBufferArray.length < count )
				mByteBufferArray = new ByteBuffer[count];

			int octetoff = (ibuff.position()<<2) + 4;
			for( int i = 0; i < count; i++ ) {
				int len = ibuff.get();
				mByteBufferArray[i] = ByteBuffer.allocate(len);
				mByteBufferArray[i].put( buff, octetoff, len );
				len = ((len+3)>>>2)<<2;
				octetoff += len + 4;
				ibuff.position( (octetoff>>>2)-4 );
			}
		}
	}
}
