//	Copyright (c) 2008 itok ( http://itok.jp/ , http://110k.net/ )
//	All rights reserved.
//
//	Redistribution and use in source and binary forms, with or without modification, 
//	are permitted provided that the following conditions are met:
//
//	- Redistributions of source code must retain the above copyright notice, 
//	  this list of conditions and the following disclaimer.
//	- Redistributions in binary form must reproduce the above copyright notice, 
//	  this list of conditions and the following disclaimer in the documentation 
//	  and/or other materials provided with the distribution.
//	- Neither the name of itok nor the names of its contributors may be used to endorse 
//	  or promote products derived from this software without specific prior written permission.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS 
//	AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
//	INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
//	AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
//	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
//	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
//	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
//	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
//	EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "PUServiceManager.h"
#import "PUCommon.h"

@interface PUServiceManager (Private)

-(void) changeService;
-(void) updateProgress:(double)value;

@end

@implementation PUServiceManager

-(id) init
{
	if (self = [super init]) {
		[NSBundle loadNibNamed:@"Service" owner:self];
		
		static const char* service_names[] = {
			"FlickrService",
			"PhotoZouService",
			"FotolifeService",
			"MixiService",
			"TwitterService",
			nil
		};
		int i;		
		NSMutableDictionary* services = [NSMutableDictionary dictionary];
		for (i = 0; ; i++) {
			const char* name = service_names[i];
			if (!name) {
				break;
			}
			
			PUService* service = [[NSClassFromString([NSString stringWithCString:name]) alloc] init];
			[services setObject:service forKey:[service title]];
			[service release];
		}
		m_serviceDic = [[NSDictionary alloc] initWithDictionary:services];
		[self setValue:[m_serviceDic allKeys] forKey:@"services"];

		
		PUUserDefaults* defaults = [PUUserDefaults standardUserDefaults];
		id title = [defaults objectForKey:@"CurrentService"];
		if (!title) {
			title = [_services objectAtIndex:0];
		}
		[self setValue:title forKey:@"serviceTitle"];

		// サイズは共通
		[self setValue:[NSArray arrayWithObjects:LOCALIZED(@"Original"), LOCALIZED(@"More Small"), LOCALIZED(@"Small"), LOCALIZED(@"Middle"), LOCALIZED(@"Large"), LOCALIZED(@"More Large"), nil] forKey:@"sizes"];
		
		m_resArr = [[NSMutableArray alloc] init];
	}
	return self;
}

-(void) dealloc
{
	[m_serviceDic release];
	[m_resArr release];
	
	[super dealloc];
}

-(NSView*) view
{
	return o_view;
}

-(void) startExport:(NSArray*)photos delegate:(id)delegate
{
	[m_resArr removeAllObjects];
	
	// 認証確認
	if (![_service checkAuth]) {
		NSRunAlertPanel(LOCALIZED(@"Please enter username and password"), @"", @"OK", @"", @"");
		return;
	}
	
	if (!(photos && [photos count] > 0)) {
		return;
	}
	
	m_delegate = delegate;

	// プログレス表示
	[self setValue:LOCALIZED(@"Converting...") forKey:@"progressText"];
	[o_prgBar setMaxValue:2.0];
	[o_prgBar setMinValue:0.0];
	[o_prgBar setDoubleValue:0.0];
	[o_prgBar setUsesThreadedAnimation:YES];
	[o_prgBar setIndeterminate:YES];
	[o_prgBar startAnimation:nil];

	[NSApp beginSheet:o_prgWindow modalForWindow:[o_view window] modalDelegate:nil didEndSelector:nil contextInfo:nil];		

	// フォーマット変換
	int num = [photos count], cnt = 0;
	NSFileManager* manager = [NSFileManager defaultManager];
	[manager removeFileAtPath:TMP_PATH handler:nil];
	[manager createDirectoryAtPath:TMP_PATH attributes:nil];	
	NSMutableArray* convertedPhotos = [NSMutableArray array];
	PUPhoto* photo;
	NSString* path;
	PUImageFormat origFormat, selectedFormat;
	PUImageSize selectedSize;
	NSEnumerator* enume = [photos objectEnumerator];
	while (photo = [enume nextObject]) {
		if (path = [photo valueForKey:@"path"]) {
			origFormat = [PUImageProcessor formatWithPath:path];
			selectedFormat = [[self valueForKeyPath:@"service.param.format"] intValue];
			selectedSize = [[self valueForKeyPath:@"service.param.size"] intValue];
//			NSLog(@"original format %d", origFormat);
//			NSLog(@"selected format %d", selectedFormat);
//			NSLog(@"selected size %d", selectedSize);
			if (![[self valueForKeyPath:@"service.spec.formats"] containsObject:[NSNumber numberWithInt:origFormat]] && selectedFormat == kPUImageFormat_Original) {
				NSRunAlertPanel(LOCALIZED(@"Invalid Format"), [path lastPathComponent], @"OK", @"", @"");
				goto loop_end;
			}
			
			if (selectedFormat != kPUImageFormat_Original) {
				NSString* dir = [TMP_PATH stringByAppendingPathComponent:[NSString stringWithFormat:@"%d", [NSDate timeIntervalSinceReferenceDate]]];
				[manager createDirectoryAtPath:dir attributes:nil];
				NSString* toPath = [dir stringByAppendingPathComponent:[[[path lastPathComponent] stringByDeletingPathExtension] stringByAppendingPathExtension:[[PUImageProcessor stringFromFormat:selectedFormat] lowercaseString]]];
				NSLog(@"converted path %@", toPath);
				if (![PUImageProcessor convertImage:path toPath:toPath format:selectedFormat size:selectedSize]) {
					NSRunAlertPanel(LOCALIZED(@"Convert Failed"), [path lastPathComponent], @"OK", @"", @"");
					goto loop_end;
				}
				[photo setValue:toPath forKey:@"path"];
			}
			[convertedPhotos addObject:photo];
		}
loop_end :
		[self updateProgress:(++cnt / (double)num)];
	}
	
	NSArray* reqs = [_service requestsWithPhotos:convertedPhotos];
	if (reqs && [reqs count] > 0) {
		[self setValue:LOCALIZED(@"Uploading...") forKey:@"progressText"];
		m_mreq = [[PUMultiRequest alloc] init];
		[m_mreq setDelegate:self];
		[m_mreq start:reqs];
	} else {
		[NSApp endSheet:o_prgWindow];
		[o_prgWindow orderOut:nil];		
		[o_prgBar stopAnimation:nil];
		NSRunAlertPanel(LOCALIZED(@"Upload Failed"), @"", @"OK", @"", @"");

		if (m_delegate && [m_delegate respondsToSelector:@selector(serviceManagerDidFinishExport:)]) {
			[m_delegate serviceManagerDidFinishExport:self];
			m_delegate = nil;
		}
	}
}

#pragma mark ___ delegate ___

-(void) multiRequestDidCompleted:(PUMultiRequest*)multiReq result:(int)result
{
	[NSApp endSheet:o_prgWindow];
	[o_prgWindow orderOut:nil];
	
	[o_prgBar stopAnimation:nil];
	
	switch (result) {
		case kPUMultiRequest_Success :
		{
			NSURL* url = [_service URLForLastUpload:m_resArr];
			if (url) {
				int ret = NSRunAlertPanel(LOCALIZED(@"Upload Completed"), LOCALIZED(@"browse uploaded photos?"), @"Yes", @"No", @"");
				if (ret == NSAlertDefaultReturn) {
					[[NSWorkspace sharedWorkspace] openURL:url];
				}
			} else {
				NSRunAlertPanel(LOCALIZED(@"Upload Completed"), @"", @"OK", @"", @"");
			}
		}
			break;
		case kPUMultiRequest_Cancel :
			NSRunAlertPanel(LOCALIZED(@"Upload Canceled"), @"", @"OK", @"", @"");
			break;
		case kPUMultiRequest_Fail :
			NSRunAlertPanel(LOCALIZED(@"Upload Failed"), @"", @"OK", @"", @"");
			break;
	}
	
	if (m_delegate && [m_delegate respondsToSelector:@selector(serviceManagerDidFinishExport:)]) {
		[m_delegate serviceManagerDidFinishExport:self];
		m_delegate = nil;
	}
	
	[m_resArr removeAllObjects];
	[multiReq release];
	m_mreq = nil;
}

-(void) multiRequest:(PUMultiRequest*)multiReq didCompletedOne:(PURequest*)request withProgress:(double)progress
{
	[self updateProgress:(progress + 1.0)];
	
	NSError* err = nil;
	id res = [request responseData:&err];
	if (!res || err) {
		if (err) {
			NSRunAlertPanel([NSString stringWithFormat:LOCALIZED(@"Error Occurred : error code %d"), [err code]], [err localizedDescription], LOCALIZED(@"OK"), @"", @"");
		} else {
			NSRunAlertPanel(LOCALIZED(@"Unknown Error Occurred"), @"", LOCALIZED(@"OK"), @"", @"");				
		}
	} else {
		[m_resArr addObject:res];
	}
}

-(BOOL) multiRequest:(PUMultiRequest*)multiReq willStartOne:(PURequest*)request
{
	return YES;
}


-(void) setServiceTitle:(id)value
{
	if (_serviceTitle) {
		[_serviceTitle autorelease];
	}
	_serviceTitle = [value retain];
	[self setValue:[m_serviceDic objectForKey:_serviceTitle] forKey:@"service"];

	PUUserDefaults* defaults = [PUUserDefaults standardUserDefaults];
	[defaults setObject:_serviceTitle forKey:@"CurrentService"];
	[defaults synchronize];
}

-(void) setService:(id)value
{
	if (_service) {
		[_service deactivate];
		[_service release];
	}
	_service = [value retain];
	[_service activate];
	[self changeService];
}

-(void) setFormatStr:(id)value
{
	if (_formatStr) {
		[_formatStr release];
	}
	PUImageFormat format;
	if ([value isEqualTo:LOCALIZED(@"Original")]) {
		format = kPUImageFormat_Original;
	} else {
		format = [PUImageProcessor formatFromString:value];
	}	
	if (![[self valueForKeyPath:@"service.spec.formats"] containsObject:[NSNumber numberWithInt:format]]) {
		value = LOCALIZED(@"Original");
	}
	_formatStr = [value retain];

	if ([_formatStr isEqualTo:LOCALIZED(@"Original")]) {
		format = kPUImageFormat_Original;
	} else {
		format = [PUImageProcessor formatFromString:_formatStr];
	}
	[self setValue:[NSNumber numberWithInt:format] forKeyPath:@"service.param.format"];
}

-(void) setSizeIdx:(id)value
{
	if (!(value && [value isKindOfClass:[NSNumber class]])) {
		return;
	}
	_sizeIdx = [value intValue];
	
	PUImageSize size = (_sizeIdx == 0) ? kPUImageSize_Original : _sizeIdx - 1;
	[self setValue:[NSNumber numberWithInt:size] forKeyPath:@"service.param.size"];
}

-(IBAction) cancelUpload:(id)sender
{
	if (m_mreq) {
		[m_mreq cancel];
	}
}

@end

@implementation PUServiceManager (Private)

-(void) changeService
{
	NSView* cstmView = [_service customView];
	if (cstmView) {
		[o_cstmView addSubview:cstmView];
	}
	
	NSMutableArray* formats = [NSMutableArray array];
	[formats addObject:LOCALIZED(@"Original")];
	NSEnumerator* enume = [[self valueForKeyPath:@"service.spec.formats"] objectEnumerator];
	id obj;
	while (obj = [enume nextObject]) {
		[formats addObject:LOCALIZED([PUImageProcessor stringFromFormat:[obj intValue]])];
	}
	[self setValue:formats forKey:@"formats"];
	
	PUImageFormat format = [[self valueForKeyPath:@"service.param.format"] intValue];
	PUImageSize size = [[self valueForKeyPath:@"service.param.size"] intValue];
	[self setValue:LOCALIZED([PUImageProcessor stringFromFormat:format]) forKey:@"formatStr"];
	[self setValue:[NSNumber numberWithInt:(size == kPUImageSize_Original) ? 0 : size + 1] forKey:@"sizeIdx"];
}

-(void) updateProgress:(double)value
{
	if ([o_prgBar doubleValue] == 0) {
		[o_prgBar setIndeterminate:NO];
	} else if (value >= [o_prgBar maxValue]) {
		[o_prgBar setIndeterminate:YES];
	}
	[o_prgBar setDoubleValue:value];	
}

@end

