//	Copyright (c) 2008 itok ( http://itok.jp/ , http://110k.net/ )
//	All rights reserved.
//
//	Redistribution and use in source and binary forms, with or without modification, 
//	are permitted provided that the following conditions are met:
//
//	- Redistributions of source code must retain the above copyright notice, 
//	  this list of conditions and the following disclaimer.
//	- Redistributions in binary form must reproduce the above copyright notice, 
//	  this list of conditions and the following disclaimer in the documentation 
//	  and/or other materials provided with the distribution.
//	- Neither the name of itok nor the names of its contributors may be used to endorse 
//	  or promote products derived from this software without specific prior written permission.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS 
//	AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
//	INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
//	AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
//	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
//	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
//	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
//	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
//	EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// こちらのコードを参考にしました
// http://www.mactech.com/articles/mactech/Vol.19/19.04/CocoaApplications/index.html

#import "PURequest.h"

#define   READ_SIZE         1024
NSString* PURequestErrorDomain = @"PURequestErrorDomain";

static CFTimeInterval s_timeout = 60.0;
static NSString* s_username = nil;
static NSString* s_password = nil;

static void* CFClientRetain(void* selfPtr);
static void CFClientRelease(void* selfPtr);
static CFStringRef CFClientDescribeCopy(void* selfPtr);
static void StreamReadCallback(CFReadStreamRef stream, CFStreamEventType type, void* userData);

static CFStreamClientContext s_context = {
	0, nil,
	CFClientRetain, 
	CFClientRelease, 
	CFClientDescribeCopy
};

@interface PURequest (Private)
-(void) getResultCode;
-(void) closeOutMessaging;
-(void) informDelegateOfCompletion;
-(void) appendData:(unsigned char*)bytes length:(int)length;
-(CFHTTPMessageRef) message;
-(void) setPostData:(NSDictionary*)dict boundary:(NSString*)boundary;
@end

@implementation PURequest

+(CFTimeInterval) timeoutInterval
{
	return s_timeout;
}

+(void) setTimeoutInterval:(CFTimeInterval)newInterval
{
	s_timeout = newInterval;
}

+(void) setUsername:(NSString*)username
{
	if (s_username) {
		[s_username autorelease];
	}
	s_username = [username copy];
}

+(NSString*) username
{
	return s_username;
}

+(void) setPassword:(NSString*)password
{
	if (s_password) {
		[s_password autorelease];
	}
	s_password = [password copy];
}

+(NSString*) password
{
	return s_password;
}

-(id) initWithURL:(NSURL*)url withMethod:(NSString*)method
{
	return [self initWithURL:url withMethod:method postData:nil];
}

-(id) initWithURL:(NSURL*)url withMethod:(NSString*)method postData:(NSDictionary*)initialData
{
	if (self = [super init]) {
		m_replyStream = NULL;
		m_rcvData = [[NSMutableData alloc] init];
		m_statusCode = kPURequest_Incomplete;
		m_context = s_context;
		m_context.info = self;
		m_timeoutTimer = nil;
		if (initialData) {
			m_postData = [[NSMutableDictionary alloc] initWithDictionary:initialData];
		} else {
			m_postData = [[NSMutableDictionary alloc] init];
		}
		if (!m_postData) {
			[self release];
			return nil;
		}
		
		// Set up the POST message and its headers
		m_message = CFHTTPMessageCreateRequest(kCFAllocatorDefault, (CFStringRef)method, (CFURLRef)url, kCFHTTPVersion1_1);
		if (!m_message) {
			[self release];
			return nil;
		}
		CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("User-Agent"), CFSTR("Generic/1.0 (Mac_PowerPC)"));
//		CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("Content-Type"), CFSTR("application/x-www-form-urlencoded"));
		CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("Host"), (CFStringRef) [url host]);
		CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("Accept"), CFSTR("text/html"));
	}
	return self;
}

-(void) dealloc
{
	if (m_message) {
		CFRelease(m_message);
		m_message = NULL;
	}
	
	[m_rcvData release];
	[m_postData release];
	if (m_timeoutTimer) {
		[m_timeoutTimer invalidate];
		[m_timeoutTimer dealloc];
		m_timeoutTimer = nil;
	}
	[super dealloc];
}

-(void) start
{
	if (m_statusCode != kPURequest_Incomplete) {

	}
	m_statusCode = kPURequest_NotReplied;
	if ([m_postData count] > 0) {
		[self setPostData:m_postData boundary:@"__boundary___"];
	}
		
	// Initialize the CFReadStream that will make the request and manage the reply
	m_replyStream = CFReadStreamCreateForHTTPRequest(kCFAllocatorDefault, m_message);

	// Register the CFReadStream's callback client
	CFReadStreamSetClient(m_replyStream, (kCFStreamEventHasBytesAvailable |kCFStreamEventErrorOccurred | kCFStreamEventEndEncountered), StreamReadCallback, &m_context);
	// Schedule the CFReadStream for service by the current run loop
	CFReadStreamScheduleWithRunLoop(m_replyStream, CFRunLoopGetCurrent(), kCFRunLoopCommonModes);
	// Fire off the request
	CFReadStreamOpen(m_replyStream);
	// Watch for timeout
	m_timeoutTimer = [NSTimer
					scheduledTimerWithTimeInterval:s_timeout
					target:self
					selector:@selector(messageTimedOut:)
					userInfo:nil
					repeats:NO];
	[m_timeoutTimer retain];
}

-(void) cancel
{
	[self closeOutMessaging];
	m_statusCode = kPURequest_Canceled;
	[self informDelegateOfCompletion];
}

-(int) statusCode 
{
	return m_statusCode; 
}

-(id) responseData:(NSError**)error
{
	if (m_statusCode > 0) {
		if (m_statusCode / 100 == 2) {
			// 200番台
			*error = nil;
		} else {
			NSString* content = [[[NSString alloc] initWithData:m_rcvData encoding:NSUTF8StringEncoding] autorelease];
			NSDictionary* userInfo = (content) ? [NSDictionary dictionaryWithObjectsAndKeys:content, NSLocalizedDescriptionKey, nil] : nil;
			*error = [NSError errorWithDomain:PURequestErrorDomain code:m_statusCode userInfo:userInfo];
		}
		return m_rcvData;
	} else {
		*error = [NSError errorWithDomain:PURequestErrorDomain code:m_statusCode userInfo:nil];
	}
	return nil;
}

-(id) delegate 
{
	return m_delegate; 
}

-(void) setDelegate:(id)aDelegate
{ 
	m_delegate = aDelegate; 
}

@end

@implementation PURequest (Private)


-(void) getResultCode
{
	if (m_replyStream) {
		CFHTTPMessageRef reply = (CFHTTPMessageRef)CFReadStreamCopyProperty(m_replyStream, kCFStreamPropertyHTTPResponseHeader);
		
		//   Pull the status code from the headers
		if (reply) {
			m_statusCode = CFHTTPMessageGetResponseStatusCode(reply);
			CFRelease(reply);
		}
	}
}

-(void) closeOutMessaging
{
	if (m_replyStream) {
		// Close the read stream.
		CFReadStreamClose(m_replyStream);
		// Deregister the callback client (learned this from WWDC session 805)
		CFReadStreamSetClient(m_replyStream, 0, NULL, NULL);
		// Take the stream out of the run loop
		CFReadStreamUnscheduleFromRunLoop(m_replyStream, CFRunLoopGetCurrent(), kCFRunLoopCommonModes);
		// Deallocate the stream pointer
		CFRelease(m_replyStream);
		// Throw the spent pointer away
		m_replyStream = NULL;
	}
	
	if (m_timeoutTimer) {
		[m_timeoutTimer invalidate];
		[m_timeoutTimer release];
		m_timeoutTimer = nil;
	}
}

-(void) informDelegateOfCompletion
{
	if (m_delegate && [m_delegate respondsToSelector:@selector(request:didCompletedWithResult:)]) {
		[m_delegate request:self didCompletedWithResult:m_statusCode];
	}
}

-(void) appendData:(unsigned char*)bytes length:(int)length
{
//	NSLog(@"appendData %d", length);
	[m_rcvData appendBytes:bytes length:length];
	m_statusCode = kPURequest_ReplyInProgress;
	[m_timeoutTimer setFireDate:[NSDate dateWithTimeIntervalSinceNow:s_timeout]];
}

-(CFHTTPMessageRef) message
{
	return m_message;
}

-(void) setPostData:(NSDictionary*)dict boundary:(NSString*)boundary
{
	NSMutableData* data = [[NSMutableData alloc] init];
	//	NSLog(@"%p", data);
    id key;
    NSEnumerator* enume = [dict keyEnumerator];
    while (key = [enume nextObject]) {
        id value = [dict valueForKey:key];
		//		NSLog(@"%@ : %@", key, value);
		[data appendData:STR_TO_DATA(([NSString stringWithFormat:@"--%@\r\n", boundary]))];
        if ([value isKindOfClass:[NSString class]]) {
			// 文字列
            [data appendData:STR_TO_DATA(([NSString stringWithFormat:@"Content-Disposition: form-data; name=\"%@\"\r\n\r\n", key]))];
            [data appendData:STR_TO_DATA((value))];
        } else if ([value isKindOfClass:[NSDictionary class]]) {
			// 辞書
			id url = [value objectForKey:KEY_URL];
			id type = [value objectForKey:KEY_TYPE];
			if (url && type && 
				[url isKindOfClass:[NSURL class]] && [url isFileURL] && 
				[type isKindOfClass:[NSString class]] && [type length] > 0) {
				[data appendData:STR_TO_DATA(([NSString stringWithFormat:@"Content-Disposition: form-data; name=\"%@\"; filename=\"%@\"\r\n", key, [[url path] lastPathComponent]]))];
				[data appendData:STR_TO_DATA(([NSString stringWithFormat:@"Content-Type: %@\r\n\r\n", type]))];
				[data appendData:[NSData dataWithContentsOfURL:url]];
			}
		}
		[data appendData:STR_TO_DATA(([NSString stringWithString:@"\r\n"]))];
		//		NSLog(@"%@", [[[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding] autorelease]);
    }
	[data appendData:STR_TO_DATA(([NSString stringWithFormat:@"--%@--\r\n", boundary]))];
	//	NSLog(@"%@", [dict description]);
	//	NSLog(@"%@", [[[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding] autorelease]);
	
	CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("Content-type"), (CFStringRef)[NSString stringWithFormat:@"multipart/form-data; boundary=%@", boundary]);
	// Put the post data in the body of the query
	CFHTTPMessageSetBody(m_message, (CFDataRef)data);
	// Now that we know how long the query body is, put the length in the header
	CFHTTPMessageSetHeaderFieldValue(m_message, CFSTR("Content-Length"), (CFStringRef)[NSString stringWithFormat: @"%d", [data length]]);
	
	[data release];
}


-(void) messageTimedOut:(NSTimer*)theTimer
{
	m_statusCode = kPURequest_TimedOut;
	[self closeOutMessaging];
	[self informDelegateOfCompletion];
}

@end

void* CFClientRetain(void* selfPtr)
{
	PURequest* object = (PURequest*)selfPtr;
	return [object retain];
}

void CFClientRelease(void* selfPtr)
{
	PURequest* object = (PURequest*)selfPtr;
	[object release];
}

CFStringRef CFClientDescribeCopy(void* selfPtr)
{
	PURequest* object = (PURequest*)selfPtr;
	return (CFStringRef)[[object description] retain];
}

void StreamReadCallback(CFReadStreamRef stream, CFStreamEventType type, void* userData)
{
	PURequest* object = (PURequest*)userData;	
	switch (type) {
		case kCFStreamEventHasBytesAvailable: 
		{
			unsigned char buffer[READ_SIZE];
			CFIndex bytesRead = CFReadStreamRead(stream, buffer, READ_SIZE);
			
			if (bytesRead > 0) {
				[object appendData:buffer length:bytesRead];
			}      
		}
			break;
		case kCFStreamEventErrorOccurred:
		case kCFStreamEventEndEncountered:
			[object getResultCode];
			[object closeOutMessaging];
			[object informDelegateOfCompletion];
			break;
		default:
			break;
	}
}
