// /home/tarai/Projects/gsaw/gsaw/MainWindow.cs created with MonoDevelop
// User: tarai at 20:03 2008/04/20
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//
using System;
using System.Collections;
using Gtk;
using Gdk;

using Holo.Image;
using Holo.Image.Generic;
using Holo.Image.Tiled;
using Holo.Operation;
using Lawrence.Common;
using Lawrence.Tool;
using Lawrence.History;
using Lawrence.View;

public partial class MainWindow: Gtk.Window
{	
	// for image operation
	private Holo.Image.Image image;
	private IPaintTool tool;
	private OperationHistory history;
	private Lawrence.View.Viewport viewport;
	private Hashtable tools;
	
	// for brush update timing
	private BaseBrushProvider brushProvider;
	private long lastUpdateTime;

	// for layer
	private Gtk.ListStore layerListStore;
	private ArrayList layerThumbnails;
	
	// for palette (temporary)
	private int paletteBaseIndex = 0;
	private ArrayList palette;
	
	public MainWindow (): base (Gtk.WindowType.Toplevel)
	{
		Build ();
		Setup();
		SetupPalette();

		SetupLayerList();
		
		view.ExposeEvent += OnViewExpose;
		view.ButtonPressEvent += OnButtonPress;
		view.MotionNotifyEvent += OnMouseMove;
		view.ButtonReleaseEvent += OnButtonRelease;
		view.ExtensionEvents=Gdk.ExtensionMode.Cursor;
		
		SetupDevices();
		
		UpdateScrollbar();
	}
	
	protected void SetupPalette() {
		paletteBaseIndex = 0;
		palette = new ArrayList();
	}
	
	protected void OnDeleteEvent (object sender, DeleteEventArgs a)
	{
		Application.Quit ();
		a.RetVal = true;
	}

	protected virtual void onInputDialogActivated (object sender, System.EventArgs e)
	{
	}
	
	public void Setup() {
		lastUpdateTime = 0;
		image = NewImage();
		viewport = NewViewport(image);
		history = new Lawrence.History.OperationHistory();
		SetupTools();
	}
	
	protected void SetupDevices() {
		Console.WriteLine("Screen.Width={0}", Display.DefaultScreen.Width);
		Gdk.Device[] devices = Display.ListDevices();
		foreach (Gdk.Device dev in devices) {
			if ((dev.Source == Gdk.InputSource.Pen ||
			    dev.Source == Gdk.InputSource.Eraser ||
			    dev.Source == Gdk.InputSource.Cursor) &&
			    dev.NumAxes > 5 //Workaround to avoid the problem: Default mouse is reported as 'Pen' type device.
			    ) {
				dev.SetMode(Gdk.InputMode.Screen);
			}
		}
	}
	
	protected void SetupLayerList() {
		layerThumbnails = new ArrayList();

		Gtk.TreeViewColumn layerColumn;
		Gtk.CellRenderer layerCell;

		// Data Model.
		layerListStore = new Gtk.ListStore(typeof(bool), typeof(bool), typeof(Gdk.Pixbuf), typeof(string));
		layerListView.Model = layerListStore;

		// Visibility Column
		layerColumn = new Gtk.TreeViewColumn ();
		layerColumn.Title = "V";
		layerCell = new Gtk.CellRendererToggle ();
		((Gtk.CellRendererToggle)layerCell).Toggled += OnLayerVisibilityToggled;
		layerColumn.PackStart (layerCell, true);
		layerColumn.AddAttribute(layerCell, "active", 0);
		
		layerListView.AppendColumn(layerColumn);
		
		// Clipped Column
		layerColumn = new Gtk.TreeViewColumn ();
		layerColumn.Title = "C";
		layerCell = new Gtk.CellRendererToggle ();
		((Gtk.CellRendererToggle)layerCell).Toggled += OnLayerClippedToggled;
		layerColumn.PackStart (layerCell, true);
		layerColumn.AddAttribute(layerCell, "active", 1);
		
		layerListView.AppendColumn(layerColumn);
		// Thumbnail Column
		layerColumn = new Gtk.TreeViewColumn ();
		layerColumn.Title = "Thumbnail";
		layerCell = new Gtk.CellRendererPixbuf ();
		layerColumn.PackStart (layerCell, true);
		layerColumn.AddAttribute(layerCell, "pixbuf", 2);
		
		layerListView.AppendColumn(layerColumn);
		// Name Column
		layerColumn = new Gtk.TreeViewColumn ();
		layerColumn.Title = "Name";
		layerCell = new Gtk.CellRendererText ();
		layerColumn.PackStart (layerCell, true);
		layerColumn.AddAttribute(layerCell, "text", 3);
		
		layerListView.AppendColumn(layerColumn);
		
		// Add List Items for Initial Layers.
		foreach (Layer layer in image.Layers) {
			Thumbnail thumb;
			Gdk.Pixbuf pixbuf;
			NewThumbnail(layer, out thumb, out pixbuf);
			layerThumbnails.Add(thumb);
			layerListStore.AppendValues(layer.Visible, layer.Clipped, pixbuf, layer.Name);
			thumb.Refresh();
		}
	}
	
	protected class ThumbnailUpdator {
		public delegate void RefreshHandler(Thumbnail sender);

		private Gdk.Pixbuf pixbuf;
		private RefreshHandler onRefresh;
		
		public RefreshHandler OnRefresh { 
			get { return onRefresh; }
			set { onRefresh = value; }
		}
		
		public ThumbnailUpdator(MainWindow window, Gdk.Pixbuf pixbuf) {
			this.pixbuf = pixbuf;			
		}
		
		public void Refresh(Thumbnail sender) {
			System.Runtime.InteropServices.Marshal.Copy(sender.Buffer, 0, pixbuf.Pixels, sender.Buffer.Length);
			if (onRefresh != null)
				onRefresh(sender);
		}
	}
	
	protected void NewThumbnail(Layer layer, out Thumbnail thumb, out Gdk.Pixbuf pixbuf) {
		thumb = new Thumbnail(layer.Surface);
		pixbuf = new Gdk.Pixbuf(Gdk.Colorspace.Rgb, true, 8, 64, 64);
		ThumbnailUpdator updator = new ThumbnailUpdator(this, pixbuf);
		thumb.OnRefresh = updator.Refresh;
		updator.OnRefresh = OnThumbnailRefreshed;
	}
	
	protected void OnThumbnailRefreshed(Thumbnail sender) {
		int index = layerThumbnails.IndexOf(sender);
		Gtk.TreePath path = new Gtk.TreePath(""+index);
		Gtk.TreeIter iter;
		layerListStore.GetIter(out iter, path);
		layerListStore.EmitRowChanged(path, iter);
		Console.WriteLine("Refresh");
	}
	
	protected void OnLayerVisibilityToggled(object sender, Gtk.ToggledArgs args) {
		TreeSelection selection = layerListView.Selection;

		TreeModel model;
		TreeIter iter;

		// The iter will point to the selected row
		if(selection.GetSelected(out model, out iter)) {
			int index = (int)model.GetPath(iter).Indices[0];
			if (index != 0) {
				image.ActiveLayer.Visible = !image.ActiveLayer.Visible;
				layerListStore.SetValue(iter, 0, image.ActiveLayer.Visible);
			}
		}
		
	}
	
	protected void OnLayerClippedToggled(object sender, Gtk.ToggledArgs args) {
		TreeSelection selection = layerListView.Selection;

		TreeModel model;
		TreeIter iter;

		// The iter will point to the selected row
		if(selection.GetSelected(out model, out iter)) {
			int index = (int)model.GetPath(iter).Indices[0];
			if (index != 0) {
				image.ActiveLayer.Clipped = !image.ActiveLayer.Clipped;
				layerListStore.SetValue(iter, 1, image.ActiveLayer.Clipped);
			}
		}
		
	}

	public Holo.Image.Image NewImage() {
		int w = 1280;
		int h = 1600;
		Holo.Image.Image image = new Holo.Image.Image(w, h);

		// Base Surface
		Layer layer = (Layer)image.GetLayerAt(0);
		BlendOperation over =	new BlendOperation();
		over.Operator += over.Apply;
		layer.Operation = over;
		image.ActiveLayerIndex = 0;
		image.QueueUpdatedArea(0, 0, w, h);
		return image;
	}
	
	public Lawrence.View.Viewport NewViewport(Holo.Image.Image image) {
		Lawrence.View.Viewport viewport = new Lawrence.View.Viewport(image);
		viewport.SetViewport(view.Allocation.Width, view.Allocation.Height, 0, 0, 2, 2);
		viewport.OnUpdateView += OnUpdateImage;
		return viewport;
	}
	
	public Layer NewLayer(Holo.Image.Image image) {
		byte[] primaryColor = {0, 0, 0};
		byte[] primaryAlpha = {0};
		IRasterFactory baseFactory = new CompactGenericRasterFactory(primaryColor, primaryAlpha, true); 
		IRasterFactory factory = new TiledRasterFactory(baseFactory);
		TiledSurface surface = new TiledSurface(0, 0, image.Width, image.Height, factory);
		Layer layer = new Layer(image, surface);
		layer.Name = "Layer#"+(image.LayerCount+1);
		BlendOperation over =	new BlendOperation();
		over.Operator += over.Apply;
		layer.Operation = over;
		layer.Surface.MakeRastersReadable(0, 0, image.Width, image.Height);
		return layer;
	}
	
	public void SetupTools() {
		tools = new Hashtable();

		BrushTool brush = new BrushTool();
		brushProvider = new BaseBrushProvider(); 
		brush.OverWrite = false;
		brushProvider.Setup(brush);
		ToBrush(brush);
		tools.Add("brush", brush);
		
		FloodFillTool fillTool = new FloodFillTool();
		BlendOperation blender = new BlendOperation();
		blender.Operator = blender.Apply;
		fillTool.Apply = blender;
		tools.Add("floodfill", fillTool);
		
		tool = brush;
	}
	
	protected void ToBrush(BrushTool tool) {
		BlendOperation over = new BlendOperation();
		over.Operator += over.Apply;
		tool.Apply = over;
//		over.Opacity = brushOpacity;
	}
	
	protected void ToEraser(BrushTool tool) {
		EraseOperation eraser = new EraseOperation();
		eraser.Operator += eraser.Apply;
		tool.Apply = eraser;
//		eraser.Opacity = brushOpacity;
	}
	
	public void OnViewExpose(object obj, ExposeEventArgs args) {
		Gdk.Rectangle area = args.Event.Area;
		if (area.X < 0)
			area.X = 0;
		if (area.Y < 0)
			area.Y = 0;
		if (area.X + area.Width >= image.Width)
			area.Width = image.Width - area.X;
		if (area.Y + area.Height >= image.Height)
			area.Height = image.Height - area.Y;
//		Console.WriteLine("Update:"+area.X+","+area.Y+","+area.Width+","+area.Height);
		viewport.RenderTo(args.Event.Window, area);
		args.RetVal = false;
	}

	void OnButtonPress(object obj, ButtonPressEventArgs args) {
		Gdk.EventButton ev = args.Event;
		PaintContext context = new PaintContext();
		context.ActiveImage = image;
		context.ActiveLayer = image.ActiveLayer;
		context.History     = history;
		ImageCoordinate coords = new ImageCoordinate();
		coords.X = ev.X;
		coords.Y = ev.Y;
		if (!ev.Device.GetAxis (ev.Axes, AxisUse.Pressure, out coords.Pressure)) {
			coords.Pressure = 1.0;
		}
		viewport.ViewportCoordToImageCoord(coords);
		coords.XTilt = 0;
		coords.YTilt = 0;
		coords.Wheel = 0;

		tool.OnStartStroke(context, coords, 0);
		args.RetVal = true;
	}
	
	public void OnMouseMove(object obj, MotionNotifyEventArgs args) {
		Gdk.EventMotion ev = args.Event;
//		Console.WriteLine("Move:"+ev.ToString());
		if ((ev.State & Gdk.ModifierType.Button1Mask) != 0 ) {
			PaintContext context = new PaintContext();
			context.ActiveImage = image;
			context.ActiveLayer = image.ActiveLayer;
			context.History     = history;
			ImageCoordinate coords = new ImageCoordinate();
			coords.X = ev.X;
			coords.Y = ev.Y;
			coords.Pressure = 1.0;
			if (!ev.Device.GetAxis (ev.Axes, AxisUse.Pressure, out coords.Pressure)) {
				coords.Pressure = 1.0;
			}
			viewport.ViewportCoordToImageCoord(coords);
			coords.XTilt = 0;
			coords.YTilt = 0;
			coords.Wheel = 0;
//			Console.WriteLine("coords={0},{1},pressure={2}\n",coords.X,coords.Y, coords.Pressure);
			tool.OnUpdateStroke(context, coords, 0);
		}
		args.RetVal = true;
	}
	

	void OnButtonRelease(object obj, ButtonReleaseEventArgs args) {
		Gdk.EventButton ev = args.Event;
		PaintContext context = new PaintContext();
		context.ActiveImage = image;
		context.ActiveLayer = image.ActiveLayer;
		context.History     = history;
		ImageCoordinate coords = new ImageCoordinate();
		coords.X = ev.X;
		coords.Y = ev.Y;
		if (!ev.Device.GetAxis (ev.Axes, AxisUse.Pressure, out coords.Pressure)) {
			coords.Pressure = 1.0;
		}
		viewport.ViewportCoordToImageCoord(coords);
		coords.XTilt = 0;
		coords.YTilt = 0;
		coords.Wheel = 0;

		tool.OnFinishStroke(context, coords, 0);
		((Thumbnail)layerThumbnails[image.ActiveLayerIndex]).Refresh();
		args.RetVal = true;
	}
	
	public void OnUpdateImage(int minX, int minY, int maxX, int maxY) {
		long time = DateTime.Now.Ticks;
		if (time - lastUpdateTime > 100) {
			view.GdkWindow.InvalidateRect(new Gdk.Rectangle(minX, minY, maxX - minX, maxY - minY), true);
			view.GdkWindow.ProcessUpdates(true);
			lastUpdateTime = time;
		} else {
			view.QueueDrawArea(minX, minY, maxX - minX, maxY - minY);
		}
	}

	protected virtual void OnBrushButtonClicked (object sender, System.EventArgs e)
	{
		tool = (IPaintTool)tools["brush"];
		ToBrush((BrushTool)tool);
	}

	protected virtual void OnEraserButtonClicked (object sender, System.EventArgs e)
	{
		tool = (IPaintTool)tools["brush"];
		ToEraser((BrushTool)tool);
	}

	protected virtual void OnFloodFillButtonClicked (object sender, System.EventArgs e)
	{
		tool = (IPaintTool)tools["floodfill"];
	}

	protected virtual void OnLayerListViewCursorChanged (object sender, System.EventArgs e)
	{
		Console.WriteLine("Cursor Changed");
		TreeSelection selection = (sender as TreeView).Selection;

		TreeModel model;
		TreeIter iter;

		// The iter will point to the selected row
		if(selection.GetSelected(out model, out iter)) {
			Console.WriteLine("Path of selected row = {0}", model.GetPath(iter));
			image.ActiveLayerIndex = (int)model.GetPath(iter).Indices[0];
		}
	}

	protected virtual void OnLayerListViewRowActivated (object o, Gtk.RowActivatedArgs args)
	{
		Console.WriteLine("Activated");
	}

	protected virtual void OnLayerListViewToggleCursorRow (object o, Gtk.ToggleCursorRowArgs args)
	{
		Console.WriteLine("ToggleCursorRow");
	}

	protected virtual void OnAddLayerButtonClicked (object sender, System.EventArgs e)
	{
		Layer newLayer = NewLayer(image);
		image.InsertLayerAfter(image.ActiveLayerIndex, newLayer);
		Thumbnail thumb;
		Gdk.Pixbuf pixbuf;
		NewThumbnail(newLayer, out thumb, out pixbuf);
		layerThumbnails.Insert(image.ActiveLayerIndex + 1, thumb);
		Gtk.TreeIter iter = layerListStore.Insert(image.ActiveLayerIndex + 1);
		layerListStore.SetValue(iter, 0, newLayer.Visible);
		layerListStore.SetValue(iter, 1, newLayer.Clipped);
		layerListStore.SetValue(iter, 2, pixbuf);
		layerListStore.SetValue(iter, 3, newLayer.Name);
		thumb.Refresh();
	}

	protected virtual void OnRemoveLayerButtonClicked (object sender, System.EventArgs e)
	{
		int index = image.ActiveLayerIndex;
		if (index == 0)
			return;
		
		if (image.RemoveLayerAt(index)) {
			Gtk.TreeIter iter;
			if (layerListStore.GetIterFromString(out iter, index.ToString()) ) {
				layerListStore.Remove(ref iter);
			} else {
				// Error
			}
		}
	}

	protected virtual void UpdateScrollbar() {
		hscrollbar.Adjustment.Upper = (int)(image.Width * (double)viewport.DestScale / viewport.SourceScale);	
		hscrollbar.Adjustment.PageSize = viewport.WidthOfView;
		hscrollbar.Adjustment.Change();
		vscrollbar.Adjustment.Upper = (int)(image.Height * (double)viewport.DestScale / viewport.SourceScale);
		vscrollbar.Adjustment.PageSize = viewport.HeightOfView;
		vscrollbar.Adjustment.Change();
	}
	
	protected virtual void OnViewConfigureEvent (object o, Gtk.ConfigureEventArgs args)
	{
		if (viewport != null)
			viewport.Resize(view.Allocation.Width, view.Allocation.Height);
		UpdateScrollbar();
	}
	
	private void SetColor(byte[] newColor) {
		byte[] newAlpha = { 0 };
		brushProvider.Color = newColor;
		//TBD. Should be moved to BaseBrushProvider.
		((BrushTool)tool).RasterFactory = new CompactMonochromeRasterFactory(newColor, newAlpha);
		foreach (byte[] color in palette) {
			if (color[0] == newColor[0] && color[1] == newColor[1] && color[2] == newColor[2]) {
				return ;
			}
		}
		palette.Add(newColor);
		if (paletteBaseIndex + 10 < palette.Count)
			paletteBaseIndex = palette.Count - 10;
		colortable.QueueDraw();
	}

	protected virtual void OnFgColorButtonColorSet (object sender, System.EventArgs e)
	{
		byte[] newColor = { (byte)(fgColorButton.Color.Red >> 8), (byte)(fgColorButton.Color.Green >> 8), (byte)(fgColorButton.Color.Blue >> 8) };
		SetColor(newColor);
	}

	protected virtual void OnOpacityScaleValueChanged (object sender, System.EventArgs e)
	{
		brushProvider.Opacity = opacityScale.Value / 100;
//		brushProvider.Opacity *= brushProvider.Opacity;
		//TBD
		BrushTool brush = (BrushTool)tool;
		if (brush != null && brush.Apply != null) {
			brush.Apply.Opacity = brushProvider.Opacity;
		}
	}

	protected virtual void OnEdgeScaleValueChanged (object sender, System.EventArgs e)
	{
		brushProvider.EdgeRate = edgeScale.Value / 100;
//		dabSurface = CreateDabSurface(brushRadius, brushRadius * 2, brushEdgeRate);
	}

	protected virtual void OnRadiusScaleValueChanged (object sender, System.EventArgs e)
	{
		brushProvider.Radius = radiusScale.Value;
//		dabSurface = CreateDabSurface(brushRadius, brushRadius * 2, brushEdgeRate);
		BrushTool brush = (BrushTool)tool;
		if (brush != null && brush.Apply != null) {
			brush.SteppingDistance = brushProvider.Radius * 2 * brushProvider.StepRate;
		}
	}

	protected virtual void OnHscrollbarMoveSlider (object o, Gtk.MoveSliderArgs args)
	{
		Console.WriteLine("H:MoveSlider");
	}

	protected virtual void OnVscrollbarMoveSlider (object o, Gtk.MoveSliderArgs args)
	{
		Console.WriteLine("V:MoveSlider");
	}

	protected virtual void OnVscrollbarValueChanged (object sender, System.EventArgs e)
	{
		Console.WriteLine("V:ValueChanged");
		viewport.MoveOffsetTo(viewport.OffsetX, (int)vscrollbar.Value);
	}

	protected virtual void OnHscrollbarValueChanged (object sender, System.EventArgs e)
	{
		Console.WriteLine("H:ValueChanged");
		viewport.MoveOffsetTo((int)hscrollbar.Value, viewport.OffsetY);
	}

	protected virtual void OnRotateClockwiseButtonClicked (object sender, System.EventArgs e)
	{
		viewport.Degree = (viewport.Degree + 10) % 360;
	}

	protected virtual void OnRotateCounterClockwiseButtonClicked (object sender, System.EventArgs e)
	{
		viewport.Degree = (viewport.Degree - 10) % 360;
	}

	protected virtual void OnOverwriteCheckClicked (object sender, System.EventArgs e)
	{
		((BrushTool)tool).OverWrite = !((BrushTool)tool).OverWrite;
	}
	
	private int GetColorIndex(object o) {
		if (o == colorarea1) {
			return 0;
		} else if (o == colorarea2) {
			return 1;
		} else if (o == colorarea3) {
			return 2;
		} else if (o == colorarea4) {
			return 3;
		} else if (o == colorarea5) {
			return 4;
		} else if (o == colorarea6) {
			return 5;
		} else if (o == colorarea7) {
			return 6;
		} else if (o == colorarea8) {
			return 7;
		} else if (o == colorarea9) {
			return 8;
		} else if (o == colorarea10) {
			return 9;
		}
		return -1;
	}

	protected virtual void OnColorareaExpose (object o, Gtk.ExposeEventArgs args)
	{
		int index = GetColorIndex(o);
		if (index < 0)
			return;
		if (index + paletteBaseIndex < palette.Count) {
			byte[] color = (byte[])palette[paletteBaseIndex + index];
			Gtk.DrawingArea widget = (Gtk.DrawingArea)o;
			Gdk.Rectangle area = args.Event.Area;
			Gdk.GC gc = new Gdk.GC(widget.GdkWindow);
			gc.RgbFgColor = new Gdk.Color(color[0], color[1], color[2]);
			widget.GdkWindow.DrawRectangle(gc, true, area);
		} else {
			return;
		}
	}

	protected virtual void OnColorAareaButtonPress (object o, Gtk.ButtonPressEventArgs args)
	{
		Console.WriteLine(o);
		int index = GetColorIndex(o);
		if (index < 0)
			return;
		if (index + paletteBaseIndex < palette.Count) {
			byte[] color = (byte[])palette[paletteBaseIndex + index];
			Console.WriteLine(color);
			fgColorButton.Color = new Gdk.Color(color[0], color[1], color[2]);
			SetColor(color);
		}
	}

	protected virtual void OnColortableUpButtonClicked (object sender, System.EventArgs e)
	{
		if (paletteBaseIndex > 0)
			paletteBaseIndex --;
		colortable.QueueDraw();
	}

	protected virtual void OnColortableDownButtonClicked (object sender, System.EventArgs e)
	{
		if (paletteBaseIndex < palette.Count - 10)
			paletteBaseIndex ++;
		colortable.QueueDraw();
	}

	protected virtual void OnUndoActionActivated (object sender, System.EventArgs e)
	{
		if (history != null)
			history.Undo();
	}

	protected virtual void OnRedoActionActivated (object sender, System.EventArgs e)
	{
		if (history != null)
			history.Redo();
	}

	protected virtual void OnZoomInActionActivated (object sender, System.EventArgs e)
	{
		if (viewport.DestScale >= viewport.SourceScale) {
			if (viewport.DestScale < viewport.SourceScale * 16) {
				viewport.Rescale(2, viewport.DestScale + 1);
			}
		} else {
			viewport.Rescale(viewport.SourceScale - 1, 2);
		}
		Console.WriteLine("zoom in : "+viewport.DestScale+"/"+viewport.SourceScale);
		UpdateScrollbar();
	}

	protected virtual void OnZoomOutActionActivated (object sender, System.EventArgs e)
	{
		if (viewport.DestScale <= viewport.SourceScale) {
			if (viewport.DestScale * 16 > viewport.SourceScale) {
				viewport.Rescale(viewport.SourceScale + 1, 2);
			}
		} else {
			viewport.Rescale(2, viewport.DestScale - 1);
		}
		Console.WriteLine("zoom out : "+viewport.DestScale+"/"+viewport.SourceScale);
		UpdateScrollbar();
	}

	protected virtual void OnZoomFitActionActivated (object sender, System.EventArgs e)
	{
		viewport.Rescale(2, 2);
		Console.WriteLine("zoom fit");
		UpdateScrollbar();
	}

	protected virtual void OnPropertiesActionToggled (object sender, System.EventArgs e)
	{
		sideviewFrame.Visible = propertiesAction.Active;
	}

	protected virtual void OnSerializeTestButtonClicked (object sender, System.EventArgs e)
	{
		Holo.Codec.ISurfaceEncoder encoder = new Holo.Codec.TRS.TRSEncoder();
		System.IO.FileStream file = new System.IO.FileStream("test.xml", System.IO.FileMode.Create);
		encoder.Encode(image.ActiveLayer.Surface, file);
		file.Close();
	}
}