// /home/tarai/Projects/gsaw/gsaw/BlendOperation.cs created with MonoDevelop
// User: tarai at 19:43 2008/04/29
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//
//#define USE_UNSAFE_OPS
//#define USE_EXTERNAL_OPS


using System;
using System.Runtime.InteropServices;

namespace Holo.Operation {
	using Holo.Image;
	
	// BlendOperation:
	// BlendOperation does following composite operation :
	//   a' = (a2 + a1 * (1 - a2))
	//   displayed c2 = premultiplied c2 = c2 * a2
	//   displayed c' = blend(displayed c2, c1, a1)
	//                = (c2 * a2 * (1 - a1)  + c1 - c1 * (1 - a1) )
	//                = (c2 * a2 + a1 * (c1 - c2 * a2))
	//                = (displayed c2 + a1 * (c1 - displayed c2))
	//   c' with alpha = displayed c' / a'
	//      = (c2 * a2 * (1 - a1)  + c1 * a1 ) / a'
	//      = (c2 * (a2 + a1 - a1 * a2) - c2 * a1 + c1 * a1) / a'
	//      = ( c2 * a' + a1 * (c1 - c2) ) / a'
	//      = c2 + a1 * (c1 - c2) / a'
	// where:
	//   underlying pixel is (c2=rgb, a2=alpha)
	//   overwrapping pixel is (c1=rgb, a1=alpha)
	//   resulting pixel is (c'=rgb, a'=alpha)
	//   blend(a, b, opacity) = b * opacity + (1 - opacity) * a
	public class BlendOperation : BasicOperation {
		
		public BlendOperation() {
		}
		
#if USE_EXTERNAL_OPS
		[DllImport("libwilmaops.so")]
		static extern void CBlend(byte[] src1Color, int src1Offset, int src1RowStride, int src1PixelStride, 
			                              int src1Channels, 
			                              byte[] src1AlphaValue, int src1AlphaOffset, int src1AlphaRowStride, int src1AlphaPixelStride,
						                  byte[] src2Color, int src2Offset, int src2RowStride, int src2PixelStride, 
			                              int src2Channels, 
			                              byte[] src2AlphaValue, int src2AlphaOffset, int src2AlphaRowStride, int src2AlphaPixelStride,
		                                  byte[] maskValue, int maskOffset, int maskRowStride, int maskPixelStride,
						                  byte[] destColor, int destOffset, int destRowStride, int destPixelStride,
						                  bool destHasAlpha, bool destHasColor, int destChannels,
						                  int width, int height, double opacity);
#elif USE_UNSAFE_OPS
		// Faster but unsafe implementation.
		unsafe private void UnsafeBlend(byte[] src1Color, int src1Offset, int src1RowStride, int src1PixelStride, 
			                              int src1Channels, 
			                              byte[] src1AlphaValue, int src1AlphaOffset, int src1AlphaRowStride, int src1AlphaPixelStride,
						                  byte[] src2Color, int src2Offset, int src2RowStride, int src2PixelStride, 
			                              int src2Channels, 
			                              byte[] src2AlphaValue, int src2AlphaOffset, int src2AlphaRowStride, int src2AlphaPixelStride,
		                                  byte[] maskValue, int maskOffset, int maskRowStride, int maskPixelStride,
		                                  byte[] destColor, int destOffset, int destRowStride, int destPixelStride,
						                  bool destHasAlpha, bool destHasColor, int destChannels,
						                  int width, int height, double opacity) {
			fixed(byte* s1c = src1Color, 
			      s2c = src2Color,
			      dc = destColor,
				  s1a = src1AlphaValue,
				  s2a = src2AlphaValue) {
				int x, y, c;
				byte* pSrc1ColorValue = s1c + src1Offset;
				byte* pSrc2ColorValue = s2c + src2Offset;
				byte* pDestColorValue = dc + destOffset;
				byte* pSrc1AlphaValue = s1a + src1AlphaOffset;
				byte* pSrc2AlphaValue = s2a + src2AlphaOffset;
				// Color buffer setting
				// Alpha buffer setting

				for (y = 0; y < height; y ++) {
					for (x = 0; x < width; x ++) {
						int src1Alpha = (int)(*pSrc1AlphaValue * opacity);
						int src2Alpha = (int)(*pSrc2AlphaValue);
						
						if (destHasAlpha) {
							int alpha = (int)src2Alpha + src1Alpha - ((int)src1Alpha * src2Alpha) / 255;
							if (alpha <= 0) {
								for (c = 0; c < destChannels; c ++)
									pDestColorValue[c] = 0;
							} else {
								for (c = 0; c < destChannels - 1; c ++) {
									pDestColorValue[c] = (byte)Math.Min(255, ((int)pSrc2ColorValue[c] + src1Alpha * (pSrc1ColorValue[c] - pSrc2ColorValue[c]) / alpha ) );
								}
								pDestColorValue[destChannels - 1] = (byte)(Math.Min(255, alpha));
							}
						} else {
							for (c = 0; c < destChannels; c ++)
								pDestColorValue[c] = (byte)Math.Min(255, ((int)pSrc2ColorValue[c] * src2Alpha * 255 + src1Alpha * (pSrc1ColorValue[c] * 255 - pSrc2ColorValue[c] * src2Alpha)) / (255 * 255) );
						}

						pDestColorValue += destPixelStride;
						pSrc1ColorValue += src1PixelStride;
						pSrc2ColorValue += src2PixelStride;
						pSrc1AlphaValue += src1AlphaPixelStride;
						pSrc2AlphaValue += src2AlphaPixelStride;
					}
					pDestColorValue += destRowStride - (width * destPixelStride);
					pSrc1ColorValue += src1RowStride - (width * src1PixelStride);
					pSrc2ColorValue += src2RowStride - (width * src2PixelStride);
					pSrc1AlphaValue += src1AlphaRowStride - (width * src1AlphaPixelStride);
					pSrc2AlphaValue += src2AlphaRowStride - (width * src2AlphaPixelStride);
				}
			}
		}
#else
		private void Blend(byte[] src1Pixels, int src1Offset, int src1RowStride, int src1PixelStride, 
							int src1Channels, 
							byte[] src1AlphaValue, int src1AlphaOffset, int src1AlphaRowStride, int src1AlphaPixelStride,
							byte[] src2Pixels, int src2Offset, int src2RowStride, int src2PixelStride, 
							int src2Channels, 
							byte[] src2AlphaValue, int src2AlphaOffset, int src2AlphaRowStride, int src2AlphaPixelStride,
                            byte[] maskValue, int maskOffset, int maskRowStride, int maskPixelStride,
							byte[] destPixels, int destOffset, int destRowStride, int destPixelStride,
							bool destHasAlpha, bool destHasColor, int destChannels,
							int width, int height, double opacity) {
			// Color buffer setting
			// Alpha buffer setting

			for (int y = 0; y < height; y ++) {
				for (int x = 0; x < width; x ++) {
					int src1Alpha = (int)(src1AlphaValue[src1AlphaOffset] * opacity);
					int src2Alpha = (int)(src2AlphaValue[src2AlphaOffset]);
					
					if (destHasAlpha) {
						int alpha = (int)src2Alpha + src1Alpha - ((int)src1Alpha * src2Alpha) / 255;
						if (alpha <= 0) {
							for (int c = 0; c < destChannels; c ++)
								destPixels[destOffset + c] = 0;
						} else {
							for (int c = 0; c < destChannels - 1; c ++)
								destPixels[destOffset + c] = (byte)Math.Min(255, ((int)src2Pixels[src2Offset+c] + src1Alpha * (src1Pixels[src1Offset + c] - src2Pixels[src2Offset + c]) / alpha ) );
							destPixels[destOffset + destChannels - 1] = (byte)(Math.Min(255, alpha));
						}
					} else {
						for (int c = 0; c < destChannels; c ++)
							destPixels[c] = (byte)Math.Min(255, ((int)src2Pixels[c] * src2Alpha * 255 + src1Alpha * (src1Pixels[c] * 255 - src2Pixels[c] * src2Alpha)) / (255 * 255) );
					}

					destOffset += destPixelStride;
					src1Offset += src1PixelStride;
					src2Offset += src2PixelStride;
					src1AlphaOffset += src1AlphaPixelStride;
					src2AlphaOffset += src2AlphaPixelStride;
				}
				destOffset += destRowStride - (width * destPixelStride);
				src1Offset += src1RowStride - (width * src1PixelStride);
				src2Offset += src2RowStride - (width * src2PixelStride);
				src1AlphaOffset += src1AlphaRowStride - (width * src1AlphaPixelStride);
				src2AlphaOffset += src2AlphaRowStride - (width * src2AlphaPixelStride);
			}
		}
#endif
		public void Apply(ISurfaceIterator surfIter, ISurfaceIterator auxIter, ISurfaceIterator maskIter, ISurfaceIterator destIter, 
		                    int w, int h, double opacity) {
			try {
				try {
					D.StopWatches.BlendOperation_Apply.Start();
				} catch (Exception e) {
					Console.WriteLine(e);
				}
				if (surfIter == null)
					return;
			byte[] surfColor = surfIter.Raster.Buffer;
			byte[] auxColor = auxIter.Raster.Buffer;
			byte[] auxAlpha = auxColor;
			byte[] surfAlpha = surfColor;

			bool surfHasAlpha = surfIter.Raster.HasAlpha;
			bool auxHasAlpha  = auxIter.Raster.HasAlpha;
			bool destHasAlpha = destIter.Raster.HasAlpha;
			
			bool surfHasColor = surfIter.Raster.HasColorChannels;
			bool auxHasColor = auxIter.Raster.HasColorChannels;
			bool destHasColor = destIter.Raster.HasColorChannels;

			int surfChannels = surfIter.Raster.NumChannels;
			int destChannels = destIter.Raster.NumChannels;
			int auxChannels = auxIter.Raster.NumChannels;
			
			int auxOffset = auxIter.OffsetInBuffer;
			int surfOffset = surfIter.OffsetInBuffer;
			
			int auxAlphaOffset = auxOffset + auxChannels - 1;
			int surfAlphaOffset = surfOffset + surfChannels - 1;

			int auxRowStride = auxIter.Raster.RowStride;
			int surfRowStride = surfIter.Raster.RowStride;
			
			int auxPixelStride = auxIter.Raster.PixelStride;
			int surfPixelStride = surfIter.Raster.PixelStride;
			
			int auxAlphaRowStride = auxIter.Raster.RowStride;
			int surfAlphaRowStride = surfIter.Raster.RowStride;
			
			int auxAlphaPixelStride = auxIter.Raster.PixelStride;
			int surfAlphaPixelStride = surfIter.Raster.PixelStride;
			
			byte[] maskValue = null;
			int maskOffset = 0;
			int maskRowStride = 0;
			int maskPixelStride = 0;


			// Initial Color Value Setting
			if (!surfHasColor) {
				surfColor = surfIter.Raster.PrimaryColor;
				surfOffset = 0;
				surfRowStride = 0;
				surfPixelStride = 0;
			}
			
			if (!auxHasColor) {
				auxColor = auxIter.Raster.PrimaryColor;
				auxOffset = 0;
				auxRowStride = 0;
				auxPixelStride = 0;
			}

			// Initial Alpha Value Setting
			if (!surfHasAlpha) {
				surfAlpha = surfIter.Raster.PrimaryAlphaValue;
				surfAlphaOffset = 0;
				surfAlphaRowStride = 0;
				surfAlphaPixelStride = 0;
			}

			if (!auxHasAlpha) {
				auxAlpha = auxIter.Raster.PrimaryAlphaValue;
				// Short-cut path
				if (auxAlpha[0] * opacity == 0) {
					if (surfIter.Raster == destIter.Raster)
						return;
					else
						; // TBD. Should be memory copy.
				}
				auxAlphaOffset = 0;
				auxAlphaRowStride = 0;
				auxAlphaPixelStride = 0;
			}

			// Mask setting.
			if (maskIter != null) {
				if (maskIter.Raster.HasAlpha) {
					maskValue = maskIter.Raster.Buffer;
					maskOffset = maskIter.OffsetInBuffer + maskIter.Raster.NumChannels - 1;
					maskRowStride = maskIter.Raster.RowStride;
					maskPixelStride = maskIter.Raster.PixelStride;
				} else {
					maskValue = maskIter.Raster.PrimaryAlphaValue;
				}
			}
			
			
#if USE_EXTERNAL_OPS
			CBlend(
#elif USE_UNSAFE_OPS
			UnsafeBlend(
#else
			Blend(
#endif
					auxColor, auxOffset, auxRowStride, auxPixelStride,
					auxChannels,
					auxAlpha, auxAlphaOffset, auxAlphaRowStride, auxAlphaPixelStride,
					surfColor, surfOffset, surfRowStride, surfPixelStride,
					surfChannels,
					surfAlpha, surfAlphaOffset, surfAlphaRowStride, surfAlphaPixelStride,
			        maskValue, maskOffset, maskRowStride, maskPixelStride,
					destIter.Raster.Buffer, destIter.OffsetInBuffer, destIter.Raster.RowStride, destIter.Raster.PixelStride,
					destHasAlpha, destHasColor, destChannels,
					w, h, opacity);
			} finally {
				D.StopWatches.BlendOperation_Apply.Stop();
			}
				
		}
	}
}