﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nft.framework.drawing;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using System.Diagnostics;
using System.Threading;
using System.Drawing;
#region arias
using SysColor = System.Drawing.Color;
using XnaColor = Microsoft.Xna.Framework.Color;
using SysRect = System.Drawing.Rectangle;
using XnaRect = Microsoft.Xna.Framework.Rectangle;
using Point = System.Drawing.Point;
using DefaultEffect = StockEffects.AlphaTestEffect;
#endregion

namespace nft.xna
{
    public class TerrainTemplate : ITerrainPieceTemplate
    {
        protected VertexBuffer vertexBuffer;
        protected readonly Vector3 brightness;

        public TerrainTemplate(float brightness) {
            this.brightness = new Vector3(brightness, brightness, brightness);
        }

        public XnaColor Brightness {
            get { return new XnaColor(brightness); }
        }

        public virtual void Draw(GraphicsDevice gd, DefaultEffect effect){
            effect.DiffuseColor = brightness;
            gd.SetVertexBuffer(vertexBuffer);
            foreach (var pass in effect.CurrentTechnique.Passes) {
                pass.Apply();
                gd.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 1);
            }
        }

        #region IDisposable 定型
        public virtual void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(Boolean disposing)
        {
            if (disposing && vertexBuffer!=null)
            {
                vertexBuffer.Dispose();
                vertexBuffer = null;
            }
        }

        public bool IsDisposed
        { get { return vertexBuffer == null; } }

        ~TerrainTemplate()
        {
            Dispose(false);
        }
        #endregion
    }

    public class TexturedTerrainTemplate : TerrainTemplate
    {
        protected XnaTexture texture;

        public TexturedTerrainTemplate(XnaTexture texture, Vector3[] vertices, Vector2[] texture_pos, float brightness) : base(brightness){
            this.texture = texture;
            this.texture.ImageSrc.AddRef();
            int n = vertices.Length;
            VertexPositionTexture[] v = new VertexPositionTexture[n];
            for (int i = 0; i < n; i++) {
                v[n] = new VertexPositionTexture(vertices[i], texture_pos[i]);
            }
            this.vertexBuffer = new VertexBuffer(XnaGraphicManager.GraphicsDevice, typeof(VertexPositionTexture), n, BufferUsage.None);
            this.vertexBuffer.SetData<VertexPositionTexture>(v);
        }

        public Texture2D Texture {
            get {
                return texture.ImageSrc.Texture2D;
            }
        }

        protected override void Dispose(bool disposing) {
            base.Dispose(disposing);
            if (disposing && texture != null) {
                texture.ImageSrc.ReleaseRef();
                texture = null;
            }
        }

        public override void Draw(GraphicsDevice gd, DefaultEffect effect) {
            effect.Texture = this.Texture;
            //effect.SetTextureRegion(this.region);
            base.Draw(gd, effect);
            //((DefaultEffect)effect).SetTextureRegion(new Microsoft.Xna.Framework.Rectangle());
        }
    }

    public class PlainTerrainTemplate : TerrainTemplate
    {
        public PlainTerrainTemplate(XnaColor baseColor, Vector3[] vertices, Vector2[] texture_pos, float brightness)
            : base(brightness) {
            int n = vertices.Length;
            VertexPositionColor[] v = new VertexPositionColor[n];
            for (int i = 0; i < n; i++) {
                v[n] = new VertexPositionColor(vertices[i], baseColor);
            }
            this.vertexBuffer = new VertexBuffer(XnaGraphicManager.GraphicsDevice, typeof(VertexPositionColor), n, BufferUsage.None);
            this.vertexBuffer.SetData<VertexPositionColor>(v);
        }

        public override void Draw(GraphicsDevice gd, DefaultEffect effect) {
            ((DefaultEffect)effect).VertexColorEnabled = true;
            base.Draw(gd, effect);
            ((DefaultEffect)effect).VertexColorEnabled = false;
        }

    }

    public class TerrainPiece : ITerrainPiece, IDrawable
    {
        protected object _tag;
        protected EffectFilter filter;
        private Vector3 pos;
        protected TerrainTemplate template;

        public TerrainPiece(TerrainTemplate templ)
        {
            this.template = templ;
        }

        public Vector3 Position3D
        {
            get { return pos; }
            set {
                if (pos.Equals(value)) return;
                pos = value;
            }
        }

        #region I3DObject implementation
        public object Tag {
            get {
                return _tag;
            }
            set {
                _tag = value;
            }
        }
        public IEffectFilter Effect {
            get {
                return filter;
            }
            set {
                filter = (EffectFilter)value;
            }
        }
        #endregion

        public virtual void RenderSelf(XnaSurface surface, DefaultEffect effect) {
            GraphicsDevice gd = XnaGraphicManager.GraphicsDevice;
            // 平行移動を適用
            Matrix wback = effect.World;
            effect.World = Matrix.CreateTranslation(Position3D);
            template.Draw(gd, effect);
        }

        static protected Vector3 ConvertIsometric(float l, float r, float h)
        {
            return new Vector3(r, h, l);
        }

        static protected XnaRect CalcClipRect(Vector2 offset, Vector3 size)
        {
            float width = size.X + size.Z;
            float height = size.Y + width / 2;
            XnaRect rct = new XnaRect((int)offset.X, (int)offset.Y, (int)width, (int)height);
            return rct;
        }

        #region ITerrainPiece implementation
        public PointF3D Location
        {
            get
            {
                return XnaUtil.ToPointF(pos);
            }
            set
            {
                pos = XnaUtil.ToVector3(value);
            }
        }
        #endregion

        #region IDisposable 定型
        public virtual void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(Boolean disposing)
        {
            template = null;
            filter = null;
        }

        public bool IsDisposed
        { get { return template == null; } }

        ~TerrainPiece()
        {
            Dispose(false);
        }
        #endregion
    }

}
