/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.solr;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.util.FessProperties;

import org.apache.solr.client.solrj.SolrServer;
import org.seasar.framework.util.StringUtil;

public class SolrServerManager implements Serializable {

    private static final long serialVersionUID = 1L;

    protected Map<String, SolrServerGroup> solrServerGroupMap = new LinkedHashMap<String, SolrServerGroup>();

    protected String selectSolrServerGroupName;

    protected String updateSolrServerGroupName;

    @Resource
    protected FessProperties solrServerProperties;

    public void addSolrServerGroup(SolrServerGroup solrServerGroup) {
        String name = solrServerGroup.getGroupName();
        if (StringUtil.isBlank(name)) {
            throw new FessSolrException("A solr server group name is a blank.");
        }
        solrServerGroupMap.put(name, solrServerGroup);
    }

    public SolrServerGroup getSelectSolrServerGroup() {
        checkSolrServerGroupName();

        return getSolrServerGroup(selectSolrServerGroupName);
    }

    public SolrServerGroup getUpdateSolrServerGroup() {
        checkSolrServerGroupName();

        return getSolrServerGroup(updateSolrServerGroupName);
    }

    public SolrServerGroup getSolrServerGroup(String solrServerGroupName) {

        checkServerGroupStatus(solrServerGroupName);

        SolrServerGroup solrServerGroup = solrServerGroupMap
                .get(solrServerGroupName);
        if (solrServerGroup == null) {
            throw new FessSolrException("The solr group(" + solrServerGroupName
                    + ") does not exist.");
        }

        return solrServerGroup;
    }

    protected void checkServerGroupStatus(String solrServerGroupName) {
        String statusKey = solrServerGroupName + ".status";
        String status = solrServerProperties.getProperty(statusKey,
                Constants.ACTIVE);
        if (!status.equals(Constants.ACTIVE)) {
            throw new FessSolrException("The current solr server group ("
                    + solrServerGroupName + ") is not available.");
        }
    }

    protected void checkSolrServerGroupName() {
        if (selectSolrServerGroupName == null) {
            selectSolrServerGroupName = solrServerProperties
                    .getProperty(Constants.CURRENT_SERVER_FOR_SELECT);
            updateSolrServerGroupName = solrServerProperties
                    .getProperty(Constants.CURRENT_SERVER_FOR_UPDATE);

            // check server name
            if (solrServerGroupMap.get(selectSolrServerGroupName) == null) {
                // clear
                selectSolrServerGroupName = null;
                updateSolrServerGroupName = null;
            }

            if (selectSolrServerGroupName == null) {
                synchronized (solrServerProperties) {
                    if (selectSolrServerGroupName != null) {
                        return;
                    }

                    int numOfGroup = solrServerGroupMap.size();
                    if (numOfGroup <= 0) {
                        throw new FessSolrException("No solr group.");
                    }

                    Set<String> nameSet = solrServerGroupMap.keySet();
                    if (nameSet.size() == 1) {
                        Iterator<String> itr = nameSet.iterator();
                        selectSolrServerGroupName = itr.next();
                        updateSolrServerGroupName = selectSolrServerGroupName;
                    } else {
                        Iterator<String> itr = nameSet.iterator();
                        selectSolrServerGroupName = itr.next();
                        updateSolrServerGroupName = itr.next();
                    }

                    solrServerProperties.setProperty(
                            Constants.CURRENT_SERVER_FOR_SELECT,
                            selectSolrServerGroupName);
                    solrServerProperties.setProperty(
                            Constants.CURRENT_SERVER_FOR_UPDATE,
                            updateSolrServerGroupName);
                    solrServerProperties.store();
                }
            }
        }
    }

    public void applyNewServerGroup() {
        synchronized (solrServerProperties) {
            selectSolrServerGroupName = updateSolrServerGroupName;
            Set<String> nameSet = solrServerGroupMap.keySet();
            String[] names = nameSet.toArray(new String[nameSet.size()]);
            int num;
            for (num = 0; num < names.length; num++) {
                if (names[num].equals(selectSolrServerGroupName)) {
                    break;
                }
            }
            // count up
            num++;
            if (num >= names.length) {
                num = 0;
            }
            updateSolrServerGroupName = names[num];

            solrServerProperties.setProperty(
                    Constants.CURRENT_SERVER_FOR_SELECT,
                    selectSolrServerGroupName);
            solrServerProperties.setProperty(
                    Constants.CURRENT_SERVER_FOR_UPDATE,
                    updateSolrServerGroupName);
            solrServerProperties.store();
        }
    }

    public String[] getSolrServerGroupNames() {
        List<String> serverGroupList = new ArrayList<String>();
        for (Map.Entry<String, SolrServerGroup> entry : solrServerGroupMap
                .entrySet()) {
            serverGroupList.add(entry.getKey());
        }
        return serverGroupList.toArray(new String[serverGroupList.size()]);
    }

    public String[] getSolrServerNames() {
        List<String> serverNameList = new ArrayList<String>();
        for (Map.Entry<String, SolrServerGroup> entry1 : solrServerGroupMap
                .entrySet()) {
            for (Map.Entry<String, SolrServer> entry2 : entry1.getValue().solrServerMap
                    .entrySet()) {
                serverNameList.add(entry1.getKey() + "." + entry2.getKey());
            }
        }
        return serverNameList.toArray(new String[serverNameList.size()]);
    }

}
