/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Collection;
import java.util.Enumeration;
import java.util.InvalidPropertiesFormatException;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import jp.sf.fess.FessSystemException;

import org.apache.commons.io.IOUtils;
import org.seasar.framework.util.StringUtil;

public class FessProperties extends Properties {

    private static final long serialVersionUID = 1L;

    public long checkInterval = 5000L;

    protected volatile long lastChecked = 0L;

    protected volatile long lastModified = 0L;

    protected File propertiesFile;

    protected volatile Properties properties;

    public FessProperties(String path) {
        // check path
        if (StringUtil.isEmpty(path)) {
            throw new FessSystemException("The path is null.");
        }

        propertiesFile = new File(path);
        if (!propertiesFile.exists()) {
            File parentDir = propertiesFile.getParentFile();
            if (!parentDir.exists()) {
                if (!parentDir.mkdir()) {
                    throw new FessSystemException(
                            "Could not create a parent directory of " + path);
                }
            } else if (!parentDir.isDirectory()) {
                throw new FessSystemException("A parent directory of " + path
                        + " is not a directory.");
            }
            properties = new Properties();
            store();
        } else if (!propertiesFile.isFile()) {
            throw new FessSystemException(path + " is not a file.");
        }
        load();
    }

    public boolean isUpdated() {
        long now = System.currentTimeMillis();
        if (now - lastChecked < checkInterval) {
            lastChecked = now;
            return false;
        }
        lastChecked = now;

        long timestamp = propertiesFile.lastModified();
        if (timestamp <= lastModified) {
            return false;
        }

        return true;
    }

    public synchronized void load() {
        Properties prop = new Properties();
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(propertiesFile);
            lastModified = propertiesFile.lastModified();
            prop.load(fis);
        } catch (IOException e) {
            throw new FessSystemException("Could not load "
                    + propertiesFile.getAbsolutePath(), e);
        } finally {
            IOUtils.closeQuietly(fis);
        }
        properties = prop;
    }

    public synchronized void store() {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(propertiesFile);
            properties.store(fos, propertiesFile.getName());
        } catch (IOException e) {
            throw new FessSystemException("Could not store "
                    + propertiesFile.getAbsolutePath(), e);
        } finally {
            IOUtils.closeQuietly(fos);
        }
        lastModified = propertiesFile.lastModified();
    }

    protected Properties getProperties() {
        if (isUpdated()) {
            load();
        }
        return properties;
    }

    public void clear() {
        getProperties().clear();
    }

    public Object clone() {
        // TODO
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public boolean contains(Object value) {
        return getProperties().contains(value);
    }

    public boolean containsKey(Object key) {
        return getProperties().containsKey(key);
    }

    public boolean containsValue(Object value) {
        return getProperties().containsValue(value);
    }

    public Enumeration<Object> elements() {
        return getProperties().elements();
    }

    public Set<java.util.Map.Entry<Object, Object>> entrySet() {
        return getProperties().entrySet();
    }

    public boolean equals(Object o) {
        return getProperties().equals(o);
    }

    public Object get(Object key) {
        return getProperties().get(key);
    }

    public String getProperty(String key, String defaultValue) {
        return getProperties().getProperty(key, defaultValue);
    }

    public String getProperty(String key) {
        return getProperties().getProperty(key);
    }

    public int hashCode() {
        return getProperties().hashCode();
    }

    public boolean isEmpty() {
        return getProperties().isEmpty();
    }

    public Enumeration<Object> keys() {
        return getProperties().keys();
    }

    public Set<Object> keySet() {
        return getProperties().keySet();
    }

    public void list(PrintStream out) {
        getProperties().list(out);
    }

    public void list(PrintWriter out) {
        getProperties().list(out);
    }

    public void load(InputStream inStream) throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public void load(Reader reader) throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public void loadFromXML(InputStream in) throws IOException,
            InvalidPropertiesFormatException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public Enumeration<?> propertyNames() {
        return getProperties().propertyNames();
    }

    public Object put(Object key, Object value) {
        return getProperties().put(key, value);
    }

    public void putAll(Map<? extends Object, ? extends Object> t) {
        getProperties().putAll(t);
    }

    public Object remove(Object key) {
        return getProperties().remove(key);
    }

    public void save(OutputStream out, String comments) {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public Object setProperty(String key, String value) {
        return getProperties().setProperty(key, value);
    }

    public int size() {
        return getProperties().size();
    }

    public void store(OutputStream out, String comments) throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public void store(Writer writer, String comments) throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public void storeToXML(OutputStream os, String comment, String encoding)
            throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public void storeToXML(OutputStream os, String comment) throws IOException {
        throw new UnsupportedOperationException("Unsupported operation.");
    }

    public Set<String> stringPropertyNames() {
        return getProperties().stringPropertyNames();
    }

    public String toString() {
        return getProperties().toString();
    }

    public Collection<Object> values() {
        return getProperties().values();
    }
}
