/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.solr;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.entity.Document;
import jp.sf.fess.util.FessProperties;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.embedded.JettySolrRunner;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.seasar.extension.unit.S2TestCase;

public class SolrServerGroupTest extends S2TestCase {
    private static final String CONTEXT_PATH = "/solr1";

    private static final int PORT = 8181;

    public SolrServerGroup solrGroup1;

    public SolrServerGroup solrGroup2;

    public FessProperties solrServerProperties;

    private JettySolrRunner jettySolrRunner;

    @Override
    protected String getRootDicon() throws Throwable {
        return "app.dicon";
    }

    @Override
    public void setUp() throws Exception {
        super.setUp();

        jettySolrRunner = new JettySolrRunner(CONTEXT_PATH, PORT);
        jettySolrRunner.start();
    }

    @Override
    protected void setUpAfterBindFields() throws Throwable {
        super.setUpAfterBindFields();

        solrServerProperties.clear();
    }

    @Override
    public void tearDown() throws Exception {
        super.tearDown();
        jettySolrRunner.stop();
    }

    public void test_checkStatus() {
        assertNotNull(solrGroup1);
        assertNotNull(solrGroup2);

        assertEquals("solrGroup1", solrGroup1.getGroupName());
        assertEquals("solrGroup2", solrGroup2.getGroupName());

        assertTrue(solrGroup1.isActive(5));
        assertTrue(solrGroup2.isActive(5));
    }

    private void cleanup(SolrServerGroup solrServerGroup) {
        // delete all docs
        solrServerGroup.deleteByQuery("*:*");
        solrServerGroup.commit();
    }

    public void test_add_collection() {
        // test
        List<SolrInputDocument> docs = new ArrayList<SolrInputDocument>();
        SolrInputDocument doc1 = new SolrInputDocument();
        doc1.addField("id", "id1", 1.0f);
        doc1.addField("url", "url1", 1.0f);
        doc1.addField("title", "doc1", 1.0f);
        doc1.addField("tstamp", 10);
        docs.add(doc1);

        SolrInputDocument doc2 = new SolrInputDocument();
        doc2.addField("id", "id2", 1.0f);
        doc2.addField("url", "url2", 1.0f);
        doc2.addField("title", "doc2", 1.0f);
        doc2.addField("tstamp", 20);
        docs.add(doc2);

        solrGroup1.add(docs);

        solrGroup1.commit();

        // assert
        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.addSortField("tstamp", SolrQuery.ORDER.desc);

        QueryResponse queryResponse = solrGroup1.query(query);
        SolrDocumentList resultDocs = queryResponse.getResults();
        assertEquals(2, resultDocs.size());
        SolrDocument resultDoc2 = resultDocs.get(0);
        assertEquals("id2", resultDoc2.getFieldValue("id"));
        assertEquals("url2", resultDoc2.getFieldValue("url"));
        assertEquals("doc2", resultDoc2.getFieldValue("title"));
        assertEquals(Long.valueOf(20), resultDoc2.getFieldValue("tstamp"));
        SolrDocument resultDoc1 = resultDocs.get(1);
        assertEquals("id1", resultDoc1.getFieldValue("id"));
        assertEquals("url1", resultDoc1.getFieldValue("url"));
        assertEquals("doc1", resultDoc1.getFieldValue("title"));
        assertEquals(Long.valueOf(10), resultDoc1.getFieldValue("tstamp"));

        // cleanup
        cleanup(solrGroup1);

        // check (deleteByQuery)
        queryResponse = solrGroup1.query(query);
        resultDocs = queryResponse.getResults();
        assertEquals(0, resultDocs.size());
    }

    public void test_add_doc() {
        // test
        SolrInputDocument doc1 = new SolrInputDocument();
        doc1.addField("id", "id1", 1.0f);
        doc1.addField("url", "url1", 1.0f);
        doc1.addField("title", "doc1", 1.0f);
        doc1.addField("tstamp", 10);

        solrGroup1.add(doc1);

        solrGroup1.commit();

        // assert
        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.addSortField("tstamp", SolrQuery.ORDER.desc);

        QueryResponse queryResponse = solrGroup1.query(query);
        SolrDocumentList resultDocs = queryResponse.getResults();
        SolrDocument resultDoc1 = resultDocs.get(0);
        assertEquals("id1", resultDoc1.getFieldValue("id"));
        assertEquals("url1", resultDoc1.getFieldValue("url"));
        assertEquals("doc1", resultDoc1.getFieldValue("title"));
        assertEquals(Long.valueOf(10), resultDoc1.getFieldValue("tstamp"));

        // cleanup
        cleanup(solrGroup1);
    }

    public void test_add_bean() {
        // test
        Document bean = new Document();
        bean.id = "id1";
        bean.url = "url1";
        bean.title = "doc1";
        bean.tstamp = 10L;

        solrGroup1.addBean(bean);

        solrGroup1.commit();

        // assert
        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.addSortField("tstamp", SolrQuery.ORDER.desc);

        QueryResponse queryResponse = solrGroup1.query(query);
        List<Document> resultBeans = queryResponse.getBeans(Document.class);
        assertEquals(1, resultBeans.size());
        Document resultBean1 = resultBeans.get(0);
        assertEquals("id1", resultBean1.id);
        assertEquals("url1", resultBean1.url);
        assertEquals("doc1", resultBean1.title);
        assertEquals(Long.valueOf(10), resultBean1.tstamp);

        // cleanup
        cleanup(solrGroup1);
    }

    public void test_add_beans() {
        // test
        List<Document> beans = new ArrayList<Document>();
        Document bean1 = new Document();
        bean1.id = "id1";
        bean1.url = "url1";
        bean1.title = "doc1";
        bean1.tstamp = 10L;
        beans.add(bean1);
        Document bean2 = new Document();
        bean2.id = "id2";
        bean2.url = "url2";
        bean2.title = "doc2";
        bean2.tstamp = 20L;
        beans.add(bean2);

        solrGroup1.addBean(beans);

        solrGroup1.commit();

        // assert
        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.addSortField("tstamp", SolrQuery.ORDER.desc);

        QueryResponse queryResponse = solrGroup1.query(query);
        List<Document> resultBeans = queryResponse.getBeans(Document.class);
        assertEquals(2, resultBeans.size());
        Document resultBean1 = resultBeans.get(1);
        assertEquals("id1", resultBean1.id);
        assertEquals("url1", resultBean1.url);
        assertEquals("doc1", resultBean1.title);
        assertEquals(Long.valueOf(10), resultBean1.tstamp);
        Document resultBean2 = resultBeans.get(0);
        assertEquals("id2", resultBean2.id);
        assertEquals("url2", resultBean2.url);
        assertEquals("doc2", resultBean2.title);
        assertEquals(Long.valueOf(20), resultBean2.tstamp);

        // cleanup
        cleanup(solrGroup1);
    }

    public void test_add_deleteById() {
        // test
        List<SolrInputDocument> docs = new ArrayList<SolrInputDocument>();
        SolrInputDocument doc1 = new SolrInputDocument();
        doc1.addField("id", "id1", 1.0f);
        doc1.addField("url", "url1", 1.0f);
        doc1.addField("title", "doc1", 1.0f);
        doc1.addField("tstamp", 10);
        docs.add(doc1);

        SolrInputDocument doc2 = new SolrInputDocument();
        doc2.addField("id", "id2", 1.0f);
        doc2.addField("url", "url2", 1.0f);
        doc2.addField("title", "doc2", 1.0f);
        doc2.addField("tstamp", 20);
        docs.add(doc2);

        solrGroup1.add(docs);

        solrGroup1.commit();

        // assert
        SolrQuery query1 = new SolrQuery();
        query1.setQuery("*:*");
        query1.addSortField("tstamp", SolrQuery.ORDER.desc);

        QueryResponse queryResponse = solrGroup1.query(query1);
        SolrDocumentList resultDocs = queryResponse.getResults();
        assertEquals(2, resultDocs.size());
        SolrDocument resultDoc2 = resultDocs.get(0);
        assertEquals("id2", resultDoc2.getFieldValue("id"));
        assertEquals("url2", resultDoc2.getFieldValue("url"));
        assertEquals("doc2", resultDoc2.getFieldValue("title"));
        assertEquals(Long.valueOf(20), resultDoc2.getFieldValue("tstamp"));
        SolrDocument resultDoc1 = resultDocs.get(1);
        assertEquals("id1", resultDoc1.getFieldValue("id"));
        assertEquals("url1", resultDoc1.getFieldValue("url"));
        assertEquals("doc1", resultDoc1.getFieldValue("title"));
        assertEquals(Long.valueOf(10), resultDoc1.getFieldValue("tstamp"));

        //test
        solrGroup1.deleteById("id2");
        solrGroup1.commit();

        // check (deleteByQuery)
        SolrQuery query2 = new SolrQuery();
        query2.setQuery("*:*");
        query2.addSortField("tstamp", SolrQuery.ORDER.desc);
        queryResponse = solrGroup1.query(query2);
        resultDocs = queryResponse.getResults();
        assertEquals(1, resultDocs.size());
        resultDoc1 = resultDocs.get(0);
        assertEquals("id1", resultDoc1.getFieldValue("id"));
        assertEquals("url1", resultDoc1.getFieldValue("url"));
        assertEquals("doc1", resultDoc1.getFieldValue("title"));
        assertEquals(Long.valueOf(10), resultDoc1.getFieldValue("tstamp"));

        // cleanup
        cleanup(solrGroup1);
    }

    public void test_ping() {
        // test
        Collection<SolrPingResponse> pingList = solrGroup1.ping();

        // check
        assertEquals(1, pingList.size());
        for (SolrPingResponse solrPingResponse : pingList) {
            assertEquals(0, solrPingResponse.getStatus());
        }

    }

    public void test_facet() {
        // test
        List<SolrInputDocument> docs = new ArrayList<SolrInputDocument>();
        SolrInputDocument doc1 = new SolrInputDocument();
        doc1.addField("id", "id1", 1.0f);
        doc1.addField("url", "url1", 1.0f);
        doc1.addField("segment", "0001", 1.0f);
        doc1.addField("title", "doc1", 1.0f);
        doc1.addField("tstamp", 10);
        docs.add(doc1);

        SolrInputDocument doc2 = new SolrInputDocument();
        doc2.addField("id", "id2", 1.0f);
        doc2.addField("url", "url2", 1.0f);
        doc2.addField("segment", "0001", 1.0f);
        doc2.addField("title", "doc2", 1.0f);
        doc2.addField("tstamp", 20);
        docs.add(doc2);

        SolrInputDocument doc3 = new SolrInputDocument();
        doc3.addField("id", "id3", 1.0f);
        doc3.addField("url", "url3", 1.0f);
        doc3.addField("segment", "0002", 1.0f);
        doc3.addField("title", "doc3", 1.0f);
        doc3.addField("tstamp", 30);
        docs.add(doc3);

        SolrInputDocument doc4 = new SolrInputDocument();
        doc4.addField("id", "id4", 1.0f);
        doc4.addField("url", "url4", 1.0f);
        doc4.addField("segment", "0002", 1.0f);
        doc4.addField("title", "doc4", 1.0f);
        doc4.addField("tstamp", 40);
        docs.add(doc4);

        solrGroup1.add(docs);

        solrGroup1.commit();

        // assert
        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.setFacet(true);
        query.addFacetField("segment");

        QueryResponse queryResponse = solrGroup1.query(query);
        List<FacetField> facets = queryResponse.getFacetFields();
        for (FacetField facet : facets) {
            List<FacetField.Count> facetEntries = facet.getValues();
            for (FacetField.Count fcount : facetEntries) {
                if ("0001".equals(fcount.getName())) {
                    assertEquals(2, fcount.getCount());
                } else if ("0002".equals(fcount.getName())) {
                    assertEquals(2, fcount.getCount());
                } else {
                    throw new FessSystemException("Wrong result: "
                            + fcount.getName());
                }
            }
        }

        query.clear();
        query.setQuery("tstamp:[20 TO 40]");
        query.setFacet(true);
        query.addFacetField("segment");

        queryResponse = solrGroup1.query(query);
        facets = queryResponse.getFacetFields();
        for (FacetField facet : facets) {
            List<FacetField.Count> facetEntries = facet.getValues();
            for (FacetField.Count fcount : facetEntries) {
                if ("0001".equals(fcount.getName())) {
                    assertEquals(1, fcount.getCount());
                } else if ("0002".equals(fcount.getName())) {
                    assertEquals(2, fcount.getCount());
                } else {
                    throw new FessSystemException("Wrong result: "
                            + fcount.getName());
                }
            }
        }

        // cleanup
        cleanup(solrGroup1);

    }
}
