/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.helper.RoleQueryHelper;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.util.RequestUtil;

public class QueryHelperImpl implements QueryHelper, Serializable {

    private static final long serialVersionUID = 1L;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected RoleQueryHelper roleQueryHelper;

    public String[] responseFields = new String[] { "id", "score", "boost",
            "contentLength", "host", "site", "lastModified", "mimetype",
            "tstamp", "title", "digest", "url" };

    public String[] highlightingFields = new String[] { "digest", "cache" };

    public String[] supportedFields = new String[] { "url", "host", "site",
            "title", "content", "contentLength", "lastModified", "mimetype",
            "label", "segment" };

    public String sortPrefix = "sort:";

    public String[] supportedSortFields = new String[] { "tstamp",
            "contentLength", "lastModified" };

    public int highlightSnippetSize = 5;

    protected String shards;

    public boolean useBigram = true;

    public String additionalQuery;

    /* (non-Javadoc)
     * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
     */
    public SearchQuery build(String query) {
        return build(query, true);
    }

    /* (non-Javadoc)
         * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
         */
    public SearchQuery build(String query, boolean envCondition) {
        String q;
        if (envCondition && additionalQuery != null
                && StringUtil.isNotBlank(query)) {
            q = query + " " + additionalQuery;
        } else {
            q = query;
        }

        SearchQuery searchQuery = buildQuery(q);
        if (!searchQuery.queryExists()) {
            return searchQuery.query("");
        }

        if ((browserTypeHelper == null && roleQueryHelper == null)
                || !envCondition) {
            return searchQuery;
        }

        StringBuilder queryBuf = new StringBuilder(255);
        queryBuf.append(searchQuery.getQuery());

        if (browserTypeHelper != null) {
            queryBuf.append(" AND type:");
            queryBuf.append(escapeValue(browserTypeHelper.getBrowserType()));
        }

        if (roleQueryHelper != null) {
            List<String> roleList = roleQueryHelper.build();
            if (!roleList.isEmpty()) {
                queryBuf.append(" AND ");
                if (roleList.size() > 1) {
                    queryBuf.append('(');
                }
                boolean isFirst = true;
                for (String role : roleList) {
                    if (isFirst) {
                        isFirst = false;
                    } else {
                        queryBuf.append(" OR ");

                    }
                    queryBuf.append("role:");
                    queryBuf.append(escapeValue(role));
                }
                if (roleList.size() > 1) {
                    queryBuf.append(')');
                }
            }
        }

        return searchQuery.query(queryBuf.toString());
    }

    protected SearchQuery buildQuery(String query) {
        String[] values = splitQuery(query);
        if (values.length == 0) {
            return new SearchQuery().query("");
        }

        List<String> highLightQueryList = new ArrayList<String>();

        SearchQuery searchQuery = new SearchQuery();

        StringBuilder queryBuf = new StringBuilder(255);
        StringBuilder titleBuf = new StringBuilder(255);
        StringBuilder contentBuf = new StringBuilder(255);
        List<String> notOperatorList = new ArrayList<String>();
        String operator = " AND ";
        String queryOperator = " AND ";
        boolean notOperatorFlag = false;
        int queryOperandCount = 0;
        int titleOperandCount = 0;
        int contentOperandCount = 0;
        for (String value : values) {
            boolean nonPrefix = false;
            // check prefix
            for (String field : supportedFields) {
                String prefix = field + ":";
                if (value.startsWith(prefix)
                        && value.length() != prefix.length()) {
                    if (queryBuf.length() > 0) {
                        queryBuf.append(operator);
                    } else if (titleBuf.length() > 0 || contentBuf.length() > 0) {
                        queryOperator = operator;
                    }
                    String targetWord = value.substring(prefix.length());
                    if (notOperatorFlag) {
                        StringBuilder buf = new StringBuilder(100);
                        buf.append(prefix);
                        appendQueryValue(buf, targetWord);
                        notOperatorList.add(buf.toString());
                        notOperatorFlag = false;
                    } else {
                        queryBuf.append(prefix);
                        appendQueryValue(queryBuf, targetWord);
                        queryOperandCount++;
                    }
                    nonPrefix = true;
                    operator = " AND ";
                    highLightQueryList.add(targetWord);
                    break;
                }
            }

            // sort
            if (value.startsWith(sortPrefix)
                    && value.length() != sortPrefix.length()) {
                String[] sortFieldPairs = value.substring(sortPrefix.length())
                        .split(",");
                for (String sortFieldPairStr : sortFieldPairs) {
                    String[] sortFieldPair = sortFieldPairStr.split("\\.");
                    if (isSupportedSortField(sortFieldPair[0])) {
                        if (sortFieldPair.length == 1) {
                            searchQuery.addSortField(sortFieldPair[0],
                                    Constants.ASC);
                        } else {
                            searchQuery.addSortField(sortFieldPair[0],
                                    sortFieldPair[1]);
                        }
                    }
                }
                continue;
            }

            if (!nonPrefix) {
                if ("AND".equals(value)) {
                    operator = " AND ";
                } else if ("OR".equals(value)) {
                    operator = " OR ";
                } else if ("NOT".equals(value)) {
                    notOperatorFlag = true;
                } else if (notOperatorFlag) {
                    StringBuilder buf = new StringBuilder(100);
                    buf.append("title:");
                    appendQueryValue(buf, value);
                    notOperatorList.add(buf.toString());

                    buf = new StringBuilder(100);
                    buf.append("content:");
                    appendQueryValue(buf, value);
                    notOperatorList.add(buf.toString());

                    operator = " AND ";
                    notOperatorFlag = false;
                    highLightQueryList.add(value);
                } else {
                    // title
                    if (titleBuf.length() > 0) {
                        titleBuf.append(operator);
                    }
                    titleBuf.append("title:");
                    appendQueryValue(titleBuf, value);
                    titleOperandCount++;

                    // content
                    if (contentBuf.length() > 0) {
                        contentBuf.append(operator);
                    }
                    contentBuf.append("content:");
                    appendQueryValue(contentBuf, value);
                    contentOperandCount++;

                    operator = " AND ";
                    highLightQueryList.add(value);
                }
            }
        }

        if (titleBuf.length() > 0 || contentBuf.length() > 0) {
            boolean append = false;
            if (queryBuf.length() > 0) {
                append = true;
            }

            int operandCount = titleOperandCount + contentOperandCount;
            if (append) {
                queryBuf.append(queryOperator);
                if (operandCount > 1) {
                    queryBuf.append('(');
                }
            }
            if (titleBuf.length() > 0 && contentBuf.length() > 0) {
                if (titleOperandCount > 1) {
                    queryBuf.append('(');
                }
                queryBuf.append(titleBuf.toString());
                if (titleOperandCount > 1) {
                    queryBuf.append(')');
                }
                queryBuf.append(" OR ");
                if (contentOperandCount > 1) {
                    queryBuf.append('(');
                }
                queryBuf.append(contentBuf.toString());
                if (contentOperandCount > 1) {
                    queryBuf.append(')');
                }
            } else if (titleBuf.length() > 0) {
                queryBuf.append(titleBuf.toString());
            } else if (contentBuf.length() > 0) {
                queryBuf.append(contentBuf.toString());
            }
            if (append && operandCount > 1) {
                queryBuf.append(')');
            }
        }

        StringBuilder searchQueryBuf = new StringBuilder(255);
        if (queryBuf.length() > 0) {
            int count = queryOperandCount + titleOperandCount
                    + contentOperandCount;
            if (count > 1) {
                searchQueryBuf.append('(');
            }
            searchQueryBuf.append(queryBuf.toString());
            if (count > 1) {
                searchQueryBuf.append(')');
            }
            operator = " AND ";
        } else {
            operator = Constants.EMPTY_STRING;
        }
        if (!notOperatorList.isEmpty()) {
            for (String notOperator : notOperatorList) {
                searchQueryBuf.append(operator);
                searchQueryBuf.append("NOT ");
                searchQueryBuf.append(notOperator);
                operator = " AND ";
            }
        }

        // set queries to request for HighLight
        RequestUtil
                .getRequest()
                .setAttribute(
                        Constants.HIGHLIGHT_QUERIES,
                        highLightQueryList
                                .toArray(new String[highLightQueryList.size()]));

        return searchQuery.query(searchQueryBuf.toString());
    }

    private void appendQueryValue(StringBuilder buf, String value) {
        // check reserved
        boolean reserved = false;
        for (int i = 0; i < Constants.RESERVED.length; i++) {
            if (Constants.RESERVED[i].equals(value)) {
                reserved = true;
                break;
            }
        }

        if (useBigram && value.length() == 1
                && !StringUtils.isAsciiPrintable(value)) {
            // if using bigram, add ?
            value = value + "?";
        }

        if (reserved) {
            buf.append('\\');
            buf.append(value);
            return;
        }

        String tildeValue = null;
        String caretValue = null;
        int tildePos = value.lastIndexOf('~');
        int caretPos = value.indexOf('^');
        if (tildePos > caretPos) {
            if (tildePos > 0) {
                tildeValue = value.substring(tildePos);
                value = value.substring(0, tildePos);
            }
        } else {
            if (caretPos > 0) {
                caretValue = value.substring(caretPos);
                value = value.substring(0, caretPos);
            }
        }
        if (value.startsWith("[") && value.endsWith("]")) {
            String[] split = value.substring(1, value.length() - 1).split(
                    " TO ");
            if (split.length == 2 && split[0].length() > 0
                    && split[1].length() > 0) {
                buf.append('[');
                buf.append(escapeValue(split[0]));
                buf.append(" TO ");
                buf.append(escapeValue(split[1]));
                buf.append(']');
            } else {
                buf.append(escapeValue(value));
            }
        } else if (value.startsWith("{") && value.endsWith("}")) {
            String[] split = value.substring(1, value.length() - 1).split(
                    " TO ");
            if (split.length == 2 && split[0].length() > 0
                    && split[1].length() > 0) {
                buf.append('{');
                buf.append(escapeValue(split[0]));
                buf.append(" TO ");
                buf.append(escapeValue(split[1]));
                buf.append('}');
            } else {
                buf.append(escapeValue(value));
            }
        } else {
            buf.append(escapeValue(value));
        }

        if (tildeValue != null) {
            buf.append(tildeValue);
        } else if (caretValue != null) {
            buf.append(caretValue);
        }
    }

    private boolean isSupportedSortField(String field) {
        for (String f : supportedSortFields) {
            if (f.equals(field)) {
                return true;
            }
        }
        return false;
    }

    protected String[] splitQuery(String query) {
        List<String> valueList = new ArrayList<String>();
        StringBuilder buf = new StringBuilder();
        boolean quoted = false;
        boolean squareBracket = false;
        boolean curlyBracket = false;
        for (int i = 0; i < query.length(); i++) {
            char c = query.charAt(i);
            switch (c) {
            case '[':
                buf.append(c);
                if (!quoted && !curlyBracket) {
                    squareBracket = true;
                }
                break;
            case ']':
                buf.append(c);
                squareBracket = false;
                break;
            case '{':
                buf.append(c);
                if (!quoted && !squareBracket) {
                    curlyBracket = true;
                }
                break;
            case '}':
                buf.append(c);
                curlyBracket = false;
                break;
            case '"':
                if (!curlyBracket && !squareBracket) {
                    quoted = !quoted;
                } else {
                    buf.append(c);
                }
                break;
            case ' ':
            case '\u3000':
                if (quoted || curlyBracket || squareBracket) {
                    buf.append(c);
                } else {
                    if (buf.length() > 0) {
                        String str = buf.toString();
                        if (str.startsWith("[") || str.startsWith("{")) {
                            int pos = str.indexOf(" TO ");
                            if (pos < 0) {
                                str = str.replace('[', ' ');
                                str = str.replace(']', ' ');
                                str = str.replace('{', ' ');
                                str = str.replace('}', ' ');
                            }
                        }
                        valueList.add(str.trim());
                    }
                    buf = new StringBuilder();
                }
                break;
            default:
                buf.append(c);
                break;
            }
        }
        if (buf.length() > 0) {
            valueList.add(buf.toString());
        }
        return valueList.toArray(new String[valueList.size()]);
    }

    public String[] getResponseFields() {
        return responseFields;
    }

    public String[] getHighlightingFields() {
        return highlightingFields;
    }

    @Override
    public int getHighlightSnippetSize() {
        return highlightSnippetSize;
    }

    public String getShards() {
        return shards;
    }

    public void setShards(String shards) {
        this.shards = shards;
    }

    String escapeValue(String value) {
        return Constants.SOLR_FIELD_RESERVED_PATTERN.matcher(value).replaceAll(
                "\\\\$1");
    }
}
