/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.task;

import java.io.File;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.db.exentity.CrawlingSession;
import jp.sf.fess.helper.CrawlingSessionHelper;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.service.CrawlingSessionService;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.FessProperties;

import org.seasar.chronos.core.TaskTrigger;
import org.seasar.chronos.core.annotation.task.Task;
import org.seasar.chronos.core.trigger.CCronTrigger;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Task
public class CrawlTask implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(CrawlTask.class);

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected CrawlingSessionService crawlingSessionService;

    private CCronTrigger trigger;

    public TaskTrigger getTrigger() {
        if (trigger == null) {
            trigger = new CCronTrigger(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        } else {
            trigger.setExpression(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        }
        return trigger;
    }

    public void setExpression(String cronExpression) {
        if (trigger != null)
            trigger.setExpression(cronExpression);
    }

    public String getExpression() {
        if (trigger != null) {
            trigger.getCronExpression();
        }
        return null;
    }

    public void doExecute() {
        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        if (systemHelper.readyCrawlProcess()) {
            systemHelper.setForceStop(false);

            // create session id
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            String sessionId = sdf.format(new Date());
            systemHelper.setSessionId(sessionId);
            // store crawling session
            CrawlingSession crawlingSession = new CrawlingSession(sessionId);
            crawlingSessionService.store(crawlingSession);
            try {
                if (Constants.TRUE.equals(crawlerProperties.getProperty(
                        Constants.SNAPSHOT_REPLICATION_PROPERTY,
                        Constants.FALSE))) {
                    Map<String, String> infoMap = null;
                    try { // replication
                        infoMap = doReplication(sessionId);
                    } finally {
                        if (infoMap != null && !infoMap.isEmpty()) {
                            try {
                                CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                                        .getComponent("crawlingSessionHelper");
                                crawlingSessionHelper.put(sessionId, infoMap);
                            } catch (Exception e) {
                                logger.warn(
                                        "Failed to store crawling information.",
                                        e);
                            }
                        }
                    }
                } else {
                    doCrawl(sessionId);
                }
            } finally {
                systemHelper.finishCrawlProcess();
            }
        } else {
            logger.warn("Crawler is running now.");
        }
    }

    public Map<String, String> doReplication(String sessionId) {
        String snapshotPath = crawlerProperties.getProperty(
                Constants.SNAPSHOT_PATH_PROPERTY, null);
        if (StringUtil.isBlank(snapshotPath)) {
            if (logger.isWarnEnabled()) {
                logger.warn("A snapshot path is a blank.");
            }
            return null;
        }
        // TODO remote snapshot path
        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        File snapshotDir = systemHelper.getSnapshotDir(snapshotPath);
        if (!snapshotDir.exists()) {
            if (logger.isWarnEnabled()) {
                logger.warn(snapshotPath + " does not exist.");
            }
            return null;
        }

        if (logger.isInfoEnabled()) {
            logger.info("Starting Replication..");
        }

        Map<String, String> infoMap = new HashMap<String, String>();
        SimpleDateFormat dateFormat = new SimpleDateFormat(
                "yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        long totalTime = System.currentTimeMillis();

        infoMap.put(Constants.REPLICATION_START_TIME,
                dateFormat.format(new Date()));
        boolean completed = false;
        try {
            // TODO search lock?

            SolrServerGroup solrServerGroup = solrServerManager
                    .getUpdateSolrServerGroup();

            // replication
            solrServerGroup.replicate(snapshotDir);

            // commit
            long startTime = System.currentTimeMillis();
            solrServerGroup.commit();
            startTime = System.currentTimeMillis() - startTime;
            if (logger.isInfoEnabled()) {
                logger.info("[EXEC TIME] replication commit time: " + startTime
                        + "ms");
            }

            String serverRotationStr = crawlerProperties.getProperty(
                    Constants.SERVER_ROTATION_PROPERTY, Constants.TRUE);
            if (Constants.TRUE.equalsIgnoreCase(serverRotationStr)) {
                // apply
                solrServerManager.applyNewServerGroup();
            }

            completed = true;
        } catch (Throwable t) {
            if (logger.isWarnEnabled()) {
                logger.warn("Interrupted a replication task.", t);
            }
        } finally {
            infoMap.put(Constants.REPLICATION_STATUS, completed ? Constants.T
                    : Constants.F);
            infoMap.put(Constants.REPLICATION_END_TIME,
                    dateFormat.format(new Date()));
            infoMap.put(Constants.REPLICATION_EXEC_TIME,
                    Long.toString(System.currentTimeMillis() - totalTime));
        }
        if (logger.isInfoEnabled()) {
            logger.info("Finished Replication");
        }
        return infoMap;
    }

    protected void doCrawl(String sessionId) {
        SingletonS2Container.getComponent(SystemHelper.class).executeCrawler(
                sessionId);
    }

    public void catchException(Exception e) {
        logger.error("Failed to execute crawl task.", e);
    }

}
