/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.solr;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;

import org.apache.solr.common.SolrInputDocument;
import org.seasar.dbflute.cbean.EntityRowHandler;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.robot.db.cbean.AccessResultCB;
import org.seasar.robot.db.exbhv.AccessResultBhv;
import org.seasar.robot.db.exentity.AccessResult;
import org.seasar.robot.entity.AccessResultData;
import org.seasar.robot.service.DataService;
import org.seasar.robot.transformer.Transformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexUpdater extends Thread {
    private static final Logger logger = LoggerFactory
            .getLogger(IndexUpdater.class);

    protected List<String> sessionIdList;

    protected SolrServerGroup solrServerGroup;

    @Resource
    protected DataService dataService;

    @Resource
    protected AccessResultBhv accessResultBhv;

    public int maxDocumentCacheSize = 20;

    protected boolean finishCrawling = false;

    public long updateInterval = 60000; // 1 min

    protected long executeTime;

    protected long documentSize;

    protected long commitPerCount = 0;

    public IndexUpdater() {
    }

    @Override
    public void run() {
        if (dataService == null) {
            throw new FessSystemException("DataService is null.");
        }

        executeTime = 0;
        documentSize = 0;

        AccessResultCB cb = new AccessResultCB();
        cb.setupSelect_AccessResultDataAsOne();
        cb.query().setSessionId_InScope(sessionIdList);
        cb.query().addOrderBy_CreateTime_Asc();
        cb.query().setStatus_Equal(org.seasar.robot.Constants.OK_STATUS);

        final List<SolrInputDocument> docList = new ArrayList<SolrInputDocument>();
        final List<org.seasar.robot.entity.AccessResult> accessResultList = new ArrayList<org.seasar.robot.entity.AccessResult>();

        long updateTime = System.currentTimeMillis();

        while (!finishCrawling || !accessResultList.isEmpty()) {

            updateTime = System.currentTimeMillis() - updateTime;
            executeTime += updateTime;

            long interval = updateInterval - updateTime;
            if (interval > 0) {
                // sleep
                try {
                    Thread.sleep(interval); // 1 min (default)
                } catch (InterruptedException e) {
                    logger.warn("Interrupted index update.", e);
                }
            }

            docList.clear();
            accessResultList.clear();

            updateTime = System.currentTimeMillis();

            accessResultBhv.selectCursor(cb,
                    new EntityRowHandler<AccessResult>() {
                        public void handle(AccessResult accessResult) {
                            accessResult.setStatus(Constants.DONE_STATUS);
                            accessResultList.add(accessResult);

                            if (accessResult.getHttpStatusCode() != 200) {
                                // invalid page
                                return;
                            }

                            AccessResultData accessResultData = accessResult
                                    .getAccessResultData();
                            if (accessResultData != null) {
                                try {
                                    Transformer transformer = SingletonS2Container
                                            .getComponent(accessResultData
                                                    .getTransformerName());
                                    if (transformer == null) {
                                        // no transformer
                                        if (logger.isDebugEnabled()) {
                                            logger
                                                    .debug("No transformer: "
                                                            + accessResultData
                                                                    .getTransformerName());
                                        }
                                        return;
                                    }
                                    Map<String, String> map = (Map<String, String>) transformer
                                            .getData(accessResultData);

                                    SolrInputDocument doc = new SolrInputDocument();
                                    // add data
                                    for (Map.Entry<String, String> entry : map
                                            .entrySet()) {
                                        doc.addField(entry.getKey(), entry
                                                .getValue());
                                    }

                                    if (docList.size() > maxDocumentCacheSize) {
                                        solrServerGroup.add(docList);
                                        docList.clear();

                                    } else {
                                        docList.add(doc);
                                    }
                                    documentSize++;
                                    // commit
                                    if (commitPerCount > 0
                                            && documentSize % commitPerCount == 0) {
                                        if (!docList.isEmpty()) {
                                            solrServerGroup.add(docList);
                                            docList.clear();
                                        }
                                        solrServerGroup.commit();
                                    }
                                } catch (Exception e) {
                                    logger.warn("Could not add a doc: "
                                            + accessResult.getUrl(), e);
                                }
                            }

                        }
                    });

            if (!docList.isEmpty()) {
                solrServerGroup.add(docList);
            }

            if (!accessResultList.isEmpty()) {
                dataService.update(accessResultList);
            }

        }

        if (logger.isInfoEnabled()) {
            logger.info("[EXEC TIME] index update time: " + executeTime + "ms");
        }
    }

    public long getExecuteTime() {
        return executeTime;
    }

    public List<String> getSessionIdList() {
        return sessionIdList;
    }

    public void setSessionIdList(List<String> sessionIdList) {
        this.sessionIdList = sessionIdList;
    }

    public SolrServerGroup getSolrServerGroup() {
        return solrServerGroup;
    }

    public void setSolrServerGroup(SolrServerGroup solrServerGroup) {
        this.solrServerGroup = solrServerGroup;
    }

    public void setFinishCrawling(boolean finishCrawling) {
        this.finishCrawling = finishCrawling;
    }

    public long getDocumentSize() {
        return documentSize;
    }

    public void setCommitPerCount(long commitPerCount) {
        this.commitPerCount = commitPerCount;
    }

    @Binding(bindingType = BindingType.MAY)
    @Override
    public void setUncaughtExceptionHandler(UncaughtExceptionHandler eh) {
        super.setUncaughtExceptionHandler(eh);
    }

    @Binding(bindingType = BindingType.MAY)
    public static void setDefaultUncaughtExceptionHandler(
            UncaughtExceptionHandler eh) {
        Thread.setDefaultUncaughtExceptionHandler(eh);
    }
}
