//
// Copyright 2009-2010 Fess Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#import "SearchResultDataSource.h"
#import "SearchResultModel.h"
#import "SearchResult.h"

#import <Three20Core/NSDateAdditions.h>

@implementation SearchResultDataSource

- (id)initWithSearchQuery:(NSString*)searchQuery {
  if (self = [super init]) {
    _searchResultModel = [[SearchResultModel alloc] initWithSearchQuery:searchQuery];
  }

  return self;
}

- (void)dealloc {
  TT_RELEASE_SAFELY(_searchResultModel);

  [super dealloc];
}

- (id<TTModel>)model {
  return _searchResultModel;
}

- (NSString*)normalizedUrl:(NSString*)str {
	if (str == nil) {
		return @"";
	}
	str = [str 
		   stringByReplacingOccurrencesOfRegex:@"&"
		   withString:@"&amp;"];
	str = [str 
		   stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
	return str;
}

- (NSString*)normalizedStr:(NSString*)str {
	if (str == nil) {
		return @"";
	}
	str = [str 
		   stringByReplacingOccurrencesOfRegex:@"&"
		   withString:@"&amp;"];
	str = [str 
		   stringByReplacingOccurrencesOfRegex:@"[:cntrl:]"
		   withString:@" "];
	str = [str 
		   stringByReplacingOccurrencesOfRegex:@"<[^>]+>"
		   withString:@" "];
	str = [str 
		   stringByReplacingOccurrencesOfRegex:@"\\s+"
		   withString:@" "];
	str = [str 
		   stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
	return str;
}

- (void)tableViewDidLoadModel:(UITableView*)tableView {
	NSMutableArray* items = [[NSMutableArray alloc] init];

	for (SearchResult* result in _searchResultModel.results) {
		//TTDPRINT(@"Response text: %@", response.text);
		NSString *urlLink = [self normalizedUrl:result.urlLink];
		NSString *contentTitle = [self normalizedStr:result.contentTitle];
		NSString *contentDescription = [self normalizedStr:result.contentDescription];
		NSString *resultText = [NSString stringWithFormat:@"<a href=\"%@\"><b>%@</b></a><p>%@</p>",
							   urlLink,
							   contentTitle,
							   contentDescription
							   ];
		TTDPRINT(@"itemText=%@", resultText);
		TTStyledText* styledText = [TTStyledText textFromXHTML:resultText
													lineBreaks:YES URLs:YES];

		// TTDASSERT(nil != styledText);
		if (nil != styledText) {
			[items addObject:[TTTableStyledTextItem itemWithText:styledText]];
		}
	}
	
	int offset = _searchResultModel.offset;
	int offsetLimit = _searchResultModel.recordCount - _searchResultModel.pageSize;
	if (offset < offsetLimit) {
		NSString *msg = [NSString stringWithFormat:NSLocalizedString(@"SearchReadMoreCell", nil), _searchResultModel.pageSize];
		[items addObject:[TTTableMoreButton itemWithText:msg]];
	}

	self.items = items;
	TT_RELEASE_SAFELY(items);
	
	UIViewController *controller = [[TTNavigator navigator] visibleViewController];
	controller.title = [NSString stringWithFormat:NSLocalizedString(@"SearchResultTitle", nil), _searchResultModel.recordCount];
}

- (NSString*)titleForLoading:(BOOL)reloading {
  if (reloading) {
    return NSLocalizedString(@"SearchResultUpdating", nil);
  } else {
    return NSLocalizedString(@"SearchResultLoading", nil);
  }
}

- (NSString*)titleForEmpty {
  return NSLocalizedString(@"SearchNoResultsFound", nil);
}

- (NSString*)subtitleForError:(NSError*)error {
  return NSLocalizedString(@"SearchSorryNoStream", nil);
}

- (void)search:(NSString*)text {
	_searchResultModel.searchQuery = text;
}

@end

