/*
 * Copyright 2009-2012 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.helper.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.helper.RoleQueryHelper;
import jp.sf.fess.util.QueryUtil;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.util.RequestUtil;

public class QueryHelperImpl implements QueryHelper, Serializable {

    private static final long serialVersionUID = 1L;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected RoleQueryHelper roleQueryHelper;

    protected String[] responseFields = new String[] { "id", "score", "boost",
            "contentLength", "host", "site", "lastModified", "mimetype",
            "tstamp", "title", "digest", "url" };

    protected String[] highlightingFields = new String[] { "digest", "cache" };

    protected String[] supportedFields = new String[] { "url", "host", "site",
            "title", "content", "contentLength", "lastModified", "mimetype",
            "label", "segment" };

    protected String[] facetFields = new String[] { "url", "host", "site",
            "title", "content", "contentLength", "lastModified", "mimetype",
            "label", "segment" };

    protected String sortPrefix = "sort:";

    protected String[] supportedSortFields = new String[] { "tstamp",
            "contentLength", "lastModified" };

    protected int highlightSnippetSize = 5;

    protected String shards;

    protected boolean useBigram = true;

    protected String additionalQuery;

    protected int maxFilterQueriesForRole = 3;

    protected int timeAllowed = -1;

    protected Map<String, String[]> requestParameterMap = new HashMap<String, String[]>();

    /* (non-Javadoc)
     * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
     */
    @Override
    public SearchQuery build(final String query) {
        return build(query, true);
    }

    /* (non-Javadoc)
         * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
         */
    @Override
    public SearchQuery build(final String query, final boolean envCondition) {
        String q;
        if (envCondition && additionalQuery != null
                && StringUtil.isNotBlank(query)) {
            q = query + " " + additionalQuery;
        } else {
            q = query;
        }

        final SearchQuery searchQuery = buildQuery(q);
        if (!searchQuery.queryExists()) {
            return searchQuery.query("");
        }

        if (browserTypeHelper == null && roleQueryHelper == null
                || !envCondition) {
            return searchQuery;
        }

        StringBuilder queryBuf = new StringBuilder(255);
        queryBuf.append(searchQuery.getQuery());

        if (browserTypeHelper != null) {
            searchQuery
                    .addFilterQuery("type:"
                            + QueryUtil.escapeValue(browserTypeHelper
                                    .getBrowserType()));
        }

        if (roleQueryHelper != null) {
            final List<String> roleList = roleQueryHelper.build();
            if (roleList.size() > maxFilterQueriesForRole) {
                // add query
                final String sq = queryBuf.toString();
                queryBuf = new StringBuilder(255);
                final boolean hasQueries = sq.contains(" AND ")
                        || sq.contains(" OR ");
                if (hasQueries) {
                    queryBuf.append('(');
                }
                queryBuf.append(sq);
                if (hasQueries) {
                    queryBuf.append(')');
                }
                queryBuf.append(" AND ");
                if (roleList.size() > 1) {
                    queryBuf.append('(');
                }
                queryBuf.append(getRoleQuery(roleList));
                if (roleList.size() > 1) {
                    queryBuf.append(')');
                }
            } else if (!roleList.isEmpty()) {
                // add filter query
                searchQuery.addFilterQuery(getRoleQuery(roleList));
            }
        }

        return searchQuery.query(queryBuf.toString());
    }

    private String getRoleQuery(final List<String> roleList) {
        final StringBuilder queryBuf = new StringBuilder(255);
        boolean isFirst = true;
        for (final String role : roleList) {
            if (isFirst) {
                isFirst = false;
            } else {
                queryBuf.append(" OR ");

            }
            queryBuf.append("role:");
            queryBuf.append(QueryUtil.escapeValue(role));
        }
        return queryBuf.toString();
    }

    protected SearchQuery buildQuery(final String query) {
        final String[] values = splitQuery(query);
        if (values.length == 0) {
            return new SearchQuery().query("");
        }

        final List<String> highLightQueryList = new ArrayList<String>();

        final SearchQuery searchQuery = new SearchQuery();

        final StringBuilder queryBuf = new StringBuilder(255);
        final StringBuilder contentBuf = new StringBuilder(255);
        final List<String> notOperatorList = new ArrayList<String>();
        String operator = " AND ";
        String queryOperator = " AND ";
        boolean notOperatorFlag = false;
        int queryOperandCount = 0;
        int contentOperandCount = 0;
        for (final String value : values) {
            boolean nonPrefix = false;
            // check prefix
            for (final String field : supportedFields) {
                final String prefix = field + ":";
                if (value.startsWith(prefix)
                        && value.length() != prefix.length()) {
                    if (queryBuf.length() > 0) {
                        queryBuf.append(operator);
                    } else if (contentBuf.length() > 0) {
                        queryOperator = operator;
                    }
                    final String targetWord = value.substring(prefix.length());
                    if (notOperatorFlag) {
                        final StringBuilder buf = new StringBuilder(100);
                        buf.append(prefix);
                        appendQueryValue(buf, targetWord);
                        notOperatorList.add(buf.toString());
                        notOperatorFlag = false;
                    } else {
                        queryBuf.append(prefix);
                        appendQueryValue(queryBuf, targetWord);
                        queryOperandCount++;
                    }
                    nonPrefix = true;
                    operator = " AND ";
                    if (!"label".equals(field)) {
                        highLightQueryList.add(targetWord);
                    }
                    break;
                }
            }

            // sort
            if (value.startsWith(sortPrefix)
                    && value.length() != sortPrefix.length()) {
                final String[] sortFieldPairs = value.substring(
                        sortPrefix.length()).split(",");
                for (final String sortFieldPairStr : sortFieldPairs) {
                    final String[] sortFieldPair = sortFieldPairStr
                            .split("\\.");
                    if (isSupportedSortField(sortFieldPair[0])) {
                        if (sortFieldPair.length == 1) {
                            searchQuery.addSortField(sortFieldPair[0],
                                    Constants.ASC);
                        } else {
                            searchQuery.addSortField(sortFieldPair[0],
                                    sortFieldPair[1]);
                        }
                    }
                }
                continue;
            }

            if (!nonPrefix) {
                if ("AND".equals(value)) {
                    operator = " AND ";
                } else if ("OR".equals(value)) {
                    operator = " OR ";
                } else if ("NOT".equals(value)) {
                    notOperatorFlag = true;
                } else if (notOperatorFlag) {
                    final StringBuilder buf = new StringBuilder(100);

                    buf.append("content:");
                    appendQueryValue(buf, value);
                    notOperatorList.add(buf.toString());

                    operator = " AND ";
                    notOperatorFlag = false;
                    highLightQueryList.add(value);
                } else {
                    // content
                    if (contentBuf.length() > 0) {
                        contentBuf.append(operator);
                    }
                    contentBuf.append("content:");
                    appendQueryValue(contentBuf, value);
                    contentOperandCount++;

                    operator = " AND ";
                    highLightQueryList.add(value);
                }
            }
        }

        if (contentBuf.length() > 0) {
            boolean append = false;
            if (queryBuf.length() > 0) {
                append = true;
            }

            if (append) {
                queryBuf.append(queryOperator);
                if (contentOperandCount > 1) {
                    queryBuf.append('(');
                }
            }
            if (contentBuf.length() > 0) {
                queryBuf.append(contentBuf.toString());
            }
            if (append && contentOperandCount > 1) {
                queryBuf.append(')');
            }
        }

        StringBuilder searchQueryBuf = new StringBuilder(255);
        if (queryBuf.length() > 0) {
            searchQueryBuf.append(queryBuf.toString());
            operator = " AND ";
        } else {
            operator = Constants.EMPTY_STRING;
        }
        if (!notOperatorList.isEmpty()) {
            final String q = searchQueryBuf.toString();
            searchQueryBuf = new StringBuilder(255);
            final int count = queryOperandCount + contentOperandCount;
            if (count > 1) {
                searchQueryBuf.append('(');
            }
            searchQueryBuf.append(q);
            if (count > 1) {
                searchQueryBuf.append(')');
            }
            for (final String notOperator : notOperatorList) {
                searchQueryBuf.append(operator);
                searchQueryBuf.append("NOT ");
                searchQueryBuf.append(notOperator);
                operator = " AND ";
            }
        }

        // set queries to request for HighLight
        RequestUtil
                .getRequest()
                .setAttribute(
                        Constants.HIGHLIGHT_QUERIES,
                        highLightQueryList
                                .toArray(new String[highLightQueryList.size()]));

        return searchQuery.query(searchQueryBuf.toString());
    }

    private void appendQueryValue(final StringBuilder buf, String value) {
        // check reserved
        boolean reserved = false;
        for (final String element : Constants.RESERVED) {
            if (element.equals(value)) {
                reserved = true;
                break;
            }
        }

        if (useBigram && value.length() == 1
                && !StringUtils.isAsciiPrintable(value)) {
            // if using bigram, add ?
            value = value + "?";
        }

        if (reserved) {
            buf.append('\\');
            buf.append(value);
            return;
        }

        String fuzzyValue = null;
        String proximityValue = null;
        String caretValue = null;
        final int tildePos = value.lastIndexOf('~');
        final int caretPos = value.indexOf('^');
        if (tildePos > caretPos) {
            if (tildePos > 0) {
                final String tildeValue = value.substring(tildePos);
                if (tildeValue.length() > 1) {
                    final StringBuilder buf1 = new StringBuilder();
                    final StringBuilder buf2 = new StringBuilder();
                    boolean isComma = false;
                    for (int i = 1; i < tildeValue.length(); i++) {
                        final char c = tildeValue.charAt(i);
                        if (c >= '0' && c <= '9') {
                            if (isComma) {
                                buf2.append(c);
                            } else {
                                buf1.append(c);
                            }
                        } else if (c == '.') {
                            if (isComma) {
                                break;
                            } else {
                                isComma = true;
                            }
                        } else {
                            break;
                        }
                    }
                    if (buf1.length() == 0) {
                        fuzzyValue = "~";
                    } else {
                        final int intValue = Integer.parseInt(buf1.toString());
                        if (intValue <= 0) {
                            // fuzzy
                            buf1.append('.').append(buf2.toString());
                            fuzzyValue = "~" + buf1.toString();
                        } else {
                            // proximity
                            proximityValue = "~" + Integer.toString(intValue);
                        }
                    }
                } else {
                    fuzzyValue = "~";
                }

                value = value.substring(0, tildePos);
            }
        } else {
            if (caretPos > 0) {
                caretValue = value.substring(caretPos);
                value = value.substring(0, caretPos);
            }
        }
        if (value.startsWith("[") && value.endsWith("]")) {
            final String[] split = value.substring(1, value.length() - 1)
                    .split(" TO ");
            if (split.length == 2 && split[0].length() > 0
                    && split[1].length() > 0) {
                buf.append('[');
                buf.append(QueryUtil.escapeValue(split[0]));
                buf.append(" TO ");
                buf.append(QueryUtil.escapeValue(split[1]));
                buf.append(']');
            } else {
                buf.append(QueryUtil.escapeValue(value));
            }
        } else if (value.startsWith("{") && value.endsWith("}")) {
            final String[] split = value.substring(1, value.length() - 1)
                    .split(" TO ");
            if (split.length == 2 && split[0].length() > 0
                    && split[1].length() > 0) {
                buf.append('{');
                buf.append(QueryUtil.escapeValue(split[0]));
                buf.append(" TO ");
                buf.append(QueryUtil.escapeValue(split[1]));
                buf.append('}');
            } else {
                buf.append(QueryUtil.escapeValue(value));
            }
        } else {
            if (proximityValue == null) {
                buf.append(QueryUtil.escapeValue(value));
            } else {
                buf.append('"').append(QueryUtil.escapeValue(value))
                        .append('"');
            }
        }

        if (fuzzyValue != null) {
            buf.append(fuzzyValue);
        } else if (proximityValue != null) {
            buf.append(proximityValue);
        } else if (caretValue != null) {
            buf.append(caretValue);
        }
    }

    private boolean isSupportedSortField(final String field) {
        for (final String f : supportedSortFields) {
            if (f.equals(field)) {
                return true;
            }
        }
        return false;
    }

    protected String[] splitQuery(final String query) {
        final List<String> valueList = new ArrayList<String>();
        StringBuilder buf = new StringBuilder();
        boolean quoted = false;
        boolean squareBracket = false;
        boolean curlyBracket = false;
        for (int i = 0; i < query.length(); i++) {
            final char c = query.charAt(i);
            switch (c) {
            case '[':
                buf.append(c);
                if (!quoted && !curlyBracket) {
                    squareBracket = true;
                }
                break;
            case ']':
                buf.append(c);
                squareBracket = false;
                break;
            case '{':
                buf.append(c);
                if (!quoted && !squareBracket) {
                    curlyBracket = true;
                }
                break;
            case '}':
                buf.append(c);
                curlyBracket = false;
                break;
            case '"':
                if (!curlyBracket && !squareBracket) {
                    quoted = !quoted;
                } else {
                    buf.append(c);
                }
                break;
            case ' ':
            case '\u3000':
                if (quoted || curlyBracket || squareBracket) {
                    buf.append(c);
                } else {
                    if (buf.length() > 0) {
                        String str = buf.toString();
                        if (str.startsWith("[") || str.startsWith("{")) {
                            final int pos = str.indexOf(" TO ");
                            if (pos < 0) {
                                str = str.replace('[', ' ');
                                str = str.replace(']', ' ');
                                str = str.replace('{', ' ');
                                str = str.replace('}', ' ');
                            }
                        }
                        valueList.add(str.trim());
                    }
                    buf = new StringBuilder();
                }
                break;
            default:
                buf.append(c);
                break;
            }
        }
        if (buf.length() > 0) {
            valueList.add(buf.toString());
        }
        return valueList.toArray(new String[valueList.size()]);
    }

    @Override
    public boolean isFacetField(final String field) {
        if (StringUtil.isBlank(field)) {
            return false;
        }
        boolean flag = false;
        for (final String f : facetFields) {
            if (field.equals(f)) {
                flag = true;
            }
        }
        return flag;
    }

    @Override
    public String buildFacetQuery(final String query) {
        final String[] values = splitQuery(query);
        if (values.length == 0) {
            return "";
        }

        final StringBuilder queryBuf = new StringBuilder(255);
        final StringBuilder contentBuf = new StringBuilder(255);
        final List<String> notOperatorList = new ArrayList<String>();
        String operator = " AND ";
        String queryOperator = " AND ";
        boolean notOperatorFlag = false;
        int queryOperandCount = 0;
        int contentOperandCount = 0;
        for (final String value : values) {
            boolean nonPrefix = false;
            // check prefix
            for (final String field : facetFields) {
                final String prefix = field + ":";
                if (value.startsWith(prefix)
                        && value.length() != prefix.length()) {
                    if (queryBuf.length() > 0) {
                        queryBuf.append(operator);
                    } else if (contentBuf.length() > 0) {
                        queryOperator = operator;
                    }
                    final String targetWord = value.substring(prefix.length());
                    if (notOperatorFlag) {
                        final StringBuilder buf = new StringBuilder(100);
                        buf.append(prefix);
                        appendQueryValue(buf, targetWord);
                        notOperatorList.add(buf.toString());
                        notOperatorFlag = false;
                    } else {
                        queryBuf.append(prefix);
                        appendQueryValue(queryBuf, targetWord);
                        queryOperandCount++;
                    }
                    nonPrefix = true;
                    operator = " AND ";
                    break;
                }
            }

            // sort
            if (value.startsWith(sortPrefix)
                    && value.length() != sortPrefix.length()) {
                // skip
                continue;
            }

            if (!nonPrefix) {
                if ("AND".equals(value)) {
                    operator = " AND ";
                } else if ("OR".equals(value)) {
                    operator = " OR ";
                } else if ("NOT".equals(value)) {
                    notOperatorFlag = true;
                } else if (notOperatorFlag) {
                    final StringBuilder buf = new StringBuilder(100);

                    buf.append("content:");
                    appendQueryValue(buf, value);
                    notOperatorList.add(buf.toString());

                    operator = " AND ";
                    notOperatorFlag = false;
                } else {
                    // content
                    if (contentBuf.length() > 0) {
                        contentBuf.append(operator);
                    }
                    contentBuf.append("content:");
                    appendQueryValue(contentBuf, value);
                    contentOperandCount++;

                    operator = " AND ";
                }
            }
        }

        if (contentBuf.length() > 0) {
            boolean append = false;
            if (queryBuf.length() > 0) {
                append = true;
            }

            if (append) {
                queryBuf.append(queryOperator);
                if (contentOperandCount > 1) {
                    queryBuf.append('(');
                }
            }
            if (contentBuf.length() > 0) {
                queryBuf.append(contentBuf.toString());
            }
            if (append && contentOperandCount > 1) {
                queryBuf.append(')');
            }
        }

        StringBuilder searchQueryBuf = new StringBuilder(255);
        if (queryBuf.length() > 0) {
            searchQueryBuf.append(queryBuf.toString());
            operator = " AND ";
        } else {
            operator = Constants.EMPTY_STRING;
        }
        if (!notOperatorList.isEmpty()) {
            final String q = searchQueryBuf.toString();
            searchQueryBuf = new StringBuilder(255);
            final int count = queryOperandCount + contentOperandCount;
            if (count > 1) {
                searchQueryBuf.append('(');
            }
            searchQueryBuf.append(q);
            if (count > 1) {
                searchQueryBuf.append(')');
            }
            for (final String notOperator : notOperatorList) {
                searchQueryBuf.append(operator);
                searchQueryBuf.append("NOT ");
                searchQueryBuf.append(notOperator);
                operator = " AND ";
            }
        }

        return searchQueryBuf.toString();
    }

    @Override
    public boolean isFacetSortValue(final String sort) {
        return "count".equals(sort) || "index".equals(sort);
    }

    /**
     * @return the responseFields
     */
    @Override
    public String[] getResponseFields() {
        return responseFields;
    }

    /**
     * @param responseFields the responseFields to set
     */
    public void setResponseFields(final String[] responseFields) {
        this.responseFields = responseFields;
    }

    /**
     * @return the highlightingFields
     */
    @Override
    public String[] getHighlightingFields() {
        return highlightingFields;
    }

    /**
     * @param highlightingFields the highlightingFields to set
     */
    public void setHighlightingFields(final String[] highlightingFields) {
        this.highlightingFields = highlightingFields;
    }

    /**
     * @return the supportedFields
     */
    public String[] getSupportedFields() {
        return supportedFields;
    }

    /**
     * @param supportedFields the supportedFields to set
     */
    public void setSupportedFields(final String[] supportedFields) {
        this.supportedFields = supportedFields;
    }

    /**
     * @return the facetFields
     */
    public String[] getFacetFields() {
        return facetFields;
    }

    /**
     * @param facetFields the facetFields to set
     */
    public void setFacetFields(final String[] facetFields) {
        this.facetFields = facetFields;
    }

    /**
     * @return the sortPrefix
     */
    public String getSortPrefix() {
        return sortPrefix;
    }

    /**
     * @param sortPrefix the sortPrefix to set
     */
    public void setSortPrefix(final String sortPrefix) {
        this.sortPrefix = sortPrefix;
    }

    /**
     * @return the supportedSortFields
     */
    public String[] getSupportedSortFields() {
        return supportedSortFields;
    }

    /**
     * @param supportedSortFields the supportedSortFields to set
     */
    public void setSupportedSortFields(final String[] supportedSortFields) {
        this.supportedSortFields = supportedSortFields;
    }

    /**
     * @return the highlightSnippetSize
     */
    @Override
    public int getHighlightSnippetSize() {
        return highlightSnippetSize;
    }

    /**
     * @param highlightSnippetSize the highlightSnippetSize to set
     */
    public void setHighlightSnippetSize(final int highlightSnippetSize) {
        this.highlightSnippetSize = highlightSnippetSize;
    }

    /**
     * @return the shards
     */
    @Override
    public String getShards() {
        return shards;
    }

    /**
     * @param shards the shards to set
     */
    public void setShards(final String shards) {
        this.shards = shards;
    }

    /**
     * @return the useBigram
     */
    public boolean isUseBigram() {
        return useBigram;
    }

    /**
     * @param useBigram the useBigram to set
     */
    public void setUseBigram(final boolean useBigram) {
        this.useBigram = useBigram;
    }

    /**
     * @return the additionalQuery
     */
    public String getAdditionalQuery() {
        return additionalQuery;
    }

    /**
     * @param additionalQuery the additionalQuery to set
     */
    public void setAdditionalQuery(final String additionalQuery) {
        this.additionalQuery = additionalQuery;
    }

    public int getMaxFilterQueriesForRole() {
        return maxFilterQueriesForRole;
    }

    public void setMaxFilterQueriesForRole(final int maxFilterQuerysForRole) {
        maxFilterQueriesForRole = maxFilterQuerysForRole;
    }

    /**
     * @return the timeAllowed
     */
    @Override
    public int getTimeAllowed() {
        return timeAllowed;
    }

    /**
     * @param timeAllowed the timeAllowed to set
     */
    public void setTimeAllowed(final int timeAllowed) {
        this.timeAllowed = timeAllowed;
    }

    public void addRequestParameter(final String name, final String... values) {
        requestParameterMap.put(name, values);
    }

    public void addRequestParameter(final String name, final String value) {
        if (value != null) {
            requestParameterMap.put(name, new String[] { value });
        }
    }

    @Override
    public Set<Entry<String, String[]>> getRequestParameterSet() {
        return requestParameterMap.entrySet();
    }
}
