/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.helper;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.db.exentity.CrawlingSession;
import jp.sf.fess.db.exentity.CrawlingSessionInfo;
import jp.sf.fess.service.CrawlingSessionService;

import org.seasar.framework.container.SingletonS2Container;

public class CrawlingSessionHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    protected CrawlingSessionService getCrawlingSessionService() {
        return SingletonS2Container.getComponent(CrawlingSessionService.class);
    }

    public String getCanonicalSessionId(String sessionId) {
        int idx = sessionId.indexOf('-');
        if (idx >= 0) {
            return sessionId.substring(0, idx);
        }
        return sessionId;
    }

    public void put(String sessionId, Map<String, String> infoMap) {
        CrawlingSession crawlingSession = getCrawlingSessionService().get(
                sessionId);
        if (crawlingSession == null) {
            crawlingSession = new CrawlingSession(sessionId);
            try {
                getCrawlingSessionService().store(crawlingSession);
            } catch (Exception e) {
                throw new FessSystemException("No crawling session.", e);
            }
        }

        List<CrawlingSessionInfo> crawlingSessionInfoList = new ArrayList<CrawlingSessionInfo>();
        for (Map.Entry<String, String> entry : infoMap.entrySet()) {
            CrawlingSessionInfo crawlingSessionInfo = new CrawlingSessionInfo();
            crawlingSessionInfo.setCrawlingSessionId(crawlingSession.getId());
            crawlingSessionInfo.setKey(entry.getKey());
            crawlingSessionInfo.setValue(entry.getValue());
            crawlingSessionInfoList.add(crawlingSessionInfo);
        }
        getCrawlingSessionService().storeInfo(crawlingSessionInfoList);
    }

    public Map<String, String> getInfoMap(String sessionId) {
        List<CrawlingSessionInfo> crawlingSessionInfoList = getCrawlingSessionService()
                .getCrawlingSessionInfoList(sessionId);
        Map<String, String> infoMap = new HashMap<String, String>();
        for (CrawlingSessionInfo crawlingSessionInfo : crawlingSessionInfoList) {
            infoMap.put(crawlingSessionInfo.getKey(),
                    crawlingSessionInfo.getValue());
        }
        return infoMap;
    }

    public String generateId(Map<String, Object> dataMap) {
        String url = (String) dataMap.get("url");
        List<String> browserTypeList = (List<String>) dataMap.get("type");
        List<String> roleTypeList = (List<String>) dataMap.get("role");
        return generateId(url, browserTypeList, roleTypeList);
    }

    private String generateId(String url, List<String> browserTypeList,
            List<String> roleTypeList) {
        StringBuilder buf = new StringBuilder(1000);
        buf.append(url);
        if (browserTypeList != null && !browserTypeList.isEmpty()) {
            Collections.sort(browserTypeList);
            buf.append(";type=");
            for (int i = 0; i < browserTypeList.size(); i++) {
                if (i != 0) {
                    buf.append(',');
                }
                buf.append(browserTypeList.get(i));
            }
        }
        if (roleTypeList != null && !roleTypeList.isEmpty()) {
            Collections.sort(roleTypeList);
            buf.append(";role=");
            for (int i = 0; i < roleTypeList.size(); i++) {
                if (i != 0) {
                    buf.append(',');
                }
                buf.append(roleTypeList.get(i));
            }
        }

        return normalize(buf.toString());
    }

    private String normalize(String value) {
        return value.replace('"', ' ');
    }
}
