/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.crud.service.BsCrawlingSessionService;
import jp.sf.fess.db.cbean.CrawlingSessionCB;
import jp.sf.fess.db.cbean.CrawlingSessionInfoCB;
import jp.sf.fess.db.exbhv.CrawlingSessionInfoBhv;
import jp.sf.fess.db.exentity.CrawlingSession;
import jp.sf.fess.db.exentity.CrawlingSessionInfo;
import jp.sf.fess.pager.CrawlingSessionPager;

import org.seasar.dbflute.cbean.coption.LikeSearchOption;
import org.seasar.framework.util.StringUtil;

public class CrawlingSessionService extends BsCrawlingSessionService {

    @Resource
    protected CrawlingSessionInfoBhv crawlingSessionInfoBhv;

    protected void setupListCondition(CrawlingSessionCB cb,
            CrawlingSessionPager crawlingSessionPager) {
        super.setupListCondition(cb, crawlingSessionPager);
        if (StringUtil.isNotBlank(crawlingSessionPager.sessionId)) {
            cb.query().setSessionId_LikeSearch(crawlingSessionPager.sessionId,
                    new LikeSearchOption().likeContain());
        }
        cb.query().addOrderBy_CreatedTime_Desc();
    }

    protected void setupStoreCondition(CrawlingSession crawlingSession) {
        if (crawlingSession == null) {
            throw new FessSystemException("Crawling Session is null.");
        }
        if (crawlingSession.getCreatedTime() == null) {
            crawlingSession.setCreatedTime(new Timestamp(new Date().getTime()));
        }
    }

    protected void setupDeleteCondition(CrawlingSession crawlingSession) {
        CrawlingSessionInfoCB cb = new CrawlingSessionInfoCB();
        cb.query().setCrawlingSessionId_Equal(crawlingSession.getId());
        crawlingSessionInfoBhv.queryDelete(cb);
    }

    public String[] getSessionIdsBefore(Date date) {
        CrawlingSessionCB cb = new CrawlingSessionCB();
        cb.query().setCreatedTime_LessEqual(new Timestamp(date.getTime()));
        List<CrawlingSession> crawlingSessionList = crawlingSessionBhv
                .selectList(cb);
        if (crawlingSessionList.isEmpty()) {
            return new String[0];
        }

        List<String> sessionIdList = new ArrayList<String>();
        List<Long> crawlingSessionIdList = new ArrayList<Long>();
        for (CrawlingSession cs : crawlingSessionList) {
            sessionIdList.add(cs.getSessionId());
            crawlingSessionIdList.add(cs.getId());
        }

        CrawlingSessionInfoCB cb2 = new CrawlingSessionInfoCB();
        cb2.query().setCrawlingSessionId_InScope(crawlingSessionIdList);
        crawlingSessionInfoBhv.queryDelete(cb2);

        crawlingSessionBhv.batchDelete(crawlingSessionList);

        return sessionIdList.toArray(new String[crawlingSessionList.size()]);
    }

    public CrawlingSession get(String sessionId) {
        CrawlingSessionCB cb = new CrawlingSessionCB();
        cb.query().setSessionId_Equal(sessionId);
        return crawlingSessionBhv.selectEntity(cb);
    }

    public void storeInfo(List<CrawlingSessionInfo> crawlingSessionInfoList) {
        if (crawlingSessionInfoList == null) {
            throw new FessSystemException("Crawling Session Info is null.");
        }

        Timestamp now = new Timestamp(new Date().getTime());
        for (CrawlingSessionInfo crawlingSessionInfo : crawlingSessionInfoList) {
            if (crawlingSessionInfo.getCreatedTime() == null) {
                crawlingSessionInfo.setCreatedTime(now);
            }
        }
        crawlingSessionInfoBhv.batchInsert(crawlingSessionInfoList);
    }

    public List<CrawlingSessionInfo> getCrawlingSessionInfoList(String sessionId) {
        CrawlingSessionInfoCB cb = new CrawlingSessionInfoCB();
        cb.query().queryCrawlingSession().setSessionId_Equal(sessionId);
        cb.query().addOrderBy_Key_Asc();
        return crawlingSessionInfoBhv.selectList(cb);
    }

    public void deleteOldSessions(String activeSessionId) {
        CrawlingSessionInfoCB cb1 = new CrawlingSessionInfoCB();
        if (StringUtil.isNotEmpty(activeSessionId)) {
            cb1.query().queryCrawlingSession().setSessionId_NotEqual(
                    activeSessionId);
        }
        crawlingSessionInfoBhv.queryDelete(cb1);
        CrawlingSessionCB cb2 = new CrawlingSessionCB();
        if (StringUtil.isNotEmpty(activeSessionId)) {
            cb2.query().setSessionId_NotEqual(activeSessionId);
        }
        crawlingSessionBhv.queryDelete(cb2);
    }
}
