/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.service;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.crud.service.BsFailureUrlService;
import jp.sf.fess.db.cbean.FailureUrlCB;
import jp.sf.fess.db.exentity.FailureUrl;
import jp.sf.fess.pager.FailureUrlPager;
import jp.sf.fess.util.FessProperties;

import org.seasar.dbflute.bhv.DeleteOption;
import org.seasar.dbflute.cbean.ListResultBean;
import org.seasar.dbflute.cbean.coption.LikeSearchOption;
import org.seasar.framework.util.StringUtil;

public class FailureUrlService extends BsFailureUrlService implements
        Serializable {

    private static final long serialVersionUID = 1L;

    @Resource
    protected FessProperties crawlerProperties;

    protected void setupListCondition(FailureUrlCB cb,
            FailureUrlPager failureUrlPager) {
        super.setupListCondition(cb, failureUrlPager);

        // setup condition
        cb.query().addOrderBy_LastAccessTime_Desc();

        buildSearchCondition(failureUrlPager, cb);
    }

    protected void setupEntityCondition(FailureUrlCB cb,
            Map<String, String> keys) {
        super.setupEntityCondition(cb, keys);

        // setup condition
        cb.setupSelect_FileCrawlingConfig();
        cb.setupSelect_WebCrawlingConfig();

    }

    protected void setupStoreCondition(FailureUrl failureUrl) {
        super.setupStoreCondition(failureUrl);

        // setup condition

    }

    protected void setupDeleteCondition(FailureUrl failureUrl) {
        super.setupDeleteCondition(failureUrl);

        // setup condition

    }

    public void deleteAll(FailureUrlPager failureUrlPager) {
        FailureUrlCB cb = new FailureUrlCB();
        buildSearchCondition(failureUrlPager, cb);
        failureUrlBhv.varyingQueryDelete(cb,
                new DeleteOption<FailureUrlCB>().allowNonQueryDelete());
    }

    private void buildSearchCondition(FailureUrlPager failureUrlPager,
            FailureUrlCB cb) {
        // search
        if (StringUtil.isNotBlank(failureUrlPager.url)) {
            cb.query().setUrl_LikeSearch(failureUrlPager.url,
                    new LikeSearchOption().likeContain());
        }

        if (StringUtil.isNotBlank(failureUrlPager.errorCountMax)) {
            cb.query().setErrorCount_LessEqual(
                    Integer.parseInt(failureUrlPager.errorCountMax));
        }
        if (StringUtil.isNotBlank(failureUrlPager.errorCountMin)) {
            cb.query().setErrorCount_GreaterEqual(
                    Integer.parseInt(failureUrlPager.errorCountMin));
        }

        if (StringUtil.isNotBlank(failureUrlPager.errorName)) {
            cb.query().setErrorName_LikeSearch(failureUrlPager.errorName,
                    new LikeSearchOption().likeContain());
        }

    }

    public List<String> getExcludedUrlList(Long webConfigId, Long fileConfigId) {
        String failureCountStr = crawlerProperties.getProperty(
                Constants.FAILURE_COUNT_THRESHOLD_PROPERTY,
                Constants.DEFAULT_FAILURE_COUNT);
        String ignoreFailureType = crawlerProperties.getProperty(
                Constants.IGNORE_FAILURE_TYPE_PROPERTY,
                Constants.DEFAULT_IGNORE_FAILURE_TYPE);
        int failureCount;
        try {
            failureCount = Integer.parseInt(failureCountStr);
        } catch (NumberFormatException ignore) {
            failureCount = 10;
        }

        if (failureCount < 0) {
            return null;
        }

        FailureUrlCB cb = new FailureUrlCB();
        cb.query().setWebConfigId_Equal(webConfigId);
        cb.query().setFileConfigId_Equal(fileConfigId);
        cb.query().setErrorCount_GreaterEqual(failureCount);
        ListResultBean<FailureUrl> list = failureUrlBhv.selectList(cb);
        if (list.isEmpty()) {
            return null;
        }

        Pattern pattern = null;
        if (StringUtil.isNotBlank(ignoreFailureType)) {
            pattern = Pattern.compile(ignoreFailureType);
        }
        List<String> urlList = new ArrayList<String>();
        for (FailureUrl failureUrl : list) {
            if (pattern != null) {
                if (!pattern.matcher(failureUrl.getUrl()).matches()) {
                    urlList.add(failureUrl.getUrl());
                }
            } else {
                urlList.add(failureUrl.getUrl());
            }
        }
        return urlList;
    }

    public void deleteByFileConfigId(Long id) {
        FailureUrlCB cb = new FailureUrlCB();
        cb.query().setFileConfigId_Equal(id);
        failureUrlBhv.varyingQueryDelete(cb,
                new DeleteOption<FailureUrlCB>().allowNonQueryDelete());
    }

    public void deleteByWebConfigId(Long id) {
        FailureUrlCB cb = new FailureUrlCB();
        cb.query().setWebConfigId_Equal(id);
        failureUrlBhv.varyingQueryDelete(cb,
                new DeleteOption<FailureUrlCB>().allowNonQueryDelete());
    }
}
