package jp.sf.fess.solr.policy;

import jp.sf.fess.Constants;
import jp.sf.fess.helper.MailHelper;

import org.codelibs.core.util.DynamicProperties;
import org.codelibs.solr.lib.policy.QueryType;
import org.codelibs.solr.lib.policy.impl.StatusPolicyImpl;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FessStatusPolicy extends StatusPolicyImpl {
    private static final Logger logger = LoggerFactory
            .getLogger(FessStatusPolicy.class);

    public String activateSubject = "[FESS] Solr status changed";

    public String deactivateSubject = "[FESS] Solr status changed";

    @Override
    public void activate(final QueryType queryType, final String serverName) {
        final String statusValue = solrGroupProperties
                .getProperty(getStatusKey(serverName));
        final String indexValue = solrGroupProperties
                .getProperty(getIndexKey(serverName));

        StringBuilder buf = null;

        if (StringUtil.isNotBlank(statusValue) && INACTIVE.equals(statusValue)) {
            // status: INACTIVE -> ACTIVE
            buf = new StringBuilder(100);
            buf.append("Status: INACTIVE -> ACTIVE\n");
        }

        switch (queryType) {
        case COMMIT:
        case OPTIMIZE:
        case ROLLBACK:
            if (StringUtil.isNotBlank(indexValue)
                    && UNFINISHED.equals(indexValue)) {
                // index: UNFINISHED -> COMPLETED             
                if (buf == null) {
                    buf = new StringBuilder(100);
                }
                buf.append("Index: UNFINISHED -> COMPLETED \n");
            }
            break;
        case ADD:
        case DELETE:
            if (StringUtil.isNotBlank(indexValue)
                    && UNFINISHED.equals(indexValue)) {
                // index: UNFINISHED -> READY             
                if (buf == null) {
                    buf = new StringBuilder(100);
                }
                buf.append("Index: UNFINISHED -> READY\n");
            }
            break;
        case PING:
        case QUERY:
        case REQUEST:
        default:
            break;
        }

        if (buf != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Server: " + serverName + "\n" + buf.toString());
            }

            send(activateSubject,
                    "Server: " + serverName + "\n" + buf.toString());
        }

        super.activate(queryType, serverName);
    }

    @Override
    public void deactivate(final QueryType queryType, final String serverName) {
        final String statusValue = solrGroupProperties
                .getProperty(getStatusKey(serverName));
        final String indexValue = solrGroupProperties
                .getProperty(getIndexKey(serverName));

        StringBuilder buf = null;

        if (StringUtil.isNotBlank(statusValue) && ACTIVE.equals(statusValue)) {
            // status: ACTIVE -> INACTIVE
            buf = new StringBuilder(100);
            buf.append("Status: ACTIVE -> INACTIVE\n");
        }

        switch (queryType) {
        case COMMIT:
        case OPTIMIZE:
        case ROLLBACK:
        case ADD:
        case DELETE:
            if (StringUtil.isNotBlank(indexValue)
                    && !UNFINISHED.equals(indexValue)) {
                // index: READY/COMPLETED -> UNFINISHED
                if (buf == null) {
                    buf = new StringBuilder(100);
                }
                buf.append("Index: ").append(indexValue)
                        .append(" -> UNFINISHED\n");
            }
            break;
        case PING:
        case QUERY:
        case REQUEST:
        default:
            break;
        }

        if (buf != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Server: " + serverName + "\n" + buf.toString());
            }

            send(deactivateSubject,
                    "Server: " + serverName + "\n" + buf.toString());
        }

        super.deactivate(queryType, serverName);
    }

    protected void send(final String subject, final String body) {
        final DynamicProperties crawlerProperties = SingletonS2Container
                .getComponent("crawlerProperties");
        if (crawlerProperties == null) {
            logger.info("crawlerProperties is not found.");
            return;
        }

        final MailHelper mailHelper = SingletonS2Container
                .getComponent("mailHelper");
        if (mailHelper == null) {
            logger.info("mailHelper is not found.");
            return;
        }

        final String toStrs = (String) crawlerProperties
                .get(Constants.NOTIFICATION_TO_PROPERTY);
        if (StringUtil.isNotBlank(toStrs)) {
            final String[] toAddresses = toStrs.split(",");

            try {
                mailHelper.send(toAddresses, subject, body);
            } catch (final Exception e) {
                logger.warn("Failed to send the notification.", e);
            }
        }
    }
}
