/*
 * Copyright 2009-2012 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.action.admin;

import java.io.File;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.crud.util.SAStrutsUtil;
import jp.sf.fess.form.admin.DesignForm;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.util.FessProperties;

import org.seasar.framework.util.FileUtil;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.exception.ActionMessagesException;
import org.seasar.struts.util.ServletContextUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DesignAction implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(DesignAction.class);

    @ActionForm
    @Resource
    protected DesignForm designForm;

    @Resource
    protected FessProperties crawlerProperties;

    public boolean editable = true;

    @Resource
    protected SystemHelper systemHelper;

    public String getHelpLink() {
        return systemHelper.getHelpLink("design");
    }

    @Execute(validator = false, input = "index.jsp")
    public String index() {
        checkEditorStatus();
        return "index.jsp";
    }

    @Execute(validator = false)
    public String back() {
        checkEditorStatus();
        return "index.jsp";
    }

    private boolean checkFileType(final String fileName) {
        if (fileName == null) {
            return false;
        }
        final String lFileName = fileName.toLowerCase();
        if (lFileName.endsWith(".jpg") || lFileName.endsWith(".jpeg")
                || lFileName.endsWith(".gif") || lFileName.endsWith(".png")) {
            return true;
        }
        return false;
    }

    @Execute(validator = true, input = "index")
    public String upload() {
        checkEditorStatus();
        final String uploadedFileName = designForm.designImageFile
                .getFileName();
        if (!checkFileType(uploadedFileName)) {
            throw new ActionMessagesException(
                    "errors.design_image_file_is_unsupported_type");
        }

        String fileName = designForm.designImageFileName;
        if (StringUtil.isBlank(fileName)) {
            fileName = uploadedFileName;
            if (StringUtil.isBlank(fileName)) {
                throw new ActionMessagesException(
                        "errors.design_image_file_name_is_not_found");
            }
        }

        // normalize filename
        try {
            int pos = fileName.indexOf('/');
            if (pos >= 0) {
                fileName = fileName.substring(pos + 1);
            }
            pos = fileName.indexOf('\\');
            if (pos >= 0) {
                fileName = fileName.substring(pos + 1);
            }
        } catch (final Exception e) {
            throw new ActionMessagesException(
                    "errors.design_image_file_name_is_invalid");
        }
        if (!checkFileType(fileName)) {
            throw new ActionMessagesException(
                    "errors.design_image_file_name_is_unsupported_type");
        }

        final File imageFile = new File(ServletContextUtil.getServletContext()
                .getRealPath("/images/" + fileName));

        try {
            FileUtil.write(imageFile.getAbsolutePath(),
                    designForm.designImageFile.getFileData());
            SAStrutsUtil.addSessionMessage("success.upload_design_image_file",
                    fileName);
            return "index?redirect=true";
        } catch (final Exception e) {
            logger.error("Failed to write an image file.", e);
            throw new ActionMessagesException(
                    "errors.failed_to_write_design_image_file");
        }

    }

    @Execute(validator = false, input = "index.jsp")
    public String edit() {
        checkEditorStatus();
        final String jspType = "view";
        final File jspFile = getJspFile(jspType);

        try {
            designForm.content = new String(FileUtil.getBytes(jspFile),
                    Constants.UTF_8);
        } catch (final UnsupportedEncodingException e) {
            throw new FessSystemException("Invalid encoding");
        }

        return "edit.jsp";
    }

    @Execute(validator = false, input = "index.jsp")
    public String editAsUseDefault() {
        checkEditorStatus();
        final String jspType = "orig/view";
        final File jspFile = getJspFile(jspType);

        try {
            designForm.content = new String(FileUtil.getBytes(jspFile),
                    Constants.UTF_8);
        } catch (final UnsupportedEncodingException e) {
            throw new FessSystemException("Invalid encoding");
        }

        return "edit.jsp";
    }

    @Execute(validator = false, input = "index.jsp")
    public String update() {
        checkEditorStatus();
        final String jspType = "view";
        final File jspFile = getJspFile(jspType);

        if (designForm.content == null) {
            designForm.content = "";
        }

        try {
            FileUtil.write(jspFile.getAbsolutePath(),
                    designForm.content.getBytes(Constants.UTF_8));
            SAStrutsUtil.addSessionMessage("success.update_design_jsp_file",
                    systemHelper.getDesignJspFileName(designForm.fileName));
            return "index?redirect=true";
        } catch (final Exception e) {
            logger.error("Failed to update " + designForm.fileName, e);
            throw new ActionMessagesException(
                    "errors.failed_to_update_jsp_file");
        }
    }

    private File getJspFile(final String jspType) {
        final String jspFileName = systemHelper
                .getDesignJspFileName(designForm.fileName);
        if (jspFileName == null) {
            throw new ActionMessagesException(
                    "errors.invalid_design_jsp_file_name");
        }
        final File jspFile = new File(ServletContextUtil.getServletContext()
                .getRealPath("/WEB-INF/" + jspType + "/" + jspFileName));
        if (jspFile == null || !jspFile.exists()) {
            throw new ActionMessagesException(
                    "errors.design_jsp_file_does_not_exist");
        }
        return jspFile;
    }

    private void checkEditorStatus() {
        if (Constants.FALSE.equals(crawlerProperties.getProperty(
                Constants.WEB_DESIGN_EDITOR_PROPERTY, Constants.TRUE))) {
            editable = false;
            throw new ActionMessagesException("errors.design_editor_disabled");
        }
    }
}