/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.solr;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.request.JavaBinUpdateRequestCodec;
import org.apache.solr.client.solrj.request.RequestWriter;
import org.apache.solr.client.solrj.request.UpdateRequest;
import org.apache.solr.common.util.ContentStream;

public class BinaryRequestWriter extends RequestWriter {

    public Collection<ContentStream> getContentStreams(SolrRequest req)
            throws IOException {
        if (req instanceof UpdateRequest) {
            UpdateRequest updateRequest = (UpdateRequest) req;
            if (isNull(updateRequest.getDocuments())
                    && isNull(updateRequest.getDeleteById())
                    && isNull(updateRequest.getDeleteQuery())
                    && (updateRequest.getDocIterator() == null)) {
                return null;
            }
            List<ContentStream> l = new ArrayList<ContentStream>();
            l.add(new DelegateContentStream(updateRequest));
            return l;
        } else {
            return super.getContentStreams(req);
        }

    }

    public String getUpdateContentType() {
        return "application/octet-stream";
    }

    public ContentStream getContentStream(final UpdateRequest request)
            throws IOException {
        final BAOS baos = new BAOS();
        new JavaBinUpdateRequestCodec().marshal(request, baos);
        return new ContentStream() {
            public String getName() {
                return null;
            }

            public String getSourceInfo() {
                return "javabin";
            }

            public String getContentType() {
                return "application/octet-stream";
            }

            public Long getSize() // size if we know it, otherwise null
            {
                return Long.valueOf((long) baos.size());
            }

            public InputStream getStream() throws IOException {
                return new ByteArrayInputStream(baos.getbuf(), 0, baos.size());
            }

            public Reader getReader() throws IOException {
                throw new FessSolrException(
                        "No reader available . this is a binarystream");
            }
        };
    }

    public void write(SolrRequest request, OutputStream os) throws IOException {
        if (request instanceof UpdateRequest) {
            UpdateRequest updateRequest = (UpdateRequest) request;
            new JavaBinUpdateRequestCodec().marshal(updateRequest, os);
        }

    }/*
     * A hack to get access to the protected internal buffer and avoid an additional copy 
     */

    class BAOS extends ByteArrayOutputStream {
        byte[] getbuf() {
            return super.buf;
        }
    }

    public String getPath(SolrRequest req) {
        if (req instanceof UpdateRequest) {
            return "/update/javabin";
        } else {
            return req.getPath();
        }
    }

    public class DelegateContentStream implements ContentStream {
        ContentStream contentStream = null;

        UpdateRequest req = null;

        public DelegateContentStream(UpdateRequest req) {
            this.req = req;
        }

        private ContentStream getDelegate() {
            if (contentStream == null) {
                try {
                    contentStream = getContentStream(req);
                } catch (IOException e) {
                    throw new FessSolrException(
                            "Unable to write xml into a stream", e);
                }
            }
            return contentStream;
        }

        public String getName() {
            return getDelegate().getName();
        }

        public String getSourceInfo() {
            return getDelegate().getSourceInfo();
        }

        public String getContentType() {
            return getUpdateContentType();
        }

        public Long getSize() {
            return getDelegate().getSize();
        }

        public InputStream getStream() throws IOException {
            return getDelegate().getStream();
        }

        public Reader getReader() throws IOException {
            return getDelegate().getReader();
        }

        public void writeTo(OutputStream os) throws IOException {
            write(req, os);

        }
    }
}
