﻿// In the original BSD license, both occurrences of the phrase "COPYRIGHT HOLDERS AND CONTRIBUTORS"
// in the disclaimer read "REGENTS AND CONTRIBUTORS".
//
// Here is the license template:
//
// Copyright (c) 2010, Masanori Usami
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided
// that the following conditions are met:
//
//  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
//  * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
// BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
// OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.using System;

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;

namespace EAAddinDSM
{
    /// <summary>
    /// MVCモデルにおけるView相当のクラス
    /// .NET FrameworkのFromを継承
    /// </summary>
    public partial class DSMDialog : Form, Observer
    {
        private EventArgs resizeEA = null;
        private DSMModel dsm_;
        private DSMController controller_;

        /// <summary>
        /// コンストラクタ
        /// InitializeComponentの呼び出しはForm継承の義務
        /// </summary>
        public DSMDialog()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 初期化処理（ダイアグラムDSM用）
        /// </summary>
        /// <param name="repository">EAリポジトリ</param>
        /// <param name="diagram">DSM化対象のダイアグラム</param>
        /// <param name="type">クラス/オブジェクト</param>
        /// <returns>true:初期化成功 false:初期化失敗</returns>
        public bool initialize(EA.Repository repository, EA.Diagram diagram, string type)
        {
            try
            {
                dsm_ = new DSMModel(repository, diagram, type);
                if (dsm_.attach(this))
                {
                    setDialogFormat();
                    controller_ = new DSMController(ref dsm_);
                    ProgressDialog prog = new ProgressDialog(ref controller_, ref dsm_);
                    prog.initialize();
                    prog.ShowDialog();
                    update();
                    return true;
                }
                return false;
            }
            catch (System.NotSupportedException e)
            {
                throw e;
            }
        }

        /// <summary>
        /// 初期化処理（パッケージDSM用）
        /// </summary>
        /// <param name="repository">EAリポジトリ</param>
        /// <param name="package">DSM化対象のパッケージ</param>
        /// <param name="type">クラス/オブジェクト</param>
        /// <returns>true:初期化成功 false:初期化失敗</returns>
        public bool initialize(EA.Repository repository, EA.Package package, string type)
        {
            try
            {
                dsm_ = new DSMModel(repository, package, type);
                if (dsm_.attach(this))
                {
                    setDialogFormat();
                    controller_ = new DSMController(ref dsm_);
                    ProgressDialog prog = new ProgressDialog(ref controller_, ref dsm_);
                    prog.initialize();
                    prog.ShowDialog();
                    update();
                    return true;
                }
                return false;
            }
            catch (System.NotSupportedException e)
            {
                throw e;
            }
        }

        /// <summary>
        /// ダイアログの表示用フォーマットを初期化する
        /// </summary>
        private void setDialogFormat()
        {
            grid_.AutoSize = true;
            grid_.SelectionMode = DataGridViewSelectionMode.CellSelect;
            grid_.ColumnHeadersVisible = false;
        }
        
        /// <summary>
        /// ダイアログリサイズ用のイベントハンドラ
        /// OnIdleイベントハンドラを追加
        /// </summary>
        /// <param name="e"></param>
        protected override void OnResize(EventArgs e)
        {
            if (resizeEA == null)
            {
                resizeEA = e;
                Application.Idle += new EventHandler(OnIdle);
            }
        }

        /// <summary>
        /// Idle用のイベントハンドラ
        /// DataViewGridの表示サイズをダイアログサイズに合わせて変更する
        /// </summary>
        /// <param name="s"></param>
        /// <param name="e"></param>
        private void OnIdle(object s, EventArgs e)
        {
            if (resizeEA != null)
            {
                grid_.Height = Size.Height - 58;
                grid_.Width = Size.Width - 8;

                base.OnResize(resizeEA);
                resizeEA = null;
                Application.Idle -= new EventHandler(OnIdle);
            }
        }

        /// <summary>
        /// DSMの再表示
        /// </summary>
        public void update()
        {
            int[,] dsm = dsm_.getDSM();
            ArrayList nameList = dsm_.getElementNameList();

            grid_.ColumnCount = nameList.Count + 1;
            grid_.RowCount = nameList.Count + 1;
            grid_.Columns[0].SortMode = DataGridViewColumnSortMode.NotSortable;
            grid_.Columns[0].DefaultCellStyle = grid_.ColumnHeadersDefaultCellStyle;
            grid_.Rows[0].DefaultCellStyle = grid_.ColumnHeadersDefaultCellStyle;
            for (int i = 1; i <= nameList.Count; i++)
            {
                grid_.Rows[i].HeaderCell.Value = (string)nameList[i - 1];
                grid_.Rows[i].Cells[0].Value = i.ToString();
                grid_.Rows[i].Frozen = true;
                grid_.Columns[i].SortMode = DataGridViewColumnSortMode.NotSortable;
                grid_.Rows[0].Cells[i].Value = i.ToString();
                for (int j = 0; j < nameList.Count; j++)
                {
                    string value;
                    if ((i - 1) == j)
                    {
                        value = "-";
                        grid_.Rows[i].Cells[j + 1].Style.BackColor = Color.DarkGray;
                    }
                    else
                    {
                        if (dsm[i - 1, j] != 0 && (i - 1) < j)
                        {
                            grid_.Rows[i].Cells[j + 1].Style.BackColor = Color.DarkOrange;
                        }
                        else if (dsm[i - 1, j] != 0 && (i - 1) > j)
                        {
                            grid_.Rows[i].Cells[j + 1].Style.BackColor = Color.DarkCyan;
                        }
                        else
                        {
                            grid_.Rows[i].Cells[j + 1].Style.BackColor = Color.White;
                        }
                        value = dsm[i - 1, j].ToString();
                    }
                    grid_.Rows[i].Cells[j + 1].Value = value;
                    grid_.Rows[i].Cells[j + 1].ReadOnly = true;
                }
            }
        }

        /// <summary>
        /// menu: partitioning
        /// partitioning処理の実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void menuItemPartitioning__Click(object sender, EventArgs e)
        {
            ProgressDialog prog = new ProgressDialog(ref controller_, ref dsm_);
            prog.partitioning();
            prog.ShowDialog();
            update();
        }

        /// <summary>
        /// menu: close
        /// ダイアログクローズ処理の実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void menuItemClose__Click(object sender, EventArgs e)
        {
            Close();
        }

        /// <summary>
        /// menu: save
        /// DSM保存処理の実行（CSVファイル）
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void menuItemSave__Click(object sender, EventArgs e)
        {
            string path = null;
            SaveFileDialog saveFileDialog = new SaveFileDialog();

            saveFileDialog.Filter = "txt files (*.csv)|*.txt|All files (*.*)|*.*";
            saveFileDialog.FilterIndex = 2;
            saveFileDialog.RestoreDirectory = true;

            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                path = saveFileDialog.FileName;
                controller_.save(path);
                MessageBox.Show("ファイル:" + path + "にDSMを出力しました。", "保存成功");
            }
            else
            {
                MessageBox.Show("保存先ファイルが指定されませんでした。\nDSMは保存されません。", "保存エラー");
            }
        }

        /// <summary>
        /// menu: tearing
        /// tearing処理の実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void menuItemTearing__Click(object sender, EventArgs e)
        {
            if (grid_.SelectedCells.Count == 1)
            {
                if ((grid_.SelectedCells[0].ColumnIndex >= 1) && (grid_.SelectedCells[0].RowIndex >= 1) &&
                    (grid_.SelectedCells[0].Value.ToString() == 1.ToString()))
                {
                    ProgressDialog prog = new ProgressDialog(ref controller_, ref dsm_);
                    prog.tearing(grid_.SelectedCells[0].RowIndex - 1, grid_.SelectedCells[0].ColumnIndex - 1);
                    prog.ShowDialog();
                    update();
                }
            }

            MessageBox.Show("セルを１つだけ選択して再度実行してください。", "Tearing実行エラー");
        }

        /// <summary>
        /// menu: undo
        /// UNDO処理の実行
        /// DSMを初期状態に戻す
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void menuItemUndo__Click(object sender, EventArgs e)
        {
            controller_.undo();
        }

    }
}
