/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package aipo.batch.mailaccount;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.resources.JetspeedResources;

import aipo.batch.utils.BatchUtils;

import com.aimluck.eip.cayenne.om.portlet.EipMMailAccount;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * <HR>
 * メールアカウントエクスポート
 * <p>
 * 
 * Aipoデータベースのメールアカウントを所定のフォルダにエクスポートする。<br>
 * <P>
 * <HR>
 * <P>
 */
public class ExportMailAccount {

  /** ロガー */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(ExportMailAccount.class.getName());

  /** プロパティファイル */
  private static final String PROPERTY_FILE =
    JetspeedResources.getString("aipo.conf", "")
      + File.separator
      + "Batch.properties";

  /** ファイルパス */
  private static final String EXPORT_FILE_PATH =
    (String) BatchUtils
      .getProperties(PROPERTY_FILE)
      .get("export.csv.file.path");

  /** ファイル名 */
  private static final String MAIL_ACCOUNT_FILE_NAME =
    (String) BatchUtils.getProperties(PROPERTY_FILE).get(
      "export.mailaccount.file.name");

  /** LinuxグループID */
  private static final String LINUX_GROUP_ID =
    (String) BatchUtils.getProperties(PROPERTY_FILE).get(
      "mailaccount.linux.user.id");

  /** ログインシェル */
  private static final String LINUX_LOGIN_SHELL =
    (String) BatchUtils.getProperties(PROPERTY_FILE).get(
      "mailaccount.linux.login.shell");

  /**
   * メールアカウントリスト作成
   * <p>
   * メールアカウント(EIP_M_MAIL_ACCOUNT)テーブルからメールアカウントを抽出し、<br>
   * CSVファイルに書き出す。<br>
   * 
   * @return boolean True(処理成功)、False(処理失敗)
   */
  public boolean exportMailAccount() throws Exception {
    BufferedWriter bw = null;
    int outCounter = 0;
    try {
      // 開始メッセージをログに出力
      logger.info("メールアカウントリスト作成処理開始");

      // メールアカウントを抽出する(システムメールアカウントは除外する)
      SelectQuery<EipMMailAccount> query =
        Database.query(EipMMailAccount.class);
      Expression exp1 =
        ExpressionFactory.noMatchExp(EipMMailAccount.USER_ID_PROPERTY, 1);
      query.setQualifier(exp1);
      query.orderAscending(EipMMailAccount.USER_ID_PROPERTY);
      List<EipMMailAccount> resultList = query.fetchList();

      File file = new File(EXPORT_FILE_PATH);
      if (!file.exists()) {
        logger.error("出力フォルダが見つかりませんでした。ファイルパス：["
          + EXPORT_FILE_PATH
          + File.separator
          + MAIL_ACCOUNT_FILE_NAME
          + "]");
        return false;
      }

      FileWriter writer =
        new FileWriter(EXPORT_FILE_PATH
          + File.separator
          + MAIL_ACCOUNT_FILE_NAME);
      bw = new BufferedWriter(writer);

      // CSVファイル出力
      for (EipMMailAccount record : resultList) {
        // POP3ユーザー名
        bw.write(record.getPop3userName());
        // LinuxグループID
        bw.write("," + LINUX_GROUP_ID);

        // POP3パスワード(復号化して出力)
        byte[] password = record.getPop3password();
        bw.write(","
          + (new String(ALMailUtils.getDecryptedMailAccountPasswd(password))));
        // ログインシェル
        bw.write("," + LINUX_LOGIN_SHELL);
        bw.newLine();
        bw.flush();
        outCounter++;
      }
      // 終了メッセージをログに出力
      logger.info("メールアカウントリスト作成処理終了　抽出件数：[" + outCounter + "]");
    } catch (Exception e) {
      logger.error("メールアカウントリスト作成に失敗しました。", e);
      return false;
    } finally {
      if (bw != null) {
        bw.close();
      }
    }
    return true;
  }

}
