/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2012 Aimluck,Inc.
 * http://www.aipo.com
 *
 * Copyright(C) 2012 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.schedule;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 月間スケジュール印刷の検索結果を管理するクラスです。
 * 
 */
public class SchedulePrintMonthlySelectData extends ScheduleMonthlySelectData {

  /** <code>printDate</code> 印刷日時 */
  private ALDateTimeField printDate;

  /** <code>target_user_alias_name</code> 表示対象のユーザ 表示名 */
  private String targetUserAliasName;

  /** 期間スケジュールの最大件数 */
  private static final int PRINT_MAX_TERM_SCHEDULE_COUNT = 10;

  /** 表示スケジュールの最大件数 */
  private static final int PRINT_MAX_SCHEDULE_COUNT = 10;

  /**
   * 初期処理
   * 
   * @param action
   *          アクションクラス
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @see com.aimluck.eip.common.ALAbstractSelectData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context) throws ALPageNotFoundException, ALDBErrorException {
    // スーパークラスのメソッドを呼び出す。
    super.init(action, rundata, context);

    // 印刷日時
    printDate = new ALDateTimeField("yyyy/MM/dd HH:mm");
    Calendar now = Calendar.getInstance();
    printDate.setValue(now.getTime());

    // 表示可能件数を変数に格納（印刷と共通で使用のため）
    maxTermSchedulueCount = PRINT_MAX_TERM_SCHEDULE_COUNT;
    maxSchedulueCount = PRINT_MAX_SCHEDULE_COUNT;

    // 表示できない期間スケジュールがあるかどうかのフラグリストをfalseで初期化する。
    // 日ごとの通常スケジュールの表示可能な件数を設定する。
    cannotDisplayFlagList = new ArrayList<List<Boolean>>();
    for (int weekIndex = 0; monthCon.getWeekList().size() > weekIndex; weekIndex++) {
      cannotDisplayFlagList.add(new ArrayList<Boolean>());
      int dayCount = monthCon.getWeekList().get(weekIndex).getDayList().size();
      for (int dayIndex = 0; dayCount > dayIndex; dayIndex++) {
        cannotDisplayFlagList.get(weekIndex).add(false);
        ScheduleDayContainer dayContainer = monthCon.getWeekList().get(weekIndex).getDayList().get(dayIndex);
        dayContainer.setLimitDisplayCount(maxTermSchedulueCount);
      }
    }
  }

  /**
   * 指定グループや指定ユーザをセッションに設定する．
   * 
   * @param rundata
   * @param context
   * @throws Exception
   * @throws ALDBErrorException
   */
  @Override
  protected void setupLists(RunData rundata, Context context) throws Exception {

    // スーパークラスのメソッドを呼び出す。
    super.setupLists(rundata, context);

    ALBaseUser targetUser = ALEipUtils.getBaseUser(Integer.parseInt(target_user_id));
    // ユーザー表示名を取得
    targetUserAliasName = targetUser.getDisplayName();
  }

  /**
   * スケジュール最大表示可能件数を取得します。
   * 
   * @return スケジュール最大表示可能件数
   */
  public int getMaxScheduleCount() {
    return PRINT_MAX_SCHEDULE_COUNT;
  }

  /**
   * 印刷日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getPrintDate() {
    return printDate;
  }

  /**
   * 印刷画面に表示するユーザ の表示名
   * 
   * @return
   */
  public String getTargetUserAliasName() {
    return targetUserAliasName;
  }

}
