/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2011 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.account;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.account.AvzMRole;
import com.aimluck.eip.cayenne.om.account.AvzMUserRole;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;

/**
 * 役割情報登録フォームデータクラスです。
 * 
 */
public class AccountRoleFormData extends ALAbstractFormData {
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AccountRoleFormData.class.getName());

  /** 役割名 */
  private ALStringField role_name;

  /**
   * 初期処理
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
  }

  /**
   * 画面フィールド初期化
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    role_name = new ALStringField();
    role_name.setFieldName("役割名");
    role_name.setTrim(true);
  }

  /**
   * バリデータ初期化
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    role_name.setNotNull(true);
    role_name.limitMaxLength(50);
  }

  /**
   * バリデーション
   * 
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    role_name.validate(msgList);
    return msgList.size() == 0;
  }

  /**
   * 役割編集画面表示処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    try {
      AvzMRole entity = AccountUtils.getAvzMRole(rundata, context);
      if (entity == null) {
        throw new Exception("指定された役割がデータベースにありません。");
      }

      role_name.setValue(entity.getRoleName());

    } catch (Exception ex) {
      String msg = "役割情報の編集画面の表示に失敗しました。";
      logger.error(msg, ex);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * 役割登録処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      Date now = Calendar.getInstance().getTime();

      // 新規オブジェクトモデル
      AvzMRole entity = Database.create(AvzMRole.class);
      entity.setRoleName(role_name.getValue());
      entity.setCreateDate(now);
      entity.setUpdateDate(now);
      Database.commit();

      // singletonオブジェクトのリフレッシュ
      ALEipManager.getInstance().reloadRole();

    } catch (Exception ex) {
      String msg = "役割情報の登録に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    }
    return true;
  }

  /**
   * 役割更新処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      AvzMRole entity = AccountUtils.getAvzMRole(rundata, context);
      if (entity == null) {
        throw new Exception("指定された役割がデータベースにありません。");
      }

      entity.setRoleName(role_name.getValue());
      entity.setUpdateDate(Calendar.getInstance().getTime());
      Database.commit();

      // singletonオブジェクトのリフレッシュ
      ALEipManager.getInstance().reloadRole();

    } catch (Exception ex) {
      String msg = "役割情報の更新に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    }
    return true;
  }

  /**
   * 役割削除処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      AvzMRole entity = AccountUtils.getAvzMRole(rundata, context);
      if (entity == null) {
        throw new Exception("指定された役割がデータベースにありません。");
      }

      List<AvzMUserRole> list = entity.getAvzMUserRoleArray();
      if (list != null && list.size() != 0) {
        Database.deleteAll(list);
      }

      Database.delete(entity);
      Database.commit();

      // singletonオブジェクトのリフレッシュ
      ALEipManager.getInstance().reloadRole();

    } catch (Exception ex) {
      String msg = "役割情報の削除に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    }
    return true;
  }

  /**
   * 役割名取得
   * 
   * @return 役割名
   */
  public ALStringField getRoleName() {
    return role_name;
  }

  /**
   * ※未使用
   * 
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return null;
  }
}
