package jp.cssj.android;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipFile;

import jp.cssj.cr.db.DatabaseHelper;
import jp.cssj.cr.epub.AssetArchiveFile;
import jp.cssj.cr.epub.BookCallback;
import jp.cssj.cr.epub.Cache;
import jp.cssj.cr.epub.CopperThread;
import jp.cssj.cr.epub.Setting;
import jp.cssj.cr.epub.SpineItemAdapter;
import jp.cssj.cr.epub.TocAdapter;
import jp.cssj.cr.gc.OpenTypeFontSourceManager;
import jp.cssj.cr.gc.RecorderGC;
import jp.cssj.homare.ua.AbortException;
import jp.cssj.print.epub.ArchiveFile;
import jp.cssj.print.epub.Container;
import jp.cssj.print.epub.Contents;
import jp.cssj.print.epub.Contents.Item;
import jp.cssj.print.epub.EPubFile;
import jp.cssj.print.epub.Toc;
import jp.cssj.print.epub.ZipArchiveFile;
import jp.cssj.resolver.cache.CachedSourceResolver;
import jp.cssj.resolver.helpers.MetaSourceImpl;
import jp.cssj.sakae.font.FontManagerImpl;
import jp.cssj.sakae.gc.font.FontManager;
import android.app.ActivityGroup;
import android.content.ContentValues;
import android.content.Intent;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.provider.MediaStore;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.Window;
import android.view.WindowManager;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.FrameLayout;
import android.widget.ListView;
import android.widget.RadioButton;
import android.widget.SeekBar;
import android.widget.TextView;

public class BookActivity extends ActivityGroup {
	private static final int MIN_TEXT_SIZE = 30;
	// public static TextView debug;

	DatabaseHelper dbh;
	private FrameLayout main;
	private TextView position;
	protected SeekBar progressBar;
	protected int goToPage = -1;

	protected Contents contents;
	protected Toc toc;
	public ArchiveFile archive;
	protected String id = "data";
	public Item item;

	protected Cache cache = new Cache();

	private CopperThread copperThread;
	private BookCallback callback;

	private RadioButton[] directions = new RadioButton[3];
	private CheckBox reverse;
	private RadioButton[] columns = new RadioButton[3];
	private SeekBar fontSize;
	private TextView fontSizeVal;
	private SeekBar margin;
	private TextView marginVal;
	public Setting setting = new Setting();
	public float xdpi, ydpi, widthIn, heightIn;
	private FontManagerImpl fm;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		DisplayMetrics metrics = new DisplayMetrics();
		Display display = this.getWindowManager().getDefaultDisplay();
		display.getMetrics(metrics);
		this.ydpi = metrics.scaledDensity * 160f;
		this.xdpi = metrics.xdpi * this.ydpi / metrics.ydpi;
		this.widthIn = (float) metrics.widthPixels / this.xdpi;
		this.heightIn = (float) metrics.heightPixels / this.ydpi;
		this.heightIn -= 24f / 160f; // 上下の24dip分縮める
		// AD1*
		this.heightIn -= 50f / 160f; // 広告(50dip)分縮める
		// *AD1

		Uri data = this.getIntent().getData();
		if (data != null) {
			File file;
			if (data.getScheme().equals("content")) {
				Cursor c = getContentResolver().query(data, null,
						null, null, null);
				c.moveToFirst();
				String filename = c.getString(c
						.getColumnIndex(MediaStore.MediaColumns.DATA));
				file = new File(filename);
			} else {
				file = new File(data.getPath());
			}
			this.id = file.getAbsolutePath();
			try {
				this.archive = new ZipArchiveFile(new ZipFile(file));
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		} else {
			this.id = "data";
			this.archive = new AssetArchiveFile(this.getAssets(), this.id);
		}

		this.getWindow().addFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
		this.requestWindowFeature(Window.FEATURE_NO_TITLE);

		int itemIndex = 0, page = 0;
		this.dbh = new DatabaseHelper(this);
		SQLiteDatabase db = this.dbh.getWritableDatabase();
		try {
			{
				Cursor c = db.rawQuery(
						"SELECT direction,reverse,columns,font_size,margin FROM setting "
								+ "WHERE file=?", new String[] { this.id });
				try {
					c.moveToFirst();
					if (!c.isAfterLast()) {
						this.setting.direction = (byte) c.getInt(0);
						this.setting.reverse = c.getInt(1) == 1;
						this.setting.columns = c.getInt(2);
						this.setting.fontSize = c.getInt(3);
						this.setting.margin = c.getInt(4);
					} else {
						ContentValues values = new ContentValues();
						values.put("file", this.id);
						values.put("direction", this.setting.direction);
						values.put("reverse", this.setting.reverse ? 1 : 0);
						values.put("columns", this.setting.columns);
						values.put("font_size", this.setting.fontSize);
						values.put("margin", this.setting.margin);
						db.insert("setting", null, values);
					}
				} finally {
					c.close();
				}
			}
			{
				Cursor c = db.rawQuery(
						"SELECT item, page FROM books WHERE file=?",
						new String[] { this.id });
				try {
					c.moveToFirst();
					if (!c.isAfterLast()) {
						ContentValues values = new ContentValues();
						values.put("last_accessed", System.currentTimeMillis());
						db.update("books", values, "file=?",
								new String[] { this.id });
						itemIndex = c.getInt(0);
						page = c.getInt(1);
					} else {
						ContentValues values = new ContentValues();
						values.put("file", this.id);
						values.put("last_accessed", System.currentTimeMillis());
						values.put("item", 0);
						values.put("page", 0);
						db.insert("books", null, values);
					}
				} finally {
					c.close();
				}
			}
		} finally {
			db.close();
		}

		setContentView(R.layout.book);
		this.main = (FrameLayout) findViewById(R.id.main);

		// About
		Button about = (Button) this.findViewById(R.id.about_fonts);
		about.setOnClickListener(new OnClickListener() {
			public void onClick(View view) {
				Intent intent = new Intent(BookActivity.this,
						AboutActivity.class);
				startActivity(intent);
			}
		});

		// スタイル
		this.columns[0] = (RadioButton) findViewById(R.id.columns_1);
		this.columns[1] = (RadioButton) findViewById(R.id.columns_2);
		this.columns[2] = (RadioButton) findViewById(R.id.columns_3);

		this.fontSize = (SeekBar) findViewById(R.id.fontSize);
		// debug =
		this.fontSizeVal = (TextView) findViewById(R.id.fontSizeVal);
		fontSizeVal
				.setText((this.fontSize.getProgress() + MIN_TEXT_SIZE) + "%");
		this.fontSize
				.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
					public void onProgressChanged(SeekBar seekBar,
							int progress, boolean fromTouch) {
						fontSizeVal.setText((progress + MIN_TEXT_SIZE) + "%");
					}

					public void onStartTrackingTouch(SeekBar seekBar) {
						fontSizeVal.setText((seekBar.getProgress() + MIN_TEXT_SIZE)
								+ "%");
					}

					public void onStopTrackingTouch(SeekBar seekBar) {
						fontSizeVal.setText((seekBar.getProgress() + MIN_TEXT_SIZE)
								+ "%");
					}
				});

		// マージン
		this.margin = (SeekBar) findViewById(R.id.margin);
		this.marginVal = (TextView) findViewById(R.id.marginVal);
		this.marginVal.setText((this.margin.getProgress() / 10f) + "mm");
		this.margin
				.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
					public void onProgressChanged(SeekBar seekBar,
							int progress, boolean fromTouch) {
						marginVal.setText((progress / 10f) + "mm");
					}

					public void onStartTrackingTouch(SeekBar seekBar) {
						marginVal.setText((seekBar.getProgress() / 10f) + "mm");
					}

					public void onStopTrackingTouch(SeekBar seekBar) {
						marginVal.setText((seekBar.getProgress() / 10f) + "mm");
					}
				});

		// 書字方向
		this.directions[0] = (RadioButton) findViewById(R.id.direction_keep);
		this.directions[1] = (RadioButton) findViewById(R.id.direction_h);
		this.directions[2] = (RadioButton) findViewById(R.id.direction_v);

		// 白黒反転
		this.reverse = (CheckBox) findViewById(R.id.reverse);

		this.restoreSetting();

		// 進行状況
		this.progressBar = (SeekBar) findViewById(R.id.progressBar);
		this.position = (TextView) findViewById(R.id.position);
		this.progressBar
				.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
					public void onProgressChanged(SeekBar seekBar,
							int progress, boolean fromTouch) {
						// nop
					}

					public void onStartTrackingTouch(SeekBar seekBar) {
						// nop
					}

					public void onStopTrackingTouch(SeekBar seekBar) {
						updateProgress();
					}
				});

		// 目次
		String appName = this.getString(R.string.app_name);
		TocAdapter tocAdapter;
		SpineItemAdapter spineItemAdapter;
		try {
			EPubFile epub = new EPubFile(this.archive);
			Container cr = epub.readContainer();
			this.contents = epub.readContents(cr.rootfiles[0]);
			this.toc = epub.readToc(this.contents);
			this.setTitle(appName + " > " + this.toc.docTitle);
			tocAdapter = new TocAdapter(this, R.layout.spine_item, epub,
					this.toc);
			spineItemAdapter = new SpineItemAdapter(this, R.layout.spine_item,
					epub, this.contents);
			for (int i = 0; i < this.contents.spine.length; ++i) {
				this.cache.addResult(null);
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		TextView title = (TextView) findViewById(R.id.bookTitle);
		title.setText(this.toc.docTitle);

		ListView tocList = (ListView) findViewById(R.id.toc);
		tocList.setAdapter(tocAdapter);
		tocList.setOnItemClickListener(new OnItemClickListener() {
			public void onItemClick(AdapterView<?> adapter, View view, int pos,
					long id) {
				if (toc.navPoints[pos].item == null) {
					return;
				}
				goTo(toc.navPoints[pos].item, 0);
			}
		});

		ListView spineList = (ListView) findViewById(R.id.spine);
		spineList.setAdapter(spineItemAdapter);
		spineList.setOnItemClickListener(new OnItemClickListener() {
			public void onItemClick(AdapterView<?> adapter, View view, int pos,
					long id) {
				goTo(contents.spine[pos], 0);
			}
		});

		if (itemIndex >= this.contents.spine.length) {
			itemIndex = this.contents.spine.length - 1;
		}
		this.goTo(this.contents.spine[itemIndex], page);

		// AD2*
		final com.google.ads.AdRequest request = new com.google.ads.AdRequest();
		final com.google.ads.AdView adView = (com.google.ads.AdView) this
				.findViewById(R.id.adView2);
		adView.setAdListener(new jp.cssj.cr.epub.DefaultAdListener(request));
		adView.loadAd(request);
		// *AD2
	}

	private boolean applySetting() {
		boolean reverse = this.reverse.isChecked();

		int columns = this.setting.columns;
		for (int i = 0; i < this.columns.length; ++i) {
			if (this.columns[i].isChecked()) {
				columns = i + 1;
				break;
			}
		}

		byte direction = this.setting.direction;
		for (byte i = 0; i < this.directions.length; ++i) {
			if (this.directions[i].isChecked()) {
				direction = i;
				break;
			}
		}

		int fontSize = this.fontSize.getProgress() + MIN_TEXT_SIZE;

		int margin = this.margin.getProgress() * 10;

		if (direction == this.setting.direction
				&& reverse == this.setting.reverse
				&& columns == this.setting.columns
				&& fontSize == this.setting.fontSize
				&& margin == this.setting.margin) {
			return false;
		}
		this.setting.direction = direction;
		this.setting.reverse = reverse;
		this.setting.columns = columns;
		this.setting.fontSize = fontSize;
		this.setting.margin = margin;
		for (int i = 0; i < cache.getResultsCount(); ++i) {
			cache.setResult(i, null);
		}
		this.saveSettings();
		return true;
	}

	private void restoreSetting() {
		this.columns[this.setting.columns - 1].setChecked(true);
		this.fontSize.setProgress(this.setting.fontSize - MIN_TEXT_SIZE);
		this.directions[this.setting.direction].setChecked(true);
		this.reverse.setChecked(this.setting.reverse);
		this.margin.setProgress(this.setting.margin / 10);
	}

	protected void saveSettings() {
		SQLiteDatabase db = this.dbh.getWritableDatabase();
		try {
			ContentValues values = new ContentValues();
			values.put("direction", this.setting.direction);
			values.put("reverse", this.setting.reverse ? 1 : 0);
			values.put("columns", this.setting.columns);
			values.put("font_size", this.setting.fontSize);
			values.put("margin", this.setting.margin);
			db.update("setting", values, "file=?", new String[] { this.id });
		} finally {
			db.close();
		}
	}

	protected void savePosition() {
		SQLiteDatabase db = this.dbh.getWritableDatabase();
		try {
			ContentValues values = new ContentValues();
			values.put("item", this.item.index);
			values.put("page", this.getPage());
			db.update("books", values, "file=?", new String[] { this.id });
		} finally {
			db.close();
		}
	}

	@Override
	protected void onResume() {
		super.onResume();
		updateProgress(BookCallback.PAGE_CHANGED);
	}

	protected synchronized int getPage() {
		return this.progressBar.getProgress();
	}

	protected synchronized void goTo(Item item, int page) {
		boolean pageCountChanged = false;
		if (this.applySetting()) {
			pageCountChanged = true;
		}
		if (this.item != item) {
			this.item = item;
			pageCountChanged = true;
		}
		if (pageCountChanged) {
			// レイアウト、アイテムの変更があればスレッド停止
			this.abort();
			this.pageCountChanged();
		}
		this.goToPage = page;
		this.start();
	}

	protected synchronized void pageCountChanged() {
		this.progressBar.setMax(this.getPageCount());
		this.updateProgress();
	}

	protected synchronized int getPageCount() {
		List<RecorderGC> pages = this.cache.getResult(this.item.index);
		int count;
		if (pages == null || pages.isEmpty()) {
			count = 0;
		} else if (pages.get(pages.size() - 1) == null) {
			count = pages.size() - 1;
		} else {
			count = pages.size();
		}
		return count;
	}

	protected void sendFile(CachedSourceResolver cached, String asset,
			String uri) throws IOException {
		File file = cached.putFile(new MetaSourceImpl(URI.create("cached:"
				+ uri)));
		OutputStream out = new FileOutputStream(file);
		try {
			InputStream in = this.getAssets().open(asset);
			try {
				byte[] buff = new byte[1024];
				for (int len = in.read(buff); len != -1; len = in.read(buff)) {
					out.write(buff, 0, len);
				}
			} finally {
				in.close();
			}
		} finally {
			out.close();
		}
	}

	public boolean dispatchKeyEvent(KeyEvent event) {
		if (this.panelView == null) {
			return super.dispatchKeyEvent(event);
		}
		if (event.getKeyCode() == KeyEvent.KEYCODE_BACK
				&& event.getAction() == KeyEvent.ACTION_DOWN) {
			// 戻るボタン
			if (this.main.getChildCount() == 1) {
				// 表示
				if (this.applySetting()) {
					this.goTo(this.item, 0);
				} else {
					this.main.addView(this.panelView);
					updateProgress(BookCallback.PAGE_CHANGED);
				}
				return false;
			}
			if (this.main.getChildCount() == 2 && this.panelActivity != null) {
				if (!this.panelActivity.back()) {
					return false;
				}
			}
		}
		return super.dispatchKeyEvent(event);
	}

	Handler handler = new Handler();

	protected void updateProgress() {
		final String text = this.getProgress();
		this.handler.post(new Runnable() {
			public void run() {
				position.setText(text);
			}
		});
	}

	protected String getProgress() {
		String item = this.getString(R.string.item);
		String page = this.getString(R.string.page);
		String pageNumber;
		if (this.progressBar.getMax() == 0) {
			if (this.isBusy()) {
				pageNumber = this.getString(R.string.loading);
			} else {
				pageNumber = "-";
			}
		} else {
			pageNumber = (this.progressBar.getProgress() + 1) + "/"
					+ this.progressBar.getMax();
			if (this.isBusy()) {
				pageNumber += " - " + this.getString(R.string.loading);
			}
		}
		String itemNumber = (this.item.index + 1) + "/"
				+ this.contents.spine.length;
		String text = item + ": " + itemNumber + " " + page + ": " + pageNumber;
		return text;
	}

	private static final int MENU_ID_ITEMS = Menu.FIRST;
	private static final int MENU_ID_TOC = Menu.FIRST + 1;
	private static final int MENU_ID_STYLE = Menu.FIRST + 2;
	private List<View> views = new ArrayList<View>();

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		boolean ret = super.onCreateOptionsMenu(menu);

		// スタイル
		menu.add(0, MENU_ID_STYLE, 0, this.getString(R.string.style)).setIcon(
				R.drawable.style);
		Button button = (Button) this.findViewById(R.id.apply_button);
		button.setOnClickListener(new OnClickListener() {
			public void onClick(View view) {
				if (applySetting()) {
					goTo(item, 0);
				}
			}
		});

		// 目次
		menu.add(0, MENU_ID_TOC, 0, this.getString(R.string.toc)).setIcon(
				R.drawable.toc);

		// ITEMS*
		// 内容
		menu.add(0, MENU_ID_ITEMS, 0, this.getString(R.string.items)).setIcon(
				R.drawable.items);
		// *ITEMS

		FrameLayout frame = (FrameLayout) this.findViewById(R.id.content);
		for (int i = 0; i < frame.getChildCount(); ++i) {
			this.views.add(frame.getChildAt(i));
		}
		frame.removeAllViews();

		return ret;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		FrameLayout frame = (FrameLayout) this.findViewById(R.id.content);
		frame.removeAllViews();
		switch (item.getItemId()) {
		case MENU_ID_ITEMS:
			frame.addView(this.views.get(0));
			break;
		case MENU_ID_TOC:
			frame.addView(this.views.get(1));
			break;
		case MENU_ID_STYLE:
			frame.addView(this.views.get(2));
			break;
		}
		if (this.main.getChildCount() == 2) {
			this.main.removeView(this.panelView);
		}
		return true;
	}

	private View panelView;
	private PanelActivity panelActivity;

	public FontManager getFontManager() {
		if (this.fm == null) {
			Log.d(this.getClass().getName(), "Setup fonts.");
			this.fm = new FontManagerImpl(new OpenTypeFontSourceManager(this));
		}
		return this.fm;
	}

	protected boolean isBusy() {
		return this.copperThread != null && this.copperThread.isAlive();
	}

	protected synchronized void start() {
		List<RecorderGC> pages = this.cache.getResult(this.item.index);
		if (pages == null || pages.isEmpty()) {
			pages = new ArrayList<RecorderGC>();
			cache.setResult(this.item.index, pages);
		} else {
			if (this.goToPage >= pages.size()) {
				if (this.copperThread != null && this.copperThread.isAlive()) {
					return;
				}
				this.goToPage = pages.size() - 1;
			}
			this.progressBar.setProgress(this.goToPage);
			this.goToPage = -1;
			if (this.callback != null) {
				this.updateProgress(BookCallback.PAGE_CHANGED);
			}
			this.savePosition();
			return;
		}

		// パネル
		this.setting.width = this.widthIn * 25.4f;
		this.setting.height = this.heightIn * 25.4f;
		PanelActivity panel = (PanelActivity) this.getLocalActivityManager()
				.getActivity("PANEL");
		if (panel == null) {
			Intent intent = new Intent(BookActivity.this, PanelActivity.class);
			Window w = this.getLocalActivityManager().startActivity("PANEL",
					intent);
			panel = (PanelActivity) this.getLocalActivityManager().getActivity(
					"PANEL");
			this.panelView = w.getDecorView();
		}
		if (this.main.getChildCount() == 1) {
			this.main.addView(panelView);
		}
		this.panelActivity = panel;
		this.panelActivity.beginLoading();

		this.copperThread = new CopperThread(this, cache, pages) {
			@Override
			protected void newPage(int page) {
				pageCountChanged();
				updateProgress(page == 1 && getPage() == 0 ? BookCallback.PAGE_CHANGED
						: BookCallback.PAGE_COUNT_CHANGED);
				updateProgress();
			}

			protected void done() {
				updateProgress(BookCallback.PAGE_FINISHED);
			}

			@Override
			public void run() {
				updateProgress(BookCallback.PAGE_COUNT_CHANGED);
				updateProgress();
				try {
					super.run();
				} finally {
					updateProgress();
				}
			}
		};
		this.copperThread.setPriority(Thread.MIN_PRIORITY + 1);
		this.copperThread.start();
	}

	private void updateProgress(final int event) {
		if (this.callback != null) {
			this.handler.post(new Runnable() {
				public void run() {
					callback.updateProgress(event);
				}
			});
		}
	}

	protected synchronized boolean hasPreviousPage() {
		if (this.getPage() <= 0 && this.item.index <= 0) {
			return false;
		}
		return true;
	}

	protected synchronized boolean hasNextPage() {
		if (this.getPage() < this.getPageCount() - 1) {
			return true;
		}
		if (this.item.index >= this.contents.spine.length - 1) {
			return false;
		}
		List<RecorderGC> pages = this.cache.getResult(this.item.index);
		if (pages == null || pages.isEmpty()
				|| pages.get(pages.size() - 1) == null) {
			return false;
		}
		return true;
	}

	protected synchronized void nextPage() {
		int p = this.getPage();
		if (p < this.getPageCount() - 1) {
			this.goTo(this.item, p + 1);
			this.updateProgress(BookCallback.PAGE_CHANGED);
			return;
		}
		if (this.item.index >= this.contents.spine.length - 1) {
			return;
		}
		List<RecorderGC> pages = this.cache.getResult(this.item.index);
		if (pages == null || pages.isEmpty()
				|| pages.get(pages.size() - 1) == null) {
			return;
		}
		this.goTo(this.contents.spine[this.item.index + 1], 0);
	}

	protected synchronized void previousPage() {
		int p = this.getPage();
		if (p > 0) {
			// 前ページに戻る
			this.goTo(this.item, p - 1);
			this.updateProgress(BookCallback.PAGE_CHANGED);
			return;
		}
		if (this.item.index <= 0) {
			// 本の先頭では何もしない
			return;
		}
		// 前の章に戻る
		this.goTo(this.contents.spine[this.item.index - 1], Integer.MAX_VALUE);
	}

	protected synchronized RecorderGC getRecorderGC() {
		List<RecorderGC> pages = this.cache.getResult(this.item.index);
		if (pages == null || pages.isEmpty()) {
			return null;
		}
		return pages.get(this.getPage());
	}

	protected void setBookCallback(BookCallback callback) {
		this.callback = callback;
	}

	private Thread abortThread = null;

	protected synchronized boolean abort() {
		if (this.abortThread != null) {
			return true;
		}
		if (this.copperThread == null || !this.copperThread.isAlive()) {
			return false;
		}
		if (this.copperThread.ua == null) {
			return true;
		}
		this.copperThread.ua.abort(AbortException.ABORT_FORCE);
		this.abortThread = new Thread() {
			public void run() {
				try {
					copperThread.join();
				} catch (InterruptedException e) {
					// ignore
				} finally {
					abortThread = null;
				}
			}
		};
		this.abortThread.start();
		return true;
	}
}