

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <string.h>
#include <stdint.h>
#include <avr/wdt.h>

#include <avr/sleep.h>
#include "macro.h"
#include "cc1100.h"
#include "cc1100_globals.h"
#include "spi1.h"
#include "i2c.h"





#define EEPROM __attribute__((section(".eeprom"))) 
ctl_data EEPROM gctl_data;
reg_slave EEPROM greg_slave[100]; 
int	EEPROM	eprom_level[14];


const time_set_t time_set[4]={
							{0x09,0x59,0x57}, //ALARM_2SEC
							{0x09,0x59,0x00}, //ALARM_1MIN
							{0x09,0x58,0x00}, //ALARM_2MIN
							{0x09,0x55,0x00}, //ALARM_5MIN
							#if 0
							{0x09,0x57,0x00}, //ALARM_3MIN
							{0x09,0x56,0x00}, //ALARM_4MIN
							{0x09,0x55,0x00}, //ALARM_5MIN
							{0x09,0x30,0x00}, //ALARM_30MIN
							{0x09,0x00,0x00}, //ALARM_1hour
	 						{0x10,0X00,0x01}, //ALARM_24hour
							#endif 
							};

reg_t  reg_net_alarm[7]=
							{
								{0x0E,0x00},
								{0x0C,0x10},
								{0x0B,0x00},
								{0x02,0x00},
								{0x01,0x00},
								{0x00,0x00},
								{0x0E,0x40}
							};

reg_t reg_next_sch_alarm[9]=
{
	{0x0e,0x00},
	{0x08,0x00},
	{0x09,0x00},
	{0x0a,0x00},
	{0x03,0x00},
	{0x02,0x00},
	{0x01,0x00},
	{0x00,0x00},
	{0x0e,0x60}
};

u8 day_cnt[4]={ 0x40,0x20,0x08,0x04};


cc1101_client 	gClient;
u8 				get_user0_data=0;
u8 				uart_data;
reg_slave 		tSlave,tSlave2;
ctl_data  		tCtl;
u16				gTemperature;
char * 			pC;

u8 				emg_status = 0x00;
u8				setup_mode = SETUP_NONE;
u8				gslave_master_ctl = 0x00;
u8 				fix_id_mel[3]={0x4d,0x33,0x33};
u16				gTemperature;
u8				gLevel;
u8				gRecvRssi=0x00, gRecvLqi= 0x00;
f_option_t		gOpt;
u8 				tLength,tTemp;

int	 			level_offset[14]={-3,-4,-4,-4,-3,-3,-1,0,1,2,3,4,5,5};
const int	 	default_offset[14]={-3,-4,-4,-4,-3,-3,-1,0,1,2,3,4,5,5};

void udelay(u16 usec)
{
	register u8 i;

	for(i = 0 ;   i < usec ; i++)
	{
		asm volatile("NOP");
	}
}


void mdelay(u16 msec)
{
	register u16 i;

	for(i = 0 ; i < msec  ; i++)
	{
		udelay(250);
		udelay(250);
		udelay(250);
		udelay(50);
	}
}


void sdelay(u16 msec)
{
	register u16 i;

	for(i=0;i<msec;i++)
	{
		mdelay(1000);
	}
}


void putc_(u8 data)
{
	while(!(UCSR0A & (1<<UDRE0)));
	UDR0 = data;
}

void putb(u8 data)
{
	if(data < 0x0a)
		putc_('0' + data);
	else 
		putc_('a' + (data-10));
}

void puthex(u8 data)
{
	putb(data/16);
	putb(data%16);
}


u8 putchr2hex(u8 data)
{
	u8 tmp;

	if(data&0x60)
		tmp = data - 0x60;
	else
		tmp = data - 0x30;

	return tmp;
}


u8 chr2hex(u8 data_h,u8 data_l)
{
	u8 test;

	data_h = putchr2hex(data_h);
	data_l = putchr2hex(data_l);
	
	test=(data_l&0x0F)|((data_h<<4)&0xF0);
	
	return test;	
}


void putstr(u8 * data)
{
	int i =0 ;

	for( i = 0 ;i < 50 ; i++)
	{
		putc_(data[i]);
		if(data[i] == 0x0d || data[i] == 0x00) break;
	}
}


char asc_to_hex(u8 asc)
{
	if((asc >= '0') && (asc <= '9')) return (asc - '0');
	else if((asc >= 'A') && (asc <= 'F')) return (asc - 'A' + 0x0a);
	else if((asc >= 'a') && (asc <= 'f')) return (asc - 'a' + 0x0a);
	else return 0xff;
}

u8 AscToHex(char * val)
{
	u8 tmp;
	
	tmp = asc_to_hex(val[0]) << 4 | asc_to_hex(val[1]);	
	
	return tmp;
}

char asc_to_int(u8 asc)
{
	if((asc >= '0') && (asc <= '9')) return (asc - '0');
	else return 0xff;
}

int AscToInt(char * val)
{
	u8 tmp;
	
	tmp = (asc_to_int(val[0]) * 10) +  asc_to_int(val[1]);	
	
	return tmp;
}


char hex_to_asc(u8 hex)
{
	char da;
	da = hex & 0x0f;
	if((da >= 0) && (da <= 9)) return ('0' + da);
	else return ('a' + da - 0x0a);
}

void hexToAsc(u8 hex,u8 * dest)
{
	dest[0] = hex_to_asc((hex >> 4 ) & 0x0f);
	dest[1] = hex_to_asc((hex >> 0 ) & 0x0f);
}



u8 my_eeprom_read_byte(u16 addr)
{
	eeprom_busy_wait();
	return eeprom_read_byte((u8 *)addr);
}

void my_eeprom_read_block(u8 * data,u8 *addr,u16 length)
{
	eeprom_busy_wait();
	return eeprom_read_block(data,addr,length);
}


void led_onoff(u8 kind,u8 onoff)
{
	if(kind == 0)
	{
		LED_R = onoff;
	}
	else
	{
		LED_O = onoff;
	}
}


void led_flasher(u8 kind,u8 count,u16 msec)
{
	u8 i=0;
	while(i<count)
	{
		led_onoff(kind,1);
		mdelay(msec);
		led_onoff(kind,0);
		mdelay(msec);
		i++;
	}
}


void inituart()
{
	UCSR0A = 0x02;	//U2X = 1
	UCSR0B = 0x00; // Rx/Tx enable
	UCSR0C = 0x06; // 8bit, 1 stop bit, no parity
	UBRR0H = 0x00; 
	UBRR0L = 0x33; // Baudrate 9600
	UCSR0B = 0x98; // rx / tx enable , rx int enable 0x98
}


void hw_setup(void)
{

	CLKPR = _BV(CLKPCE);	// CLKPCErbg1ɂ
	CLKPR = 0b0001;			// 2ɂ



    DDRC=0x00;      
	DDRD=0xC8;
	DDRB=0x0D;

	/*pull_up*/
	PORTC = (1<<PC0) | (1<<PC1);
	PORTD = 0xf7; 

	EICRA &= ~(1<<ISC00);	//INT0̊荞ݔ𗧉茟mŃgKŌŒ
	EICRA &= ~(1<<ISC01);

	#if 0
	EICRA &= ~(1<<ISC10);	//INT1̊荞ݔ𗧉茟mŃgKŌŒ
	EICRA &= ~(1<<ISC11); 
	#endif 
}



u16 ad_get_val(u8 ad_ch,u8 adlar)
{
	u8 i=0;
	u8 ad_data_l=0;
	u8 ad_data_h=0;
	u16 ad=0;

	//u8 temp=0xc0;

	ADCSRA = 0b10000100; // AD:1 ADJn:0 ADN:0 AD:0 AD:0 ck/16

	// reg set
	ADMUX = 0xc0 | adlar | (ad_ch & 0x0f);

	// dummy
	for(i = 0 ; i  < 4 ; i++)
	{
		ADCSRA |= _BV(ADSC);						//ϊJn vٰߓ
		while((ADCSRA&0x10)==0x00);
		ADCSRA |= (1<<ADIF);						//ADIF clear
	}
	
	// read
	ADCSRA |= _BV(ADSC);						//ϊJn vٰߓ
	while((ADCSRA&0x10)==0x00);
	ADCSRA |= (1<<ADIF);						//ADIF clear

	ad_data_l = ADCL;
	ad_data_h = ADCH;

	ad = ad_data_h;
	ad = (ad << 2);
	ad = (ad & 0x03FC) | ((ad_data_l >> 6) & 0x03);

	ADCSRA = 0b00000100; // AD:0 ADJn:0 ADN:0 AD:0 AD:0 ck/16
	return ad;
}



void temperature_test_enable(void)
{
	cc1100_cmd_idle();
	PORTC &= ~(1<<PC2);	//pull-up disenable
	cc1100_write_reg(CC1100_REG_PTEST,0xBF);
	cc1100_cfg_gdo0(0x80);
}


void temperature_test_disenable(void)
{
	cc1100_cfg_gdo0(0x3F);
	cc1100_write_reg(CC1100_REG_PTEST,0x7F);
}


u16 temperature_get(void)
{
	PORTC &= ~(1<<PC2);	//pull-up disenable
	temperature_test_enable();

	DIDR0 |= (1<<ADC2D);		//ADC2 Digital disable

	gTemperature = ad_get_val(AD_PIN_TEMPERATURE,AD_VAL_FROM_L);

	DIDR0 &= ~(1<<ADC2D);		//ADC2 Digital enable

	temperature_test_disenable();
	PORTC |= (1<<PC2);		//pull-up enable

	return gTemperature;
}


int reg_slave_num_serch(u8 * id)
{
	u8 i=0;
	u8 Serial[6];

	memcpy(Serial,fix_id_mel,3);

	for(i=0;i<100;i++)
	{
		my_eeprom_read_block((u8 *)&tSlave,(u8 *)&greg_slave[i],sizeof(reg_slave));

		memcpy((u8 *)&Serial[3],(u8 *)tSlave.serial,3);
		if(memcmp((u8 *)Serial,(u8 *)id,6) == 0)
		{
			return i;
		}
	}
	return -1;
}
	 


u8 get_freq_data(void)
{
	return (~(PIND >> 4)) & 0x03;
}

u8 get_mode_data(void)
{
	return (~(PINC)) & 0x03;
}



void rtc_alarm_disenable(void)
{
	u8 ret;

	ret = rtc_1byte_data_write(0x0E,0x00);
	if(ret == 0)
		ret = rtc_1byte_data_write(0x0E,0x00);
	if(ret == 0)
		ret = rtc_1byte_data_write(0x0E,0x00);

}


u8 set_cur_status(u8 status)
{
	rtc_1byte_data_write(0x0d,status);
}

u8 get_cur_status()
{
	return rtc_1byte_data_read(0x0d);
}




void level_change_hex(u8 * data)
{
	int i;

	for(i = 0 ;i < 14 ; i+=2)
	{
		 data[i/2] = (((level_offset[i] + 7) << 4) & 0xf0) | ((level_offset[i+1] + 7) & 0x0f);
	}
}

void hex_change_level(u8 * data)
{
	int i;

	for(i = 0 ;i < 7 ; i++)
	{
		level_offset[i * 2] = (int)((data[i] >> 4 ) & 0x0f) - 7;
		level_offset[i* 2 + 1] = (int)((data[i]  ) & 0x0f) - 7;
	}

	eeprom_busy_wait();		
	eeprom_write_block(level_offset,&eprom_level,sizeof(level_offset));
}

void slave_fifo_preset(fifo_t * fifo)
{

	memcpy(fifo->slave_id,gClient.status.serial,6);
	memset(fifo->master_id,0xff,3);
	fifo->user_data0 = gClient.status.user_data0;
	fifo->user_data1 = gslave_master_ctl;
	fifo->bat_val = gClient.status.bat_val;
	fifo->setup_mode = (setup_mode == SETUP_NONE ? 0 : 1) ;
	fifo->master_ctl = 0x00;
	fifo->version_l = PRG_VERSION;
	fifo->rssi = get_cur_status();
	if(gClient.status.bat_val < 0xbf)
		fifo->user_data0 |= (1<<0);	// BATT_LOW active

	//set_cur_status(gClient.fifo.rssi);



	if(setup_mode != SETUP_NONE)
	{
		fifo->length = sizeof(fifo_t);
		fifo->opt.env_data[0] = (gTemperature >> 8) & 0xff;
		fifo->opt.env_data[1] = (gTemperature >> 0) & 0xff;
		fifo->opt.env_data[2] = my_eeprom_read_byte((u16)&gctl_data.adj_freq);	//serial no 01;
		fifo->opt.env_data[3] = my_eeprom_read_byte((u16)&gctl_data.freq_add_sub);	//serial no 01;
		fifo->opt.env_data[4] = my_eeprom_read_byte((u16)&gctl_data.adj_temper);	//serial no 01;
		fifo->opt.env_data[5] = my_eeprom_read_byte((u16)&gctl_data.temper_add_sub);	//serial no 01;
		level_change_hex(fifo->opt.level);
	}
	else
	{
		fifo->length = sizeof(fifo_t) - sizeof(f_option_t);
	}
}



void tx_fifo_write(u8 * data,u8 length)
{
	data[0] = length + 2;
	cc1100_fifo_put(data,1);
	cc1100_fifo_put(data,length+2);
}



u8 rx_fifo_read(u8 * data)
{
	u8 length;

	cc1100_fifo_get(&length,1); 

	cc1100_fifo_get(data,length);

	cc1100_fifo_get(&gClient.fifo.rssi,1);
	set_cur_status(gClient.fifo.rssi);

	cc1100_fifo_get(&gClient.fifo.lqi,1);

	return length - 2;
}



u8 frq_data;

void cc1101_carrier_wave_setup(void)
{
	
	volatile u16	frq;
	volatile int 	fr;
	volatile u16 	offset;
	volatile u16 	level;
	volatile u16 	temperature;
	

	frq_data = get_freq_data();
	
	frq = 0x6607 + (frq_data * 0x00fc);

	if(my_eeprom_read_byte((u16)&gctl_data.freq_add_sub) == 0x80)
		frq -= my_eeprom_read_byte((u16)&gctl_data.adj_freq);
	else
		frq += my_eeprom_read_byte((u16)&gctl_data.adj_freq);


	temperature = temperature_get();


	if(my_eeprom_read_byte((u16)&gctl_data.temper_add_sub) == 0x80)
		temperature -= my_eeprom_read_byte((u16)&gctl_data.adj_temper);
	else
		temperature += my_eeprom_read_byte((u16)&gctl_data.adj_temper);


	offset = 0x0339 - temperature;
	
	level = 13 - (offset / 0x0c);
	

	if(offset & 0x8000)
		level = 13;
	else if((offset / 0x0c)>13)
		level = 0;

	gLevel = level;

	frq += level_offset[level];


	cc1100_write_reg(CC1100_REG_FREQ2, 0x10);
	cc1100_write_reg(CC1100_REG_FREQ1, (frq >> 8)  & 0xff);
	cc1100_write_reg(CC1100_REG_FREQ0, frq & 0xff);
}


void cc1101_carrier_frq_setup(unsigned long fr)
{
	unsigned long frq;
	
//	frq = (unsigned long)0x00102762 + ((unsigned long)fr * 0x000000fc);
	frq = (unsigned long)0x000ee276 + ((unsigned long)fr * 0x000000fc);


	cc1100_write_reg(CC1100_REG_FREQ2, (frq >> 16) & 0xff);
	cc1100_write_reg(CC1100_REG_FREQ1, (frq >> 8)  & 0xff);
	cc1100_write_reg(CC1100_REG_FREQ0, frq & 0xff);
}


void init_voltcomparator(void)
{
	ACSR = 0x4b;
	DIDR1 = 0x02;
}



	 


void add_batt_low(void)
{
	u16 bat_val;

	B_CTL_HIGH;
	mdelay(1);


	bat_val = ad_get_val(AD_PIN_BATT_LOW,AD_VAL_FROM_L);

	gClient.status.bat_val = (bat_val>>2);


	B_CTL_LOW;
}




u8 	sig_status(void)
{
	u8 status=0x00;

	if(EMG == 0)
		status = 0x08;

	add_batt_low(); 

	return 0;
}









void carrier_sense()
{
	while(PINC&0x04);		//Carrier sense
}

void carrier_no_sense()
{
	while(!(PINC&0x04));		//Carrier sense
}

void cc1101_tx_carrier(cc1101_client * client)
{

	cc1101_carrier_wave_setup();

	cc1100_cfg_txoff_mode(CC1100_TXOFF_MODE_STAY_TX);
	cc1100_cfg_manchester_en(CC1100_MANCHESTER_DISABLE);
	cc1100_write_reg(CC1100_REG_MDMCFG3,0x33);
	cc1100_cfg_mod_format(CC1100_MODULATION_ASK);
	cc1100_write_reg(CC1100_REG_FREND0,0x10);
	cc1100_cfg_gdo0(CC1100_GDOx_CLK_XOSC_1);

	/* IDLE */
	cc1100_cmd_idle();
	/* MANCAL*/
	cc1100_cmd_calibrate();
	/* FS WAKEUP */
	cc1100_cmd_flush_tx();

	cc1100_cfg_gdo0(0x0e);
	carrier_sense();

	cc1100_cmd_tx();
}


void cc1101_tx_carrier_test(unsigned long fr)
{

	cc1101_carrier_frq_setup(fr);

	cc1100_cfg_txoff_mode(CC1100_TXOFF_MODE_STAY_TX);
	cc1100_cfg_manchester_en(CC1100_MANCHESTER_DISABLE);
	cc1100_write_reg(CC1100_REG_MDMCFG3,0x33);
	cc1100_cfg_mod_format(CC1100_MODULATION_ASK);
	cc1100_write_reg(CC1100_REG_FREND0,0x10);
	cc1100_cfg_gdo0(CC1100_GDOx_CLK_XOSC_1);

	/* IDLE */
	cc1100_cmd_idle();
	/* MANCAL*/
	cc1100_cmd_calibrate();
	/* FS WAKEUP */
	cc1100_cmd_flush_tx();

	cc1100_cfg_gdo0(0x0e);
	carrier_sense();

	cc1100_cmd_tx();
}




u8 gStatus=0x00;

u8 cc1101_tx(u8 * data , u8 length,u8 th)
{

	cc1101_rtx_reg_set(1); // rx
	cc1101_carrier_wave_setup();
	cc1100_cfg_gdo0(0x0e);

	cc1100_cmd_idle();
	cc1100_cmd_flush_rx();
	cc1100_cmd_calibrate();
	cc1100_cmd_rx();
	
	mdelay(5);

	if(PINC & 0x04)
	{
		cc1100_cmd_idle();
		mdelay(195);
		if(th == 0)
			return 0;

	}


	cc1101_rtx_reg_set(0); // tx
	cc1101_carrier_wave_setup();
	cc1100_cmd_idle();
	cc1100_cmd_calibrate();
	tx_fifo_write(data,length);
	
	cc1100_write_reg(CC1100_REG_MCSM1,0x00);
	cc1100_cfg_gdo0(0x09);



	cc1100_cfg_gdo0(CC1100_GDOx_SYNC_WORD);
	cc1100_cmd_tx();
	while(!(PINC&0x04));	//data send start
	while(PINC&0x04);		//data send end

	return 1;
}



u8 cc1101_rx(u8 * data,u8 loop)
{
	u8 cnt_wait_syn_ack=0;
	u8 length;

	cc1101_rtx_reg_set(1); // rx
	cc1101_carrier_wave_setup();
	cc1100_cfg_gdo0(CC1100_GDOx_SYNC_WORD);

	cc1100_cmd_idle();
	cc1100_cmd_flush_rx();
	cc1100_cmd_calibrate();
	cc1100_cmd_rx();
	
	while(!(PINC & 0x04))
	{
		mdelay(1);
		if(++cnt_wait_syn_ack > 50)
			return 0;
	}

	while(1)
	{
		if(!( PINC & 0x04) )
		{	
			if((cc1100_status_crc_lqi() & 0x80))
			{

				length = rx_fifo_read(data);
				return length;
			}
			break;
		}
	}

	return 0;
}




u8 my_memcmp(u8 * src,u8 * dest,u8 len)
{
	u8 i;

	for(i= 0 ; i < len ; i++)
	{
		if(src[i] != dest[i]) return i;
	}

	return 0;
}



ISR(INT0_vect)
{

	EIMSK &= ~(1<<INT0);

	spi_signalpin_opendrain_nonactive();

	rtc_alarm_disenable();
	//led_flasher(5,50);
}




/*
 	input_data
 	b0 b1 		type 00=master,01=slave,10=carrier,11=setup
*/

void device_status_setup(void)
{
	memcpy((u8 *)&gClient.status.serial,fix_id_mel,3);
	my_eeprom_read_block((u8 *)&gClient.status.serial[3],(u8 *)&gctl_data.serial,3);
	gClient.status.send_err_cnt=0;
}





void rtc_set_next_alarm(u8 alarm)
{
	int i;
	u8 ret;


	if(alarm < 4)
	{
		reg_net_alarm[3].data = time_set[alarm].hour;
		reg_net_alarm[4].data = time_set[alarm].min;
		reg_net_alarm[5].data = time_set[alarm].sec;
	
		for(i = 0 ;i < 7 ; i++)
		{
			ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
		}
	}
	else
	{
		reg_next_sch_alarm[3].data = day_cnt[alarm-4];

		for(i = 0 ;i < 9 ; i++)
		{
			ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
		}
	}
}

u8	get_stop_signal()
{
	int i;


	for(i = 0 ;i < 500 ; i++)
	{
		if(EMG == 1)
			return 0;
		mdelay(1);
	}

	return 1;
}



void temperature_carrier_test(void)
{
	cc1100_cfg_gdo2(0x30);		//26MHz XOSC/1  output
	while(1)
	{
		//wdt_reset();
		if(get_stop_signal() == 1) power_off();
		led_flasher(1,5,50);
		add_batt_low();
		cc1101_tx_carrier((cc1101_client *)&gClient);
		sdelay(5);
		temperature_test_enable();
		sdelay(1);
		temperature_test_disenable();
	}
}

void carrier_test(void)
{
	unsigned long fr;
	cc1100_cfg_gdo2(0x30);		//26MHz XOSC/1  output
	led_flasher(1,5,50);
	fr = 0;
	while(1)
	{
		//wdt_reset();
		if(get_stop_signal() == 1) power_off();
		cc1101_tx_carrier_test(fr);
		mdelay(100);
		fr++;
		//if(fr >= 96)
		if(fr >= 788L)
		{
			led_flasher(1,5,50);
			fr = 0;
		}
	}
}





u8 slave_oper()
{
	u8	ret;
	u8	cnt;
	u8 i=0;
	fifo_t * fifo;
	int length;


	fifo = &gClient.fifo;

	add_batt_low();
	//led_onoff(1,1);
	cnt = 0;
	for(i = 0  ; i < 7 ;i++)
	{
		slave_fifo_preset(fifo);
		ret = cc1101_tx((u8 *)fifo,fifo->length,cnt > 5 ? 1 : 0);


		if(ret == 0)
			cnt++;

		if(ret == 1)
		{
			//wdt_reset();
			length = cc1101_rx((u8 *)fifo,0);
			if(length > 0)
			{
				if(my_memcmp(gClient.status.serial,fifo->slave_id,6) == 0)
				{
					gslave_master_ctl = fifo->master_ctl;
					fifo->length=0;
					//led_onoff(1,0);
					return 1;
				}
			}
		}
	}
	//led_onoff(1,0);
	return 0;
}


void sleep_set(u8 alarm,u8 bo)
{
	if(bo == 0)
		rtc_set_next_alarm(alarm);

	cc1100_cmd_idle();
	cc1100_cmd_pwd();

	spi_signalpin_opendrain_active();
	EIMSK |= (1<<INT0);
	sleep_mode();
	//led_onoff(1,0);

}


void power_off()
{
	PWR_HOLD = 0;
	while(1);
}
void slave_main()
{
	u8 ret=0;
	u8 alarm=0;

	while(1)
	{
		//wdt_reset();
		switch(gClient.status.status)
		{
			case SL_BOOT:
				// config read
				alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
				gClient.status.status = SL_SLEEP;
				break;
			case SL_SCH_SEND:
				gClient.status.user_data0 = ((sig_status() & 0x1A) | 0x10);
				ret = slave_oper((cc1101_client *)&gClient);
				if(ret == 1)
				{
					alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
					rtc_set_next_alarm(alarm);
					led_flasher(1,10,100);
					power_off();
				}
				else
				{
					alarm = ALARM_2SEC;
					gClient.status.send_err_cnt++;
					if(gClient.status.send_err_cnt > 8)
					{
						alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
						rtc_set_next_alarm(alarm);
						led_flasher(0,10,100);
						power_off();
					}
				}
				break;
			case SL_SEND:
				led_onoff(0,1);
				if(gClient.status.send_err_cnt  == 0)
					gClient.status.user_data0 =  0x28;	//EMG_SIG_SET

				ret = slave_oper((cc1101_client *)&gClient);
				if(ret == 1)
				{
					led_flasher(1,10,100);
					alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
					rtc_set_next_alarm(alarm);
					power_off();
				}
				else
				{
					alarm = ALARM_2SEC;
					gClient.status.send_err_cnt++;
					if(gClient.status.send_err_cnt > 6)
					{
						led_flasher(0,10,100);
						alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
						rtc_set_next_alarm(alarm);
						power_off();
					}
				}
				break;
		}

		sleep_set(alarm,0);

	}
}



u8 setup_oper()
{
	u8	ret;
	int 	length;
	fifo_t * fifo;


	fifo = &gClient.fifo;

	

	temperature_get();

	slave_fifo_preset(fifo);
	ret = cc1101_tx((u8 *)fifo,sizeof(fifo_t),0);

	if(ret == 1)
	{
		length = cc1101_rx((u8 *)fifo,0);
		if(length > 0)
		{
			if(length == sizeof(fifo_t) && fifo->setup_mode == SETUP_WRITE)
			{
				memcpy((u8 *)tCtl.serial,(u8 *)&fifo->opt.env_data[0],3);
				tCtl.adj_freq = fifo->opt.env_data[4];
				tCtl.freq_add_sub = fifo->opt.env_data[5];
				tCtl.adj_temper = fifo->opt.env_data[6];
				tCtl.temper_add_sub = fifo->opt.env_data[7];
	
				memcpy((u8 *)&gClient.status.serial[3],(u8 *)tCtl.serial,3);
				memcpy((u8 *)&fifo->slave_id[3],(u8 *)tCtl.serial,3);

				eeprom_busy_wait();		
				eeprom_write_block(&tCtl, &gctl_data, sizeof(ctl_data)); 

				hex_change_level(fifo->opt.level);


				cc1101_tx_carrier((cc1101_client *)&gClient);
				sdelay(5);

				return 1;
			}
			fifo->length=0;
		}
	}
	return 0;
}

extern u8 pt_data[];

void setup_main()
{
	while(1)
	{
		if(setup_mode == SETUP_READY)
		{
			led_flasher(1,1,1000);
		}
		else
		{
			if(get_stop_signal() == 1) power_off();

			led_flasher(1,5,100);
			pt_data[0] = 0xc0;
			cc1101_8PATABLE_write_reg();
			setup_oper();

			pt_data[0] = 0x60;
			cc1101_8PATABLE_write_reg();
			add_batt_low();
			cc1101_tx_carrier((cc1101_client *)&gClient);
			sdelay(2);
		}
	}
}


void check_env(void)
{
	u8  i;
	u8 * pD;

	
	my_eeprom_read_block((u8 *)&tCtl,(u8 *)&gctl_data,sizeof(ctl_data));

	pD = (u8 *)&tCtl;

	//for(i = 0 ;i  < 5 ; i++)
	for(i = 0 ;i  < 10 ; i++)
	{
		if(pD[i] != 0xff && pD[i] != 0x00) break;
	}

	//if(i == 5)
	if(i == 10)
	{
		// init;
		tCtl.adj_freq = 0x24;
		tCtl.freq_add_sub = 0x80;
		tCtl.adj_temper = 0x18;
		tCtl.temper_add_sub = 0x80;
		eeprom_busy_wait();		
		eeprom_write_block(&tCtl,&gctl_data,sizeof(ctl_data));
	}

	my_eeprom_read_block((u8 *)level_offset,(u8 *)&eprom_level,sizeof(level_offset));


	pD = (u8 *)&level_offset;

	//for(i = 0 ;i  < 5 ; i++)
	for(i = 0 ;i  < 28 ; i++)
	{
		if(pD[i] != 0xff && pD[i] != 0x00) break;
	}


	//if(i == 5)
	if(i == 28)
	{
		eeprom_busy_wait();		
		eeprom_write_block(default_offset,&eprom_level,sizeof(level_offset));
		memcpy(level_offset,default_offset,sizeof(level_offset));

	}
}

extern u8 pt_data[] ;


u8	get_emg_signal()
{
	int i;


	for(i = 0 ;i < 500 ; i++)
	{
		if(EMG == 1)
			return 0;
		mdelay(1);
	}

	return 1;
}


int main()
{
	int i;
	hw_setup();
	frq_data = get_freq_data();
	spi_signalpin_opendrain_nonactive();
	PWR_HOLD = 1;
	led_onoff(0,0);
	led_onoff(1,0);
	i2c_init();
	inituart();
	check_env();
	device_status_setup();
	cc1100_init();
	cc1101_init_reg();
	gClient.status.type = get_mode_data();
	if(gClient.status.type == SLAVE)
		pt_data[0] = 0x60;
	cc1101_8PATABLE_write_reg();


	#if 0
	if(get_cur_status() == 0x00)
	{
		set_cur_status(0x0a);
		rtc_1byte_data_write(0x0E,0x00);		//Alarm enable disenable
	}
	#endif 


	if(gClient.status.type == MASTER)
	{
		set_sleep_mode(SLEEP_MODE_PWR_DOWN);
		sei();

		gClient.status.status = SL_BOOT;
		if(RTC_INT == 0)
		{
			gClient.status.status = SL_SCH_SEND;
			rtc_alarm_disenable();
		}
		else if(get_emg_signal() == true)
			gClient.status.status = SL_SEND;

		if(gClient.status.status != SL_BOOT )
			slave_main();
		else
			power_off();
	}
	else
	{
		if(gClient.status.type == SLAVE )
		{
			carrier_test();
		}
		else if(gClient.status.type == TEST_MDOE)
		{
			temperature_carrier_test();
		}
		else if(gClient.status.type == SETUP)
		{
			setup_mode = SETUP_TRX;;
			setup_main();
		}
	}
}


