/**
 * @file bmpimg.c
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "bmpimg.h"
#include "bmpfile.h"

static const unsigned char FONT_X = 5;              /**< フォントの横方向サイズ。 */
static const unsigned char FONT_Y = 7;              /**< フォントの縦方向サイズ。 */
static const unsigned char FONT_MIN_CODE = 0x20;    /**< フォントテーブルの最小コード番号。 */
static const unsigned char FONT_MAX_CODE = 0x7F;    /**< フォントテーブルの最大コード番号。 */
static const unsigned char font5x7_data[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, // (white space)
    0x00, 0x00, 0x5F, 0x00, 0x00, // !
    0x00, 0x07, 0x00, 0x07, 0x00, // "
    0x14, 0x7F, 0x14, 0x7F, 0x14, // #
    0x24, 0x2A, 0x7F, 0x2A, 0x12, // $
    0x23, 0x13, 0x08, 0x64, 0x62, // %
    0x36, 0x49, 0x55, 0x22, 0x50, // &
    0x00, 0x05, 0x03, 0x00, 0x00, // '
    0x00, 0x1C, 0x22, 0x41, 0x00, // (
    0x00, 0x41, 0x22, 0x1C, 0x00, // )
    0x08, 0x2A, 0x1C, 0x2A, 0x08, // *
    0x08, 0x08, 0x3E, 0x08, 0x08, // +
    0x00, 0x50, 0x30, 0x00, 0x00, // ,
    0x08, 0x08, 0x08, 0x08, 0x08, // -
    0x00, 0x60, 0x60, 0x00, 0x00, // .
    0x20, 0x10, 0x08, 0x04, 0x02, // /
    0x3E, 0x51, 0x49, 0x45, 0x3E, // 0
    0x00, 0x42, 0x7F, 0x40, 0x00, // 1
    0x42, 0x61, 0x51, 0x49, 0x46, // 2
    0x21, 0x41, 0x45, 0x4B, 0x31, // 3
    0x18, 0x14, 0x12, 0x7F, 0x10, // 4
    0x27, 0x45, 0x45, 0x45, 0x39, // 5
    0x3C, 0x4A, 0x49, 0x49, 0x30, // 6
    0x01, 0x71, 0x09, 0x05, 0x03, // 7
    0x36, 0x49, 0x49, 0x49, 0x36, // 8
    0x06, 0x49, 0x49, 0x29, 0x1E, // 9
    0x00, 0x36, 0x36, 0x00, 0x00, // :
    0x00, 0x56, 0x36, 0x00, 0x00, // ;
    0x00, 0x08, 0x14, 0x22, 0x41, // <
    0x14, 0x14, 0x14, 0x14, 0x14, // =
    0x41, 0x22, 0x14, 0x08, 0x00, // >
    0x02, 0x01, 0x51, 0x09, 0x06, // ?
    0x32, 0x49, 0x79, 0x41, 0x3E, // @
    0x7E, 0x11, 0x11, 0x11, 0x7E, // A
    0x7F, 0x49, 0x49, 0x49, 0x36, // B
    0x3E, 0x41, 0x41, 0x41, 0x22, // C
    0x7F, 0x41, 0x41, 0x22, 0x1C, // D
    0x7F, 0x49, 0x49, 0x49, 0x41, // E
    0x7F, 0x09, 0x09, 0x01, 0x01, // F
    0x3E, 0x41, 0x41, 0x51, 0x32, // G
    0x7F, 0x08, 0x08, 0x08, 0x7F, // H
    0x00, 0x41, 0x7F, 0x41, 0x00, // I
    0x20, 0x40, 0x41, 0x3F, 0x01, // J
    0x7F, 0x08, 0x14, 0x22, 0x41, // K
    0x7F, 0x40, 0x40, 0x40, 0x40, // L
    0x7F, 0x02, 0x04, 0x02, 0x7F, // M
    0x7F, 0x04, 0x08, 0x10, 0x7F, // N
    0x3E, 0x41, 0x41, 0x41, 0x3E, // O
    0x7F, 0x09, 0x09, 0x09, 0x06, // P
    0x3E, 0x41, 0x51, 0x21, 0x5E, // Q
    0x7F, 0x09, 0x19, 0x29, 0x46, // R
    0x46, 0x49, 0x49, 0x49, 0x31, // S
    0x01, 0x01, 0x7F, 0x01, 0x01, // T
    0x3F, 0x40, 0x40, 0x40, 0x3F, // U
    0x1F, 0x20, 0x40, 0x20, 0x1F, // V
    0x7F, 0x20, 0x18, 0x20, 0x7F, // W
    0x63, 0x14, 0x08, 0x14, 0x63, // X
    0x03, 0x04, 0x78, 0x04, 0x03, // Y
    0x61, 0x51, 0x49, 0x45, 0x43, // Z
    0x00, 0x00, 0x7F, 0x41, 0x41, // [
    0x02, 0x04, 0x08, 0x10, 0x20, // /
    0x41, 0x41, 0x7F, 0x00, 0x00, // ]
    0x04, 0x02, 0x01, 0x02, 0x04, // ^
    0x40, 0x40, 0x40, 0x40, 0x40, // _
    0x00, 0x01, 0x02, 0x04, 0x00, // `
    0x20, 0x54, 0x54, 0x54, 0x78, // a
    0x7F, 0x48, 0x44, 0x44, 0x38, // b
    0x38, 0x44, 0x44, 0x44, 0x20, // c
    0x38, 0x44, 0x44, 0x48, 0x7F, // d
    0x38, 0x54, 0x54, 0x54, 0x18, // e
    0x08, 0x7E, 0x09, 0x01, 0x02, // f
    0x08, 0x14, 0x54, 0x54, 0x3C, // g
    0x7F, 0x08, 0x04, 0x04, 0x78, // h
    0x00, 0x44, 0x7D, 0x40, 0x00, // i
    0x20, 0x40, 0x44, 0x3D, 0x00, // j
    0x00, 0x7F, 0x10, 0x28, 0x44, // k
    0x00, 0x41, 0x7F, 0x40, 0x00, // l
    0x7C, 0x04, 0x18, 0x04, 0x78, // m
    0x7C, 0x08, 0x04, 0x04, 0x78, // n
    0x38, 0x44, 0x44, 0x44, 0x38, // o
    0x7C, 0x14, 0x14, 0x14, 0x08, // p
    0x08, 0x14, 0x14, 0x18, 0x7C, // q
    0x7C, 0x08, 0x04, 0x04, 0x08, // r
    0x48, 0x54, 0x54, 0x54, 0x20, // s
    0x04, 0x3F, 0x44, 0x40, 0x20, // t
    0x3C, 0x40, 0x40, 0x20, 0x7C, // u
    0x1C, 0x20, 0x40, 0x20, 0x1C, // v
    0x3C, 0x40, 0x30, 0x40, 0x3C, // w
    0x44, 0x28, 0x10, 0x28, 0x44, // x
    0x0C, 0x50, 0x50, 0x50, 0x3C, // y
    0x44, 0x64, 0x54, 0x4C, 0x44, // z
    0x00, 0x08, 0x36, 0x41, 0x00, // {
    0x00, 0x00, 0x7F, 0x00, 0x00, // |
    0x00, 0x41, 0x36, 0x08, 0x00, // }
    0x08, 0x08, 0x2A, 0x1C, 0x08, // ->
    0x08, 0x1C, 0x2A, 0x08, 0x08  // <-
};

/**
 * @brief 文字を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x 横方向描画開始ピクセル位置。
 * @param y 縦方向描画開始ピクセル位置。
 * @param size 1ピクセルのサイズ。
 * @param c 文字。
 * @param color 描画色。
 */
static void draw_char(bmpimg_t *p, const int x, const int y, const int size, const char c, bmpcol_t *color) {
    int i, j;
    if ((FONT_MIN_CODE <= c) && (c <= FONT_MAX_CODE)) {
        int aofs = (c - FONT_MIN_CODE) * FONT_X;
        for (i = 0; i < FONT_X; i++) {
            unsigned char pat = font5x7_data[aofs + i];
            for (j = 0; j < FONT_Y; j++) {
                if (pat & (1 << j)) {
                    bmpimg_fill_box(p, x + (i * size), y + (j * size), x + (i * size) + size - 1, y + (j * size) + size - 1, color);
                }
            }
        }
    } else {
        for (i = 0; i < FONT_X; i++) {
            unsigned char pat = (i % 2) ? 0x55 : 0xAA;
            for (j = 0; j < FONT_Y; j++) {
                if (pat & (1 << j)) {
                    bmpimg_fill_box(p, x + (i * size), y + (j * size), x + (i * size) + size - 1, y + (j * size) + size - 1, color);
                }
            }
        }
    }
}

/**
 * @brief BMP画像をオープンする。
 *
 * @param p BMP画像ハンドラ。
 * @param w 画像横方向サイズ。
 * @param h 画像縦方向サイズ。
 * @param writer ピクセル書き込み関数。
 * @param writer_extobj ピクセル書き込み関数に渡すパラメータ。
 * @param reader ピクセル読み込み関数。
 * @param reader_extobj ピクセル読み込み関数に渡すパラメータ。
 */
void bmpimg_open(
        bmpimg_t *p,
        const int w,
        const int h,
        PIXEL_WRITER writer,
        void *writer_extobj,
        PIXEL_READER reader,
        void *reader_extobj)
{
    p->w = w;
    p->h = h;
    p->writer        = writer;
    p->writer_extobj = writer_extobj;
    p->reader        = reader;
    p->reader_extobj = reader_extobj;
}

/**
 * @brief ボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_box(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color)
{
    int x, y;
    if ((p->w < x1) || (p->w < x2) || (p->h < y1) || (p->h < y2)) {
        return -1;
    }
    for (y = y1; y <= y2; y++) {
        p->writer(x1, y, color->r, color->g, color->b, p->writer_extobj);
        p->writer(x2, y, color->r, color->g, color->b, p->writer_extobj);
    }
    for (x = x1; x <= x2; x++) {
        p->writer(x, y1, color->r, color->g, color->b, p->writer_extobj);
        p->writer(x, y2, color->r, color->g, color->b, p->writer_extobj);
    }
    return 0;
}

/**
 * @brief ボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_fill_box(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color)
{
    int x, y;
    if ((p->w < x1) || (p->w < x2) || (p->h < y1) || (p->h < y2)) {
        return -1;
    }
    for (y = y1; y <= y2; y++) {
        for (x = x1; x <= x2; x++) {
            p->writer(x, y, color->r, color->g, color->b, p->writer_extobj);
        }
    }
    return 0;
}

/**
 * @brief 文字列を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x 横方向描画開始ピクセル位置。
 * @param y 縦方向描画開始ピクセル位置。
 * @param size 1ピクセルのサイズ。
 * @param text 文字列。
 * @param color 描画色。
 */
int bmpimg_draw_string(bmpimg_t *p, const int x, const int y, const int size, const char *text, bmpcol_t *color)
{
    char *strp = (char *)text;
    int cnt = 0;
    while (*strp) {
        draw_char(p, x + ((FONT_X * size) * cnt) + (size * cnt), y, size, *strp, color);
        strp++;
        cnt++;
    }
    return 0;
}

static void val_swap(int *a, int *b)
{
    int c;
    c = *a;
    *a = *b;
    *b = c;
}

static int val_abs(int a)
{
    if (a < 0) {
        return -a;
    }
    return a;
}

/**
 * @brief 線を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_line(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color)
{
    int tmp_x1 = x1;
    int tmp_y1 = y1;
    int tmp_x2 = x2;
    int tmp_y2 = y2;
    int steep = val_abs(tmp_y2 - tmp_y1) > val_abs(tmp_x2 - tmp_x1);
    int deltax, deltay;
    int error;
    int ystep, y;
    int x;
    if (steep) {
        val_swap(&tmp_x1, &tmp_y1);
        val_swap(&tmp_x2, &tmp_y2);
    }
    if (tmp_x1 > tmp_x2) {
        val_swap(&tmp_x1, &tmp_x2);
        val_swap(&tmp_y1, &tmp_y2);
    }
    deltax = tmp_x2 - tmp_x1;
    deltay = val_abs(tmp_y2 - tmp_y1);
    error = deltax / 2;
    y = tmp_y1;
    if (tmp_y1 < tmp_y2) {
        ystep = 1;
    } else {
        ystep = -1;
    }
    for (x = tmp_x1; x <= tmp_x2; x++) {
        if (steep) {
            bmpimg_fill_box(p, y, x, y, x, color);
        } else {
            bmpimg_fill_box(p, x, y, x, y, color);
        }
        error = error - deltay;
        if (error < 0) {
            y = y + ystep;
            error = error + deltax;
        }
    }
    return 0;
}

/**
 * @brief 点を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x X座標。
 * @param y Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_pixel(bmpimg_t *p, const int x, const int y, bmpcol_t *color)
{
    p->writer(x, y, color->r, color->g, color->b, p->writer_extobj);
    return 0;
}

/**
 * @brief チェックボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param state 選択状態。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
void bmpimg_draw_checkbox(
        bmpimg_t *p,
        const int x1,
        const int y1,
        const int x2,
        const int y2,
        const int state,
        bmpcol_t *color)
{
    bmpimg_draw_box(p, x1, y1, x2, y2, color);
    if (state) {
        bmpimg_draw_line(p, x1, y1, x2, y2, color);
        bmpimg_draw_line(p, x1, y2, x2, y1, color);
    }
}

/**
 * @brief プログレスバーを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param min 最小値。
 * @param max 最大値。
 * @param value 現在値。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
void bmpimg_draw_progressbar(
        bmpimg_t *p,
        const int x1,
        const int y1,
        const int x2,
        const int y2,
        const int min,
        const int max,
        const int value,
        bmpcol_t *color)
{
    int tmp_max, tmp_val, pix;
    bmpimg_draw_box(p, x1, y1, x2, y2, color);
    if ((value < min) || (max < value)) {
        return;
    }
    tmp_max = max - min;
    tmp_val = value - min;
    pix = ((x2 - x1) * tmp_val) / tmp_max;
    bmpimg_fill_box(p, x1 + 1, y1 + 1, x1 + pix - 1, y2 - 1, color);
}

/**
 * @brief BMPファイルから読み込む。
 *
 * @param p BMP画像ハンドラ。
 * @param reader ファイル読み込み関数。
 * @param reader_extobj ファイル読み込み関数に渡す外部オブジェクト。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_bmp_read(
        bmpimg_t *p,
        FILE_READER reader,
        void *reader_extobj)
{
    return bmpfile_read(reader, reader_extobj, p->writer, p->writer_extobj);
}

/**
 * @brief BMPファイルに書き込む。
 *
 * @param p BMP画像ハンドラ。
 * @param writer ファイル書き込み関数。
 * @param writer_extobj ファイル書き込み関数に渡す外部オブジェクト。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_bmp_write(
        bmpimg_t *p,
        FILE_WRITER writer,
        void *writer_extobj)
{
    return bmpfile_write(p->w, p->h, writer, writer_extobj, p->reader, p->reader_extobj);
}

/**
 * @brief BMP画像をクローズする。
 *
 * @param bit ビット深度。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_close(bmpimg_t *p)
{
    return 0;
}

