/**
 * @file bmpfile.c
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  Tiny BMP I/O Module
 *  Version 0.0.1
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "bmpfile.h"
#include "bmplowio.h"

/**
 * @brief BMPファイルを読み込む。
 * @details
 * この関数はBMPファイルが「どこに格納されているのか？」について全く感知しない。
 * また、読み込んだピクセルデータを「どこに格納するのか？」についても感知しない。
 * ユーザは、これら二つの抽象化された操作を提供する関数へのポインタを提供する必要がある。
 *
 * 例えば、小規模組み込みシステムにおいて、BMPファイルのイメージがSDカードやフラッシュ上に書かれているかもしれない。
 * また、読み込んだピクセルをSPI接続されたSRAMに格納したり、そのままディスプレイに表示したい事もあるだろう。
 * いずれにせよこれらの機能を実現するのは、この関数に渡すそれぞれの関数内部で実装詳細を提供すれば良い。
 *
 * @param func_fread ファイル読み込み関数。
 * @param extobj_func_fread ファイル読み込み関数に渡すパラメータ。
 * @param func_pixel_write ピクセル書き込み関数。
 * @param extobj_pixel_write ピクセル書き込み関数に渡すパラメータ。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpfile_read(
        int (*func_fread)(void *buf, const unsigned int size, void *extobj_func_fread),
        void *extobj_func_fread,
        void (*func_pixel_write)(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj_pixel_write),
        void *extobj_pixel_write)
{
    bmp_file_t filehead;
    bmp_info_t infohead;
    bmp_rgbquad_t rgbquad;

    bmplowio_header_read(func_fread, extobj_func_fread, &filehead, &infohead);
    if (bmplowio_have_palette(BMP_INFO_GET_BIT_COUNT(&infohead))) {
        bmplowio_palette_read(func_fread, extobj_func_fread, &rgbquad, (1 << infohead.biBitCount));
    }
    bmplowio_image_read(func_fread, extobj_func_fread, &filehead, &infohead, func_pixel_write, extobj_pixel_write);

    return 0;
}

/**
 * @brief BMPファイルを書き込む。
 * @details
 * この関数はBMPファイルが「どこに格納されているのか？」について全く感知しない。
 * また、書き込むためのピクセルデータが「どこに格納されているのか？」についても感知しない。
 * ユーザは、これら二つの抽象化された操作を提供する関数へのポインタを提供する必要がある。
 *
 * 例えば、小規模組み込みシステムにおいて、BMPファイルのイメージをSDカードやフラッシュ上に書きたいかもしれない。
 * また、書き込むピクセルはSPI接続されたSRAMに格納されているか、ディスプレイ上のメモリから直接取得したい事もあるだろう。
 * いずれにせよこれらの機能を実現するのは、この関数に渡すそれぞれの関数内部で実装詳細を提供すれば良い。
 *
 * @param func_fwrite ファイル書き込み関数。
 * @param extobj_func_fwrite ファイル書き込み関数に渡すパラメータ。
 * @param func_pixel_read ピクセル読み込み関数。
 * @param extobj_pixel_read ピクセル読み込み関数に渡すパラメータ。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpfile_write(
        const int w,
        const int h,
        int (*func_fwrite)(const void *buf, const unsigned int size, void *extobj_func_fwrite),
        void *extobj_func_fwrite,
        void (*func_pixel_read)(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj_pixel_read),
        void *extobj_pixel_read)
{
    bmp_file_t filehead;
    bmp_info_t infohead;
    bmp_rgbquad_t rgbquad;

    BMP_FILE_SET_TYPE(&filehead, BMP_FILE_MAGIC_TEXT);
    BMP_FILE_SET_SIZE(&filehead, bmplowio_calc_framebytesize(24, w, h) + 54);
    BMP_FILE_SET_RESERVED1(&filehead, 0);
    BMP_FILE_SET_RESERVED2(&filehead, 0);
    BMP_FILE_SET_OFFBITS(&filehead, 54);

    BMP_INFO_SET_SIZE(&infohead, 40);
    BMP_INFO_SET_WIDTH(&infohead, w);
    BMP_INFO_SET_HEIGHT(&infohead, h);
    BMP_INFO_SET_PLANES(&infohead, 1);
    BMP_INFO_SET_BIT_COUNT(&infohead, 24);
    BMP_INFO_SET_COMPRESSION(&infohead, 0);
    BMP_INFO_SET_SIZE_IMAGE(&infohead, bmplowio_calc_framebytesize(24, w, h));
    BMP_INFO_SET_X_PELS_PER_METER(&infohead, 0);
    BMP_INFO_SET_Y_PELS_PER_METER(&infohead, 0);
    BMP_INFO_SET_CLR_USED(&infohead, 0);
    BMP_INFO_SET_CLR_IMPORTANT(&infohead, 0);

    bmplowio_header_write(func_fwrite, extobj_func_fwrite, &filehead, &infohead);
    if (bmplowio_have_palette(BMP_INFO_GET_BIT_COUNT(&infohead))) {
        bmplowio_palette_write(func_fwrite, extobj_func_fwrite, &rgbquad, (1 << infohead.biBitCount));
    }
    bmplowio_image_write(func_fwrite, extobj_func_fwrite, &filehead, &infohead, func_pixel_read, extobj_pixel_read);

    return 0;
}

