/**
 * @file fifo.h
 * @author Shinichiro Nakamura
 * @brief FIFOモジュールの定義。
 */

/*
 * ===============================================================
 *  Natural Tiny Logger (NT-Logger)
 * ===============================================================
 * Copyright (c) 2010-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef FIFO_H
#define FIFO_H

/**
 * @brief FIFOハンドラの定義。
 */
typedef struct FIFO FIFO;

/**
 * @brief FIFOモジュールをオープンする。
 *
 * @param depth FIFOの深さ。
 *
 * @retval !NULL FIFOハンドラ。
 * @retval NULL 失敗。
 */
FIFO *fifo_open(const int depth);

/**
 * @brief FIFOが一杯かどうかを検査する。
 *
 * @param p FIFOハンドラ。
 *
 * @retval 1 FIFOが一杯である。
 * @retval 0 FIFOが一杯でない。
 */
int fifo_full(FIFO *p);

/**
 * @brief FIFOが空かどうかを検査する。
 *
 * @param p FIFOハンドラ。
 *
 * @retval 1 FIFOが空である。
 * @retval 0 FIFOが空でない。
 */
int fifo_empty(FIFO *p);

/**
 * @brief FIFOに格納されているオブジェクト数を取得する。
 *
 * @param p FIFOハンドラ。
 *
 * @return オブジェクト数。
 */
int fifo_count(FIFO *p);

/**
 * @brief FIFOにオブジェクトをプッシュする。
 * @details
 * このモジュールはオブジェクトへのポインタのみを保持する。
 * このため、スタック上に置いたオブジェクトなどをプッシュする事はできない。
 * 必ずヒープ上のオブジェクトへのポインタを渡すこと。
 *
 * @param p FIFOハンドラ。
 * @param obj オブジェクト。
 *
 * @retval 1 成功。
 * @retval 0 失敗。
 */
int fifo_push(FIFO *p, void *obj);

/**
 * @brief FIFOからオブジェクトをプルする。
 *
 * @param p FIFOハンドラ。
 * @param obj オブジェクト。
 *
 * @retval 1 成功。
 * @retval 0 失敗。
 */
int fifo_pull(FIFO *p, void **obj);

/**
 * @brief FIFOモジュールをクローズする。
 *
 * @param p FIFOハンドラ。
 *
 * @retval 1 成功。
 * @retval 0 失敗。
 */
int fifo_close(FIFO *p);

#endif

