/**
 * @file ntlcap.c
 * @author Shinichiro Nakamura
 * @brief Natural Tiny Logger (NT-Logger)のホスト側キャプチャ。
 */

/*
 * ===============================================================
 *  Natural Tiny Logger (NT-Logger)
 * ===============================================================
 * Copyright (c) 2010-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "captime.h"
#include "serial.h"
#include "optparse.h"

/**
 * @brief オプションを初期化する。
 *
 * @param P オプション構造体へのポインタ。
 */
#define OPTION_INIT(P) \
    do { \
        (P)->optflag = 0; \
        strcpy((P)->portname, ""); \
        (P)->baudrate = SerialBaud9600; \
        strcpy((P)->output, ""); \
        (P)->count = 0; \
    } while (0)

#define OPTFLAG_PORTNAME    (1 << 0)    /**< ポート名。 */
#define OPTFLAG_BAUDRATE    (1 << 1)    /**< ボーレート。 */
#define OPTFLAG_OUTPUT      (1 << 2)    /**< 出力ファイル名。 */
#define OPTFLAG_COUNT       (1 << 3)    /**< 記録イベント数。 */

/**
 * @brief オプション構造体。
 */
typedef struct {
    unsigned char optflag;      /**< オプション指定確認用フラグ。 */
    char portname[BUFSIZ];      /**< ポート名。 */
    enum SerialBaud baudrate;   /**< ボーレート。 */
    char output[BUFSIZ];        /**< 出力ファイル名。 */
    int count;                  /**< 記録イベント数。 */
} option_t;

/**
 * @brief オプション分割装置用のコールバック関数。
 *
 * @param option オプション文字。
 * @param argument オプション文字に対する引数。
 * @param extobj 外部オブジェクト。
 *
 * @retval 0 解析継続。
 * @retval !0 解析中断。
 */
static int option_callback(
        const char option, const char *argument, void *extobj)
{
    option_t *opt = (option_t *)extobj;
    switch (option) {
        case 'p':
            strcpy(opt->portname, argument);
            opt->optflag |= OPTFLAG_PORTNAME;
            break;
        case 'b':
            switch (atoi(argument)) {
                case 2400:
                    opt->baudrate = SerialBaud2400;
                    break;
                case 4800:
                    opt->baudrate = SerialBaud4800;
                    break;
                case 9600:
                    opt->baudrate = SerialBaud9600;
                    break;
                case 19200:
                    opt->baudrate = SerialBaud19200;
                    break;
                case 38400:
                    opt->baudrate = SerialBaud38400;
                    break;
                case 57600:
                    opt->baudrate = SerialBaud57600;
                    break;
                case 115200:
                    opt->baudrate = SerialBaud115200;
                    break;
                default:
                    printf("Unsupported baudrate found.\n");
                    exit(1);
            }
            opt->optflag |= OPTFLAG_BAUDRATE;
            break;
        case 'o':
            strcpy(opt->output, argument);
            opt->optflag |= OPTFLAG_OUTPUT;
            break;
        case 'c':
            opt->count = atoi(argument);
            opt->optflag |= OPTFLAG_COUNT;
            break;
    }
    return 0;
}

/**
 * @brief エントリーポイント。
 *
 * @param argc 引数の数。
 * @param argv 引数のポインタのポインタ。
 *
 * @return シェルに返す値。
 */
int main(int argc, char **argv)
{
    int i;
    option_t opt;
    CAPTIME *cap;
    SERIAL *ser;
    FILE *fp;

    /*
     * オプション解析を実行する。
     */
    OPTION_INIT(&opt);
    optparse_char(argc, argv, &opt, option_callback);
    if (opt.optflag !=
            (OPTFLAG_PORTNAME |
             OPTFLAG_BAUDRATE |
             OPTFLAG_OUTPUT |
             OPTFLAG_COUNT)) {
        printf("ntlcap -p <port name> -b <baud rate> -o <output> -c <count>\n");
        return 1;
    }

    /*
     * キャプチャ時間モジュールを開く。
     */
    cap = captime_open();

    /*
     * シリアルを開く。
     */
    ser = serial_open(opt.portname, opt.baudrate);
    if (ser == NULL) {
        printf("Serial open failed.\n");
        return 1;
    }

    /*
     * 出力ファイルを開く。
     */
    fp = fopen(opt.output, "w");
    if (fp == NULL) {
        printf("File open failed.\n");
        return 1;
    }

    /*
     * 指定されたイベント数に達するまでファイルに保存する。
     */
    for (i = 0; i < opt.count; i++) {
        unsigned char evt;
        captime_t t;
        serial_read(ser, &evt, 1);
        captime_read(cap, &t);
        fprintf(fp, "%4d/%02d/%02d %02d:%02d:%02d.%06d %02X\n",
                t.year, t.month, t.day,
                t.hour, t.min, t.sec,
                t.usec, evt);
    }

    /*
     * 出力ファイルを閉じる。
     */
    fclose(fp);

    /*
     * シリアルを閉じる。
     */
    serial_close(ser);

    /*
     * キャプチャ時間モジュールを閉じる。
     */
    captime_close(cap);

    return 0;
}

