/**
 * @file uzura.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 * @brief BlueTank ACB-BF592 Application Sample Codes.
 */

/*
 * ===============================================================
 *  BlueTank
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <stdint.h>
#include <string.h>
#include <cdefBF592-A.h>
#include <builtins.h>
#include <sys/exception.h>
#include "lcd.h"
#include "led.h"
#include "pff.h"
#include "ssm2603.h"
#include "at24c.h"
#include "uzura.h"
#include "mmc.h"
#include "uart.h"
#include "bfin_util.h"

#define SCLOCK_HZ           (100000000) /**< システムクロック(100MHz) */
#define DMA_SAMPLE_SIZE     (256)       /**< 1回のサンプルサイズ */

static void setup_pll(uint8_t mul_val, uint8_t div_val);
static void __attribute__((interrupt_handler)) sport_rx_isr();

/**
 * SPORT受信バッファ(ダブルバッファ)
 */
static int32_t sport_buffer_rx[2][DMA_SAMPLE_SIZE];
/**
 * SPORT受信バッファ(ダブルバッファ)
 */
static int32_t sport_buffer_tx[2][DMA_SAMPLE_SIZE];
/**
 * DMAがどのバッファ使用をしているか
 */
static volatile int32_t bufidx_dma_target;
/**
 * データ転送完了フラグ
 */
static volatile int32_t data_ready;

static void setup_pll(uint8_t mul_val, uint8_t div_val)
{
    *pSIC_IWR = IWR_ENABLE(0);  /* PLLのみIWRを許す */
    *pPLL_DIV = div_val;
    *pPLL_CTL = (mul_val) << 9;
    asm("cli r0; csync; idle; sti r0;": : :"R0");
    *pSIC_IWR = IWR_ENABLE_ALL;
}

static void __attribute__((interrupt_handler)) sport_rx_isr()
{
    *pDMA1_IRQ_STATUS = DMA_DONE;
    ssync();
    /* オーディオコーデックからデータ到着 */
    data_ready = 1;
    /* DMAが使用するバッファを変更 */
    bufidx_dma_target = (bufidx_dma_target ^ 1) & 1;
}

void uzura_init(UZURA *p, void *user_data)
{
    FATFS fatfs;
#if 0
    DIR dir;
    FILINFO finfo;
#endif

    p->effect = NULL;
    p->system = NULL;
    p->user_data = user_data;

    /*
     * PLLを設定する。
     */
    setup_pll(16, 4);

    uart_init();

    /*
     * LEDを初期化する。
     */
    led_init();

    /*
     * LCDを初期化する。
     */
    lcd_init();
    lcd_clear();
    lcd_goto(0, 0);
    lcd_puts("BlueTank");
    lcd_goto(0, 1);
    lcd_puts(" UZURA! ");

    /*
     * オーディオコーデックを初期化する。
     * 初期化された時点ではミュートされている。
     */
    ssm2603_init();

    /*
     * I2C ROMモジュールを初期化する。
     */
    at24c_init();

    /*
     * SDカードをマウントする。
     */
    pf_mount(&fatfs);

    /*
     * バッファをクリアする。
     */
    memset(sport_buffer_rx, 0 , sizeof(sport_buffer_rx));
    memset(sport_buffer_tx, 0 , sizeof(sport_buffer_tx));
    bufidx_dma_target = 0;
    data_ready = 0;

    /*
     * 割り込みハンドラの登録。
     * IVG9が登録先。
     */
    bfin_util_register_handler(ik_ivg9, sport_rx_isr);
    *pSIC_IMASK |= IRQ_DMA1;

    /*
     * SPORT RXの設定
     * 外部クロック、外部SYNC、MSB First
     * 24ビットデータ、 ステレオ
     */
    *pSPORT0_RCR1 = RFSR | RCKFE;
    *pSPORT0_RCR2 = SLEN(31) | RSFSE;

    /*
     * SPORT TXの設定
     * 外部クロック、外部SYNC、MSB First
     * 24ビットデータ、 ステレオ
     */
    *pSPORT0_TCR1 = TFSR | TCKFE;
    *pSPORT0_TCR2 = SLEN(31) | TSFSE;

    /*
     * DMA1(SPORT0 RX)の設定
     */
    *pDMA1_PERIPHERAL_MAP = 0x1000;
    *pDMA1_CONFIG = FLOW_AUTO | DI_EN | DI_SEL | DMA2D | WDSIZE_32 | WNR;
    *pDMA1_START_ADDR = sport_buffer_rx;
    *pDMA1_X_COUNT = DMA_SAMPLE_SIZE;
    *pDMA1_X_MODIFY = sizeof(int32_t);
    *pDMA1_Y_COUNT = 2;
    *pDMA1_Y_MODIFY = sizeof(int32_t);

    /*
     * DMA2(SPORT0 TX)の設定
     */
    *pDMA2_PERIPHERAL_MAP = 0x2000;
    *pDMA2_CONFIG = FLOW_AUTO | DMA2D | WDSIZE_32;
    *pDMA2_START_ADDR = sport_buffer_tx;
    *pDMA2_X_COUNT = DMA_SAMPLE_SIZE;
    *pDMA2_X_MODIFY = sizeof(int32_t);
    *pDMA2_Y_COUNT = 2;
    *pDMA2_Y_MODIFY = sizeof(int32_t);
    ssync();

    /*
     * DMAを有効
     */
    *pDMA1_IRQ_STATUS = DMA_DONE;
    *pDMA1_CONFIG |= DMAEN;
    *pDMA2_CONFIG |= DMAEN;
    ssync();

    /*
     * SPORT0を有効
     */
    *pSPORT0_TCR1 |= TSPEN;
    *pSPORT0_RCR1 |= RSPEN;
    ssync();

    /*
     * ミュートを解除する。
     */
    ssm2603_mute(false);

    /*
     * AT24C initialize.
     */
    at24c_init();
}

void uzura_set_effect(UZURA *p, UZURA_EFFECT_FUNC effect)
{
    p->effect = effect;
}

void uzura_set_system(UZURA *p, UZURA_SYSTEM_FUNC system)
{
    p->system = system;
}

void uzura_execute(UZURA *p)
{
    int i, j;
    char lcdbuf[16];
    int32_t bufidx_dma_done;

    FontSet fontset;

    /*
     * Initialize fontset
     */
    lcd_font_init(&fontset);
    for (i = 0; i < (int)LCD_FONT_CHARS; i++) {
        for (j = 0; j < i + 1; j++) {
            lcd_font_draw_line(&fontset, (UserFont)i, 0, (7 - j), 4, (7 - j), 1);
        }
    }
    lcd_font_setup_all(&fontset);

    /*
     * Draw the text string.
     */
    for (i = 0; i < 8; i++) {
        lcdbuf[i] = 0x08 + i;
    }
    lcdbuf[8] = 0x00;
    lcd_goto(0, 0);
    lcd_puts(lcdbuf);

    lcd_puts("[MENU 1]");
    lcd_puts("[MENU 2]");
    lcd_puts("[MENU 3]");
    lcd_puts("[MENU 4]");

    while (1) {
        asm("idle;");
        if (0 != data_ready) {
            /*
             * フラグをクリアする。
             */
            data_ready = 0;

            /*
             * DMAが完了したバッファを使用してオーディオ処理を行なう。
             */
            bufidx_dma_done = bufidx_dma_target ^ 1;
            if (p->effect != NULL) {
                p->effect(
                        p,
                        sport_buffer_rx[bufidx_dma_done],
                        sport_buffer_tx[bufidx_dma_done],
                        DMA_SAMPLE_SIZE);
            }
        }
        if (p->system != NULL) {
            p->system(p);
        }
    }
}

