/**
 * @file uart.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 * @brief BlueTank ACB-BF592 Application Sample Codes.
 */

/*
 * ===============================================================
 *  BlueTank
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <cdefBF592-A.h>
#include <builtins.h>
#include "uart.h"

/* システムクロック(98.304MHz) */
#define SCLOCK_HZ       (98304000)

/* ボーレート(115200bps) */
#define UART0_BAUDRATE  (115200)

/* UART0_DLL/UART0_DLHの設定値 */
#define UART0_DIVISOR   (SCLOCK_HZ / 16 / UART0_BAUDRATE)

void uart_init(void)
{
    /* UARTイネーブル */
    *pUART0_GCTL = 1;

    /* UART分周比の設定 */
    *pUART0_LCR |= DLAB;
    *pUART0_DLL = UART0_DIVISOR & 0xFF;
    *pUART0_DLH = UART0_DIVISOR >> 8;
    *pUART0_LCR &= ~DLAB;

    /* UARTモード設定 パリティ無し 8bit 1ストップビット */
    *pUART0_LCR = WLS(8);

    /* UART割込み禁止 */
    *pUART0_IER = 0;

    /* PF11/PF12ピンをUARTに割り振る */
    *pPORTF_MUX &= ~(PF11 | PF12);  /* PF11,PF12を0にする */
    *pPORTF_FER |= PF11 | PF12;     /* PF11,PF12をUART0に割り振る */
}

void uart_putc(char c)
{
    /* UART0 LSRのTHREが1になるまで待つ。1ならば送信レジスタ空き */
    while (0 == (*pUART0_LSR & THRE)) {
        ssync();
    }
    *pUART0_THR = c;
    ssync();
}

char uart_getc(void)
{
    /* UART0_LSRのDRが1になるまで待つ。1ならば受信レジスタにデータあり */
    while (0 == (*pUART0_LSR & DR)) {
        ssync();
    }
    return *pUART0_RBR;
}

